#!/usr/bin/perl

use strict;
use warnings;

package Jls::Combination;

use Exporter;
our @ISA = qw/Exporter/;
our @EXPORT = 'new binomial get_n get_k get_array get_binomial increment';

=head 1

# Returns the Pascal combinatorial coefficient C(n,k).

sub binomial
# $n_ // the total # of objects
# $k_ // the # of objects chosen

# Returns an initialized combinatorial array.

sub new # Combination 
# $n_ // #(items)
# $k_ // #(chosen)

# Returns the total # of objects.

sub get_n # Combination

# Returns an array reference to the objects chosen.

sub get_array # Combination

# Returns the # of objects chosen.

sub get_k # Combination

# Returns the Pascal combinatorial coefficient C(n,k).

sub get_binomial # Combination

# Increments a combinatorial array.
# Returns 1, if increment successful; 0, otherwise.

sub increment 

=cut

# Returns the Pascal combinatorial coefficient C(n,k).

sub binomial
# $n_ // the total # of objects
# $k_ // the # of objects chosen
{
    my $n = $_ [0];
    my $k = $_ [1];
    
    my $binomial = 1;
    
    for (my $i = 0; $i < $k; $i++)
    {
        $binomial *= ($n - $i) / ($i + 1);
    }
    
    return $binomial;
}

# Returns an initialized combinatorial array.

sub new # Combination 
# $n_ // #(items)
# $k_ // #(chosen)
{
    my $class = shift;

    my $n_ = $_ [0];
    my $k_ = $_ [1];

    if ($n_ < $k_) { die 'Combination::new : ' . $n_ . ' < ' . $k_; }

    my @a = reverse ((0...($k_ - 1)));
    
    my $self = 
    {
        _n => $n_,
        _array => \@a,
    };

    return bless $self, $class;
}

# Returns the total # of objects.

sub get_n # Combination
{
    return $_ [0]->{_n};
}

# Returns the # of objects chosen.

sub get_k # Combination
{
    return scalar (@{$_ [0]->get_array ()});
}

# Returns an array reference to the objects chosen.

sub get_array # Combination
{
    return $_ [0]->{_array};
}

# Returns the Pascal combinatorial coefficient C(n,k).

sub get_binomial # Combination
{
    return binomial ($_ [0]->get_n (), $_ [0]->get_k ());
}

# Increments a combinatorial array.
# Returns 1, if increment successful; 0, otherwise.

sub increment # Combination 
{
    my $r_a = $_ [0]->{_array};
    
    if (scalar (@$r_a) == 0) { return 0; }
    
    my $i = 0;
    
    for ($i = 0; $i < scalar (@$r_a); $i++)
    {
        if ($$r_a [$i] < $_ [0]->{_n} - $i - 1) { last; } 
    }
    
    if (scalar (@$r_a) <= $i) { return 0; } 
    
    $$r_a [$i]++;
    
    for (my $j = $i - 1; $j >= 0; $j--)
    {
        $$r_a [$j] = $$r_a [$j + 1] + 1;
    }
    
    return 1;
}

1;
