#!/usr/bin/perl

use strict;
use warnings;

package Jls::TapInput;

use Exporter;
our @ISA = 'Exporter';
our @EXPORT = 'push_string_onto_retrieval_lists push_file_onto_retrieval_lists push_file_of_files_onto_retrieval_lists';

=head 1

# The routines in this file are based on the following object:
#
# $retrieval_lists = an array reference \@list, 
#     whose elements all have the form of the following hash reference.
#
# $retrieval_list =
# {
#     id, 
#     weight, 
#     number_of_relevant_records_stated,
#     list => \@list, // a list of hashes to the next structure
# };
#
# @list contains
#
# my $hash = 
# {
#     is_relevant, // 0 : irrelevant ; 1 : relevant
#     score, // score or E-value
# };

# Adds a string to a set of retrieval lists.
# The string must contain at least one retrieval list.

sub push_string_onto_retrieval_lists
# $retrieval_lists_ // an array reference \@list, as above
# $string_ // Contains retrieval list(s).

# Adds a file to a set of retrieval lists.
# The file must contain at least one retrieval list.

sub push_file_onto_retrieval_lists 
# $retrieval_lists_ // an array reference \@list, as above
# $file_ // Contains retrieval list(s).

# Adds a file of filenames to a set of retrieval lists.
# Each filename must appear on a separate line.
# For each filename, the corresponding file must contain at least one retrieval list.

sub push_file_of_files_onto_retrieval_lists 
# $retrieval_lists_ // an array reference \@list, as above
# $file_of_filenames_ // Contains a string, which contains retrieval list(s).

=cut

sub line_to_number_of_relevant_records_stated 
# $line_
{
    my $line_ = $_ [0];

    my @datum = split (/\s+/, $line_);

    return $datum [0] + 0; # convert to integer
}

sub line_to_retrieval_item 
# $line_
{
    my $line_ = $_ [0];

    my @datum = split (/\s+/, $line_);

    $datum [0] += 0; # convert to integer
    $datum [1] += 0.0; # convert to real

    my $hash = 
    {
        is_relevant => $datum [0],
        score => $datum [1],
    };

    return $hash;
}

sub line_to_id_and_weight
# $line_ // id and possibly weight also
{
    use Jls::File;
    
    my $line_ = $_ [0];

    my @array = split (/\s+/, $line_);
    my @lines = ();
    
    foreach my $str (@array)
    {
        if (! Jls::File::is_blank ($str)) { push (@lines, $str); } 
    }
    
    if ($#lines == 0) { push (@lines, '1'); }
    
    return @lines;
}

sub lines_to_one_retrieval_list
# \@lines_ // one retrieval list, with no blank lines
{
    my $lines_ = $_ [0];

    my $i = 0;

    my @id = line_to_id_and_weight ($lines_->[$i++]);

    my $number_of_relevant_records_stated = 
        line_to_number_of_relevant_records_stated ($lines_->[$i++]);
        
    my @list = ();
    for ( ; $i < scalar (@$lines_); $i++)
    {
        my $hash = line_to_retrieval_item ($lines_->[$i]);
        push (@list, $hash);
    }

    my $retrieval_list = 
    {
        id => $id [0],
        weight => $id [1],
        number_of_relevant_records_stated => $number_of_relevant_records_stated,
        list => \@list,
    };

    return $retrieval_list;
}

# Adds a string to a set of retrieval lists.
# The string must contain at least one retrieval list.

sub push_string_onto_retrieval_lists
# $retrieval_lists_ // an array reference \@list, as above
# $string_ // Contains retrieval list(s).
{
    use Jls::TapCheck;
    
    (my $retrieval_lists_, my $string_) = @_;

    my $msg = check_retrieval_lists ($string_);
    if ($msg) { die $msg; }
    
    my @lines = split ('\n', $string_); 

    for (my $i = 0; $i <= $#lines; $i++)
    {
        for ( ; $i <= $#lines && Jls::File::is_blank ($lines [$i]); $i++) {}
        
        my @array = ();
        
        for ( ; $i <= $#lines && ! Jls::File::is_blank ($lines [$i]); $i++) 
        {
            push (@array, $lines [$i]);
        }
        
        if ($#array != -1)
        {
            my $retrieval_list = Jls::TapInput::lines_to_one_retrieval_list (\@array);
            push (@$retrieval_lists_, $retrieval_list);
        }
    }
    
    return $retrieval_lists_;
}

# Adds a file to a set of retrieval lists.
# The file must contain at least one retrieval list.

sub push_file_onto_retrieval_lists 
# $retrieval_lists_ // an array reference \@list, as above
# $file_ // Contains retrieval list(s).
{
    use Jls::File;
    
    (my $retrieval_lists_, my $file_) = @_;

    my $string = Jls::File::file_to_string ($file_);
    push_string_onto_retrieval_lists ($retrieval_lists_, $string);

    return $retrieval_lists_;
}

# Adds a file of filenames to a set of retrieval lists.
# Each filename must appear on a separate line.
# For each filename, the corresponding file must contain at least one retrieval list.

sub push_file_of_files_onto_retrieval_lists 
# $retrieval_lists_ // an array reference \@list, as above
# $file_of_filenames_ // Contains a string, which contains retrieval list filename(s).
{
    use Jls::File;
    
    (my $retrieval_lists_, my $file_of_filenames_) = @_;

    my @files = split ("\n", Jls::File::file_to_string ($file_of_filenames_));
    
    foreach my $file (@files)
    {
        if (! Jls::File::is_blank ($file)) 
        { 
            Jls::TapInput::push_file_onto_retrieval_lists ($retrieval_lists_, $file);
        }
    } 

    return $retrieval_lists_;
}

1;
