/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.threadpool;

import java.io.Closeable;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.Executor;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.RejectedExecutionHandler;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import org.apache.logging.log4j.message.ParameterizedMessage;
import org.apache.lucene.util.Counter;
import org.apache.lucene.util.IOUtils;
import org.elasticsearch.common.Nullable;
import org.elasticsearch.common.component.AbstractComponent;
import org.elasticsearch.common.io.stream.StreamInput;
import org.elasticsearch.common.io.stream.StreamOutput;
import org.elasticsearch.common.io.stream.Writeable;
import org.elasticsearch.common.settings.Setting;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.unit.SizeValue;
import org.elasticsearch.common.unit.TimeValue;
import org.elasticsearch.common.util.concurrent.AbstractRunnable;
import org.elasticsearch.common.util.concurrent.EsAbortPolicy;
import org.elasticsearch.common.util.concurrent.EsExecutors;
import org.elasticsearch.common.util.concurrent.EsRejectedExecutionException;
import org.elasticsearch.common.util.concurrent.EsThreadPoolExecutor;
import org.elasticsearch.common.util.concurrent.ThreadContext;
import org.elasticsearch.common.util.concurrent.XRejectedExecutionHandler;
import org.elasticsearch.common.xcontent.ToXContent;
import org.elasticsearch.common.xcontent.XContentBuilder;
import org.elasticsearch.node.Node;
import org.elasticsearch.threadpool.ExecutorBuilder;
import org.elasticsearch.threadpool.FixedExecutorBuilder;
import org.elasticsearch.threadpool.ScalingExecutorBuilder;
import org.elasticsearch.threadpool.ThreadPoolInfo;
import org.elasticsearch.threadpool.ThreadPoolStats;

public class ThreadPool
extends AbstractComponent
implements Closeable {
    public static final Map<String, ThreadPoolType> THREAD_POOL_TYPES;
    private Map<String, ExecutorHolder> executors = new HashMap<String, ExecutorHolder>();
    private final ScheduledThreadPoolExecutor scheduler;
    private final CachedTimeThread cachedTimeThread;
    static final ExecutorService DIRECT_EXECUTOR;
    private final ThreadContext threadContext;
    private final Map<String, ExecutorBuilder> builders;
    public static Setting<TimeValue> ESTIMATED_TIME_INTERVAL_SETTING;

    public Collection<ExecutorBuilder> builders() {
        return Collections.unmodifiableCollection(this.builders.values());
    }

    public ThreadPool(Settings settings, ExecutorBuilder<?> ... customBuilders) {
        super(settings);
        assert (Node.NODE_NAME_SETTING.exists(settings));
        HashMap builders = new HashMap();
        int availableProcessors = EsExecutors.boundedNumberOfProcessors(settings);
        int halfProcMaxAt5 = ThreadPool.halfNumberOfProcessorsMaxFive(availableProcessors);
        int halfProcMaxAt10 = ThreadPool.halfNumberOfProcessorsMaxTen(availableProcessors);
        int genericThreadPoolMax = ThreadPool.boundedBy(4 * availableProcessors, 128, 512);
        builders.put("generic", new ScalingExecutorBuilder("generic", 4, genericThreadPoolMax, TimeValue.timeValueSeconds(30L)));
        builders.put("index", new FixedExecutorBuilder(settings, "index", availableProcessors, 200));
        builders.put("bulk", new FixedExecutorBuilder(settings, "bulk", availableProcessors, 200));
        builders.put("get", new FixedExecutorBuilder(settings, "get", availableProcessors, 1000));
        builders.put("search", new FixedExecutorBuilder(settings, "search", ThreadPool.searchThreadPoolSize(availableProcessors), 1000));
        builders.put("management", new ScalingExecutorBuilder("management", 1, 5, TimeValue.timeValueMinutes(5L)));
        builders.put("listener", new FixedExecutorBuilder(settings, "listener", halfProcMaxAt10, -1));
        builders.put("flush", new ScalingExecutorBuilder("flush", 1, halfProcMaxAt5, TimeValue.timeValueMinutes(5L)));
        builders.put("refresh", new ScalingExecutorBuilder("refresh", 1, halfProcMaxAt10, TimeValue.timeValueMinutes(5L)));
        builders.put("warmer", new ScalingExecutorBuilder("warmer", 1, halfProcMaxAt5, TimeValue.timeValueMinutes(5L)));
        builders.put("snapshot", new ScalingExecutorBuilder("snapshot", 1, halfProcMaxAt5, TimeValue.timeValueMinutes(5L)));
        builders.put("fetch_shard_started", new ScalingExecutorBuilder("fetch_shard_started", 1, 2 * availableProcessors, TimeValue.timeValueMinutes(5L)));
        builders.put("force_merge", new FixedExecutorBuilder(settings, "force_merge", 1, -1));
        builders.put("fetch_shard_store", new ScalingExecutorBuilder("fetch_shard_store", 1, 2 * availableProcessors, TimeValue.timeValueMinutes(5L)));
        for (ExecutorBuilder<?> builder : customBuilders) {
            if (builders.containsKey(builder.name())) {
                throw new IllegalArgumentException("builder with name [" + builder.name() + "] already exists");
            }
            builders.put(builder.name(), builder);
        }
        this.builders = Collections.unmodifiableMap(builders);
        this.threadContext = new ThreadContext(settings);
        HashMap executors = new HashMap();
        for (Map.Entry entry : builders.entrySet()) {
            Object executorSettings = ((ExecutorBuilder)entry.getValue()).getSettings(settings);
            ExecutorHolder executorHolder = ((ExecutorBuilder)entry.getValue()).build(executorSettings, this.threadContext);
            if (executors.containsKey(executorHolder.info.getName())) {
                throw new IllegalStateException("duplicate executors with name [" + executorHolder.info.getName() + "] registered");
            }
            this.logger.debug("created thread pool: {}", (Object)((ExecutorBuilder)entry.getValue()).formatInfo(executorHolder.info));
            executors.put(entry.getKey(), executorHolder);
        }
        executors.put("same", new ExecutorHolder(DIRECT_EXECUTOR, new Info("same", ThreadPoolType.DIRECT)));
        this.executors = Collections.unmodifiableMap(executors);
        this.scheduler = new ScheduledThreadPoolExecutor(1, EsExecutors.daemonThreadFactory(settings, "scheduler"), new EsAbortPolicy());
        this.scheduler.setExecuteExistingDelayedTasksAfterShutdownPolicy(false);
        this.scheduler.setContinueExistingPeriodicTasksAfterShutdownPolicy(false);
        this.scheduler.setRemoveOnCancelPolicy(true);
        TimeValue estimatedTimeInterval = ESTIMATED_TIME_INTERVAL_SETTING.get(settings);
        this.cachedTimeThread = new CachedTimeThread(EsExecutors.threadName(settings, "[timer]"), estimatedTimeInterval.millis());
        this.cachedTimeThread.start();
    }

    public long relativeTimeInMillis() {
        return this.cachedTimeThread.relativeTimeInMillis();
    }

    public long absoluteTimeInMillis() {
        return this.cachedTimeThread.absoluteTimeInMillis();
    }

    public Counter estimatedTimeInMillisCounter() {
        return this.cachedTimeThread.counter;
    }

    public ThreadPoolInfo info() {
        ArrayList<Info> infos = new ArrayList<Info>();
        for (ExecutorHolder holder : this.executors.values()) {
            String name = holder.info.getName();
            if ("same".equals(name)) continue;
            infos.add(holder.info);
        }
        return new ThreadPoolInfo(infos);
    }

    public Info info(String name) {
        ExecutorHolder holder = this.executors.get(name);
        if (holder == null) {
            return null;
        }
        return holder.info;
    }

    public ThreadPoolStats stats() {
        ArrayList<ThreadPoolStats.Stats> stats = new ArrayList<ThreadPoolStats.Stats>();
        for (ExecutorHolder holder : this.executors.values()) {
            String name = holder.info.getName();
            if ("same".equals(name)) continue;
            int threads = -1;
            int queue = -1;
            int active = -1;
            long rejected = -1L;
            int largest = -1;
            long completed = -1L;
            if (holder.executor() instanceof ThreadPoolExecutor) {
                ThreadPoolExecutor threadPoolExecutor = (ThreadPoolExecutor)holder.executor();
                threads = threadPoolExecutor.getPoolSize();
                queue = threadPoolExecutor.getQueue().size();
                active = threadPoolExecutor.getActiveCount();
                largest = threadPoolExecutor.getLargestPoolSize();
                completed = threadPoolExecutor.getCompletedTaskCount();
                RejectedExecutionHandler rejectedExecutionHandler = threadPoolExecutor.getRejectedExecutionHandler();
                if (rejectedExecutionHandler instanceof XRejectedExecutionHandler) {
                    rejected = ((XRejectedExecutionHandler)rejectedExecutionHandler).rejected();
                }
            }
            stats.add(new ThreadPoolStats.Stats(name, threads, queue, active, rejected, largest, completed));
        }
        return new ThreadPoolStats(stats);
    }

    public ExecutorService generic() {
        return this.executor("generic");
    }

    public ExecutorService executor(String name) {
        ExecutorHolder holder = this.executors.get(name);
        if (holder == null) {
            throw new IllegalArgumentException("no executor service found for [" + name + "]");
        }
        return holder.executor();
    }

    public ScheduledExecutorService scheduler() {
        return this.scheduler;
    }

    public Cancellable scheduleWithFixedDelay(Runnable command, TimeValue interval, String executor) {
        return new ReschedulingRunnable(command, interval, executor, this);
    }

    public ScheduledFuture<?> schedule(TimeValue delay, String executor, Runnable command) {
        if (!"same".equals(executor)) {
            command = new ThreadedRunnable(command, this.executor(executor));
        }
        return this.scheduler.schedule(new LoggingRunnable(command), delay.millis(), TimeUnit.MILLISECONDS);
    }

    public void shutdown() {
        this.cachedTimeThread.running = false;
        this.cachedTimeThread.interrupt();
        this.scheduler.shutdown();
        for (ExecutorHolder executor : this.executors.values()) {
            if (!(executor.executor() instanceof ThreadPoolExecutor)) continue;
            ((ThreadPoolExecutor)executor.executor()).shutdown();
        }
    }

    public void shutdownNow() {
        this.cachedTimeThread.running = false;
        this.cachedTimeThread.interrupt();
        this.scheduler.shutdownNow();
        for (ExecutorHolder executor : this.executors.values()) {
            if (!(executor.executor() instanceof ThreadPoolExecutor)) continue;
            ((ThreadPoolExecutor)executor.executor()).shutdownNow();
        }
    }

    public boolean awaitTermination(long timeout, TimeUnit unit) throws InterruptedException {
        boolean result = this.scheduler.awaitTermination(timeout, unit);
        for (ExecutorHolder executor : this.executors.values()) {
            if (!(executor.executor() instanceof ThreadPoolExecutor)) continue;
            result &= ((ThreadPoolExecutor)executor.executor()).awaitTermination(timeout, unit);
        }
        this.cachedTimeThread.join(unit.toMillis(timeout));
        return result;
    }

    static int boundedBy(int value, int min2, int max) {
        return Math.min(max, Math.max(min2, value));
    }

    static int halfNumberOfProcessorsMaxFive(int numberOfProcessors) {
        return ThreadPool.boundedBy((numberOfProcessors + 1) / 2, 1, 5);
    }

    static int halfNumberOfProcessorsMaxTen(int numberOfProcessors) {
        return ThreadPool.boundedBy((numberOfProcessors + 1) / 2, 1, 10);
    }

    static int twiceNumberOfProcessors(int numberOfProcessors) {
        return ThreadPool.boundedBy(2 * numberOfProcessors, 2, Integer.MAX_VALUE);
    }

    public static int searchThreadPoolSize(int availableProcessors) {
        return availableProcessors * 3 / 2 + 1;
    }

    public static boolean terminate(ExecutorService service, long timeout, TimeUnit timeUnit) {
        if (service != null) {
            service.shutdown();
            if (ThreadPool.awaitTermination(service, timeout, timeUnit)) {
                return true;
            }
            service.shutdownNow();
            return ThreadPool.awaitTermination(service, timeout, timeUnit);
        }
        return false;
    }

    private static boolean awaitTermination(ExecutorService service, long timeout, TimeUnit timeUnit) {
        try {
            if (service.awaitTermination(timeout, timeUnit)) {
                return true;
            }
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        }
        return false;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static boolean terminate(ThreadPool pool, long timeout, TimeUnit timeUnit) {
        if (pool != null) {
            block4: {
                boolean bl;
                try {
                    pool.shutdown();
                    if (!ThreadPool.awaitTermination(pool, timeout, timeUnit)) break block4;
                    bl = true;
                }
                catch (Throwable throwable) {
                    IOUtils.closeWhileHandlingException((Closeable[])new Closeable[]{pool});
                    throw throwable;
                }
                IOUtils.closeWhileHandlingException((Closeable[])new Closeable[]{pool});
                return bl;
            }
            pool.shutdownNow();
            boolean bl = ThreadPool.awaitTermination(pool, timeout, timeUnit);
            IOUtils.closeWhileHandlingException((Closeable[])new Closeable[]{pool});
            return bl;
        }
        return false;
    }

    private static boolean awaitTermination(ThreadPool pool, long timeout, TimeUnit timeUnit) {
        try {
            if (pool.awaitTermination(timeout, timeUnit)) {
                return true;
            }
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        }
        return false;
    }

    @Override
    public void close() throws IOException {
        this.threadContext.close();
    }

    public ThreadContext getThreadContext() {
        return this.threadContext;
    }

    public static boolean assertNotScheduleThread(String reason) {
        assert (!Thread.currentThread().getName().contains("scheduler")) : "Expected current thread [" + Thread.currentThread() + "] to not be the scheduler thread. Reason: [" + reason + "]";
        return true;
    }

    static {
        HashMap<String, ThreadPoolType> map = new HashMap<String, ThreadPoolType>();
        map.put("same", ThreadPoolType.DIRECT);
        map.put("generic", ThreadPoolType.SCALING);
        map.put("listener", ThreadPoolType.FIXED);
        map.put("get", ThreadPoolType.FIXED);
        map.put("index", ThreadPoolType.FIXED);
        map.put("bulk", ThreadPoolType.FIXED);
        map.put("search", ThreadPoolType.FIXED);
        map.put("management", ThreadPoolType.SCALING);
        map.put("flush", ThreadPoolType.SCALING);
        map.put("refresh", ThreadPoolType.SCALING);
        map.put("warmer", ThreadPoolType.SCALING);
        map.put("snapshot", ThreadPoolType.SCALING);
        map.put("force_merge", ThreadPoolType.FIXED);
        map.put("fetch_shard_started", ThreadPoolType.SCALING);
        map.put("fetch_shard_store", ThreadPoolType.SCALING);
        THREAD_POOL_TYPES = Collections.unmodifiableMap(map);
        DIRECT_EXECUTOR = EsExecutors.newDirectExecutorService();
        ESTIMATED_TIME_INTERVAL_SETTING = Setting.timeSetting("thread_pool.estimated_time_interval", TimeValue.timeValueMillis(200L), Setting.Property.NodeScope);
    }

    static final class ReschedulingRunnable
    extends AbstractRunnable
    implements Cancellable {
        private final Runnable runnable;
        private final TimeValue interval;
        private final String executor;
        private final ThreadPool threadPool;
        private volatile boolean run = true;

        ReschedulingRunnable(Runnable runnable, TimeValue interval, String executor, ThreadPool threadPool) {
            this.runnable = runnable;
            this.interval = interval;
            this.executor = executor;
            this.threadPool = threadPool;
            threadPool.schedule(interval, executor, this);
        }

        @Override
        public void cancel() {
            this.run = false;
        }

        @Override
        public boolean isCancelled() {
            return !this.run;
        }

        @Override
        public void doRun() {
            if (this.run) {
                this.runnable.run();
            }
        }

        @Override
        public void onFailure(Exception e) {
            this.threadPool.logger.warn(() -> new ParameterizedMessage("failed to run scheduled task [{}] on thread pool [{}]", (Object)this.runnable.toString(), (Object)this.executor), (Throwable)e);
        }

        @Override
        public void onRejection(Exception e) {
            this.run = false;
            if (this.threadPool.logger.isDebugEnabled()) {
                this.threadPool.logger.debug(() -> new ParameterizedMessage("scheduled task [{}] was rejected on thread pool [{}]", (Object)this.runnable, (Object)this.executor), (Throwable)e);
            }
        }

        @Override
        public void onAfter() {
            if (this.run) {
                try {
                    this.threadPool.schedule(this.interval, this.executor, this);
                }
                catch (EsRejectedExecutionException e) {
                    this.onRejection(e);
                }
            }
        }
    }

    public static interface Cancellable {
        public void cancel();

        public boolean isCancelled();
    }

    public static class Info
    implements Writeable,
    ToXContent {
        private final String name;
        private final ThreadPoolType type;
        private final int min;
        private final int max;
        private final TimeValue keepAlive;
        private final SizeValue queueSize;

        public Info(String name, ThreadPoolType type) {
            this(name, type, -1);
        }

        public Info(String name, ThreadPoolType type, int size) {
            this(name, type, size, size, null, null);
        }

        public Info(String name, ThreadPoolType type, int min2, int max, @Nullable TimeValue keepAlive, @Nullable SizeValue queueSize) {
            this.name = name;
            this.type = type;
            this.min = min2;
            this.max = max;
            this.keepAlive = keepAlive;
            this.queueSize = queueSize;
        }

        public Info(StreamInput in) throws IOException {
            this.name = in.readString();
            this.type = ThreadPoolType.fromType(in.readString());
            this.min = in.readInt();
            this.max = in.readInt();
            this.keepAlive = in.readOptionalWriteable(TimeValue::new);
            this.queueSize = in.readOptionalWriteable(SizeValue::new);
        }

        @Override
        public void writeTo(StreamOutput out) throws IOException {
            out.writeString(this.name);
            out.writeString(this.type.getType());
            out.writeInt(this.min);
            out.writeInt(this.max);
            out.writeOptionalWriteable(this.keepAlive);
            out.writeOptionalWriteable(this.queueSize);
        }

        public String getName() {
            return this.name;
        }

        public ThreadPoolType getThreadPoolType() {
            return this.type;
        }

        public int getMin() {
            return this.min;
        }

        public int getMax() {
            return this.max;
        }

        @Nullable
        public TimeValue getKeepAlive() {
            return this.keepAlive;
        }

        @Nullable
        public SizeValue getQueueSize() {
            return this.queueSize;
        }

        @Override
        public XContentBuilder toXContent(XContentBuilder builder, ToXContent.Params params) throws IOException {
            builder.startObject(this.name);
            builder.field("type", this.type.getType());
            if (this.min != -1) {
                builder.field("min", this.min);
            }
            if (this.max != -1) {
                builder.field("max", this.max);
            }
            if (this.keepAlive != null) {
                builder.field("keep_alive", this.keepAlive.toString());
            }
            if (this.queueSize == null) {
                builder.field("queue_size", -1);
            } else {
                builder.field("queue_size", this.queueSize.singles());
            }
            builder.endObject();
            return builder;
        }

        static final class Fields {
            static final String TYPE = "type";
            static final String MIN = "min";
            static final String MAX = "max";
            static final String KEEP_ALIVE = "keep_alive";
            static final String QUEUE_SIZE = "queue_size";

            Fields() {
            }
        }
    }

    static class ExecutorHolder {
        private final ExecutorService executor;
        public final Info info;

        ExecutorHolder(ExecutorService executor, Info info) {
            assert (executor instanceof EsThreadPoolExecutor || executor == DIRECT_EXECUTOR);
            this.executor = executor;
            this.info = info;
        }

        ExecutorService executor() {
            return this.executor;
        }
    }

    static class CachedTimeThread
    extends Thread {
        final long interval;
        final TimeCounter counter;
        volatile boolean running = true;
        volatile long relativeMillis;
        volatile long absoluteMillis;

        CachedTimeThread(String name, long interval) {
            super(name);
            this.interval = interval;
            this.relativeMillis = TimeValue.nsecToMSec(System.nanoTime());
            this.absoluteMillis = System.currentTimeMillis();
            this.counter = new TimeCounter();
            this.setDaemon(true);
        }

        long relativeTimeInMillis() {
            return this.relativeMillis;
        }

        long absoluteTimeInMillis() {
            return this.absoluteMillis;
        }

        @Override
        public void run() {
            while (this.running) {
                this.relativeMillis = TimeValue.nsecToMSec(System.nanoTime());
                this.absoluteMillis = System.currentTimeMillis();
                try {
                    Thread.sleep(this.interval);
                }
                catch (InterruptedException e) {
                    this.running = false;
                    return;
                }
            }
        }

        private class TimeCounter
        extends Counter {
            private TimeCounter() {
            }

            public long addAndGet(long delta) {
                throw new UnsupportedOperationException();
            }

            public long get() {
                return CachedTimeThread.this.relativeMillis;
            }
        }
    }

    class ThreadedRunnable
    implements Runnable {
        private final Runnable runnable;
        private final Executor executor;

        ThreadedRunnable(Runnable runnable, Executor executor) {
            this.runnable = runnable;
            this.executor = executor;
        }

        @Override
        public void run() {
            this.executor.execute(this.runnable);
        }

        public int hashCode() {
            return this.runnable.hashCode();
        }

        public boolean equals(Object obj) {
            return this.runnable.equals(obj);
        }

        public String toString() {
            return "[threaded] " + this.runnable.toString();
        }
    }

    class LoggingRunnable
    implements Runnable {
        private final Runnable runnable;

        LoggingRunnable(Runnable runnable) {
            this.runnable = runnable;
        }

        @Override
        public void run() {
            try {
                this.runnable.run();
            }
            catch (Exception e) {
                ThreadPool.this.logger.warn(() -> new ParameterizedMessage("failed to run {}", (Object)this.runnable.toString()), (Throwable)e);
                throw e;
            }
        }

        public int hashCode() {
            return this.runnable.hashCode();
        }

        public boolean equals(Object obj) {
            return this.runnable.equals(obj);
        }

        public String toString() {
            return "[threaded] " + this.runnable.toString();
        }
    }

    public static enum ThreadPoolType {
        DIRECT("direct"),
        FIXED("fixed"),
        SCALING("scaling");

        private final String type;
        private static final Map<String, ThreadPoolType> TYPE_MAP;

        public String getType() {
            return this.type;
        }

        private ThreadPoolType(String type) {
            this.type = type;
        }

        public static ThreadPoolType fromType(String type) {
            ThreadPoolType threadPoolType = TYPE_MAP.get(type);
            if (threadPoolType == null) {
                throw new IllegalArgumentException("no ThreadPoolType for " + type);
            }
            return threadPoolType;
        }

        static {
            HashMap<String, ThreadPoolType> typeMap = new HashMap<String, ThreadPoolType>();
            for (ThreadPoolType threadPoolType : ThreadPoolType.values()) {
                typeMap.put(threadPoolType.getType(), threadPoolType);
            }
            TYPE_MAP = Collections.unmodifiableMap(typeMap);
        }
    }

    public static class Names {
        public static final String SAME = "same";
        public static final String GENERIC = "generic";
        public static final String LISTENER = "listener";
        public static final String GET = "get";
        public static final String INDEX = "index";
        public static final String BULK = "bulk";
        public static final String SEARCH = "search";
        public static final String MANAGEMENT = "management";
        public static final String FLUSH = "flush";
        public static final String REFRESH = "refresh";
        public static final String WARMER = "warmer";
        public static final String SNAPSHOT = "snapshot";
        public static final String FORCE_MERGE = "force_merge";
        public static final String FETCH_SHARD_STARTED = "fetch_shard_started";
        public static final String FETCH_SHARD_STORE = "fetch_shard_store";
    }
}

