package de.julianegner.multiplatformTooltip

import androidx.compose.foundation.hoverable
import androidx.compose.foundation.interaction.HoverInteraction
import androidx.compose.foundation.interaction.MutableInteractionSource
import androidx.compose.foundation.layout.Box
import androidx.compose.runtime.Composable
import androidx.compose.runtime.CompositionLocalProvider
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.ui.Modifier
import androidx.compose.ui.geometry.Size
import androidx.compose.ui.layout.onSizeChanged
import androidx.compose.ui.unit.DpOffset
import androidx.compose.ui.unit.dp
import androidx.compose.ui.zIndex
import kotlin.time.Duration
import kotlin.time.Duration.Companion.seconds

/**
 * A wrapper that shows a tooltip when the user hovers over the content (desktop) or long-presses (mobile).
 * The tooltip can be a simple text or a custom composable.
 * @param text The text to show in the tooltip. Ignored if [tooltipComposable] is provided.
 * @param offset The offset to apply to the tooltip position. If null, the tooltip is positioned automatically.
 * @param autocloseAfter The duration after which the tooltip will automatically close. Defaults to 3 seconds.
 * @param tooltipComposable A custom composable to show as the tooltip. If null, a simple text tooltip is shown.
 * @param content The content that triggers the tooltip on hover (desktop) or long-press (mobile).
 *
 * one of text or tooltipComposable MUST be provided to show a tooltip
 */
@Composable
fun TooltipWrapper(
    text: String? = null,
    offset: DpOffset? = null,
    autocloseAfter: Duration = 3.seconds,
    tooltipComposable: (@Composable () -> Unit)? = null,
    content: @Composable () -> Unit
) {
    val tooltipWrapperZIndex = remember { TooltipCounter.next() }
    val interactionSource = remember { MutableInteractionSource() }
    val isTooltipVisible = remember { mutableStateOf(false) }

    LaunchedEffect(interactionSource) {
        interactionSource.interactions.collect { interaction ->
            when (interaction) {
                is HoverInteraction.Enter -> {
                    isTooltipVisible.value = true
                }
                is HoverInteraction.Exit -> {
                    isTooltipVisible.value = false
                }
            }
        }
    }

    // Auto-close tooltip and reset long-press state
    LaunchedEffect(isTooltipVisible.value) {
        if (isTooltipVisible.value) {
            kotlinx.coroutines.delay(autocloseAfter)
            isTooltipVisible.value = false
        }
    }

    val showTooltip = {
        isTooltipVisible.value = true
    }

    val tooltipContext = TooltipContext(
        showTooltip = showTooltip
    )

    val boxSize = remember { mutableStateOf(Size.Zero) }
    Box (
        modifier = Modifier
            // we need this to ensure that the tooltip is shown above TooltipWrappers that are printed later
            .zIndex(tooltipWrapperZIndex.toFloat())
            .onSizeChanged { size ->
                boxSize.value = Size(size.width.toFloat(), size.height.toFloat())
            }
            .hoverable(interactionSource = interactionSource, enabled = true)
            .longPressGesture(
                interactionSource = interactionSource,
                onLongPress = {
                    showTooltip()
                }
            )
    ) {
        CompositionLocalProvider(LocalTooltipContext provides tooltipContext) {
            if (isTooltipVisible.value) {
                // show custom tooltip if provided
                if (tooltipComposable != null) {
                    tooltipComposable()
                }
                else if (!text.isNullOrEmpty()) {
                    // auto position the tooltip if no offset is provided
                    val yOffset =
                        if (isPlatformMobile)
                            ((boxSize.value.height / 4) + 5f).dp
                        else
                            ((boxSize.value.height / 2) + 5f).dp
                    val tooltipOffset =
                        offset ?: DpOffset(
                            x = (boxSize.value.width / 4).dp,
                            y = yOffset
                        )
                    Tooltip(text, tooltipOffset)
                }
            }
            content()
        }
    }
}

