package de.julianegner.multiplatformTooltip

import androidx.compose.foundation.BorderStroke
import androidx.compose.foundation.interaction.MutableInteractionSource
import androidx.compose.foundation.interaction.PressInteraction
import androidx.compose.foundation.layout.PaddingValues
import androidx.compose.material.Button
import androidx.compose.material.ButtonColors
import androidx.compose.material.ButtonDefaults
import androidx.compose.material.ButtonElevation
import androidx.compose.material.MaterialTheme
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.remember
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Shape
import androidx.compose.ui.platform.LocalViewConfiguration
import kotlinx.coroutines.delay
import kotlinx.coroutines.flow.collectLatest

/**
 * A button that shows a tooltip when long-pressed (on mobile) or hovered (on desktop).
 * The [TooltipButton] is a composable that wraps a standard [Button] and adds tooltip functionality.
 *
 * The parameters are the same as of a standard [Button], but adding [tooltipText] while removing [interactionSource],
 * as the latter is set by the [TooltipWrapper] used inside of [TooltipButton].
 *
 * @param tooltipText The text to display in the tooltip.
 * @param onClick The action to perform when the button is clicked.
 * @param modifier The modifier to be applied to the button.
 * @param enabled Controls the enabled state of the button. When `false`, the button will not respond to user input.
 * @param elevation The elevation of the button. See [ButtonDefaults.elevation].
 * @param shape The shape of the button. See [MaterialTheme.shapes].
 * @param border The border of the button. See [ButtonDefaults.outlinedBorder].
 * @param colors The color scheme of the button. See [ButtonDefaults.buttonColors].
 * @param contentPadding The padding values for the content inside the button. See [ButtonDefaults.ContentPadding].
 * @param content The content of the button, typically a [androidx.compose.material.Text] composable.
 */
@Composable
fun TooltipButton(
    tooltipText: String,
    onClick: () -> Unit,
    modifier: Modifier = Modifier,
    enabled: Boolean = true,
    elevation: ButtonElevation? = ButtonDefaults.elevation(),
    shape: Shape = MaterialTheme.shapes.small,
    border: BorderStroke? = null,
    colors: ButtonColors = ButtonDefaults.buttonColors(),
    contentPadding: PaddingValues = ButtonDefaults.ContentPadding,
    content: @Composable () -> Unit
) {
    TooltipWrapper(tooltipText) {
        val tooltipContext = LocalTooltipContext.current
        val interactionSource = remember { MutableInteractionSource() }
        val viewConfiguration = LocalViewConfiguration.current

        LaunchedEffect(interactionSource) {
            var isLongClick = false
            interactionSource.interactions.collectLatest { interaction ->
                when (interaction) {
                    is PressInteraction.Press -> {
                        isLongClick = false
                        delay(viewConfiguration.longPressTimeoutMillis)
                        isLongClick = true
                        tooltipContext.showTooltip.invoke()
                    }
                    is PressInteraction.Release -> {
                        if (isLongClick.not()) onClick()
                    }
                }
            }
        }

        Button(
            onClick = {},
            modifier = modifier,
            enabled = enabled,
            interactionSource = interactionSource,
            elevation = elevation,
            shape = shape,
            border = border,
            colors = colors,
            contentPadding = contentPadding,
        ) {
            content()
        }
    }
}
