package de.julianegner.multiplatformTooltip

import androidx.compose.foundation.Indication
import androidx.compose.foundation.clickable
import androidx.compose.foundation.gestures.detectTapGestures
import androidx.compose.foundation.interaction.MutableInteractionSource
import androidx.compose.runtime.Composable
import androidx.compose.ui.Modifier
import androidx.compose.ui.input.pointer.pointerInput
import androidx.compose.ui.layout.layout
import androidx.compose.ui.semantics.Role
import androidx.compose.ui.zIndex

val standardTooltipModifier = Modifier
    .zIndex(1f)
    .layout { measurable, constraints ->
        // Measure the tooltip but don't add it to the layout
        val placeable = measurable.measure(constraints)
        layout(0,0) { // Set the size to 0 to avoid taking up space and move other elements
            placeable.place(0, 0)
        }
    }

// the val is not accessible from an app using the library, so to make it accessible we need this function
fun Modifier.standardTooltipModifier() = standardTooltipModifier

@Composable
fun Modifier.tooltipCompatibleClick(
    enabled: Boolean = true,
    onClickLabel: String? = null,
    role: Role? = null,
    onClick: () -> Unit): Modifier {
    val tooltipContext = LocalTooltipContext.current
    return this
        .clickable(
            enabled = enabled,
            onClickLabel = onClickLabel,
            role = role
        ) { onClick() }
        .pointerInput("child_gesture") {
            detectTapGestures(
                onTap = {onClick()},
                onLongPress = {
                    tooltipContext.showTooltip.invoke()
                }
            )
        }
}

/**
 * this is needed to make sure that both click and long-press are detected
 * log-press is needed for mobile
 *
 * @param interactionSource The interaction source to track interaction events.
 * @param indication The visual indication to show when the element is pressed.
 * @param enabled Whether the clickable is enabled or not. When false, the clickable will not
 * respond to click events.
 * @param onClickLabel The content description of the clickable for accessibility services.
 * @param role The semantic role of the clickable.
 * @param onClick The lambda to be invoked when the element is clicked.
 *
 * @return The modified [Modifier] that handles both click and long-press events.
 */
@Composable
fun Modifier.tooltipCompatibleClick(
    interactionSource: MutableInteractionSource?,
    indication: Indication?,
    enabled: Boolean = true,
    onClickLabel: String? = null,
    role: Role? = null,
    onClick: () -> Unit): Modifier {
    val tooltipContext = LocalTooltipContext.current
    return this
        .clickable(
            interactionSource = interactionSource,
            indication = indication,
            enabled = enabled,
            onClickLabel = onClickLabel,
            role = role
        ) { onClick() }
        .pointerInput("child_gesture") {
            detectTapGestures(
                onTap = {onClick()},
                onLongPress = {
                    tooltipContext.showTooltip.invoke()
                }
            )
        }
}
