package de.julianegner.multiplatformTooltip

import androidx.compose.foundation.background
import androidx.compose.foundation.hoverable
import androidx.compose.foundation.interaction.HoverInteraction
import androidx.compose.foundation.interaction.MutableInteractionSource
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.offset
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.shape.RoundedCornerShape
import androidx.compose.runtime.Composable
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.ui.Modifier
import androidx.compose.material.Text
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.ui.draw.clip
import androidx.compose.ui.geometry.Size
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.layout.layout
import androidx.compose.ui.layout.onSizeChanged
import androidx.compose.ui.unit.DpOffset
import androidx.compose.ui.unit.TextUnit
import androidx.compose.ui.unit.TextUnitType
import androidx.compose.ui.unit.dp
import androidx.compose.ui.zIndex
import kotlin.time.Duration
import kotlin.time.Duration.Companion.seconds

val standardTooltipModifier = Modifier
        .zIndex(1f)
        .layout { measurable, constraints ->
            // Measure the tooltip but don't add it to the layout
            val placeable = measurable.measure(constraints)
            layout(0,0) { // Set the size to 0 to avoid taking up space and move other elements
                placeable.place(0, 0)
            }
}

object TooltipCounter {
    private var current = 1000 // Start value

    fun next(): Int {
        return current--
    }

    fun reset() {
        current = 1000
    }
}

/**
 * A wrapper that shows a tooltip when the user hovers over the content.
 * The tooltip can be a simple text or a custom composable.
 * @param text The text to show in the tooltip. Ignored if [tooltipComposable] is provided.
 * @param offset The offset to apply to the tooltip position. If null, the tooltip is positioned automatically.
 * @param autocloseAfter The duration after which the tooltip will automatically close. Defaults to 3 seconds.
 * @param tooltipComposable A custom composable to show as the tooltip. If null, a simple text tooltip is shown.
 * @param content The content that triggers the tooltip on hover.
 *
 * one of text or tooltipComposable MUST be provided to show a tooltip
 */
@Composable
fun TooltipWrapper(
    text: String? = null,
    offset: DpOffset? = null,
    autocloseAfter: Duration = 3.seconds,
    tooltipComposable: (@Composable () -> Unit)? = null,
    content: @Composable () -> Unit
) {
    val tooltipWrapperZIndex = remember { TooltipCounter.next() }

    val interactionSource = remember { MutableInteractionSource() }
    val isHovered = remember { mutableStateOf(false) }

    LaunchedEffect(interactionSource) {
        interactionSource.interactions.collect { interaction ->
            when (interaction) {
                is HoverInteraction.Enter -> isHovered.value = true
                is HoverInteraction.Exit -> isHovered.value = false
            }
        }
    }

    // Auto-close tooltip
    LaunchedEffect(isHovered.value) {
        if (isHovered.value) {
            kotlinx.coroutines.delay(autocloseAfter)
            isHovered.value = false
        }
    }

    val boxSize = remember { mutableStateOf(Size.Zero) }
    Box (
        modifier = Modifier
            // we need this to ensure that the tooltip is shown above TooltipWrappers that are printed later
            .zIndex(tooltipWrapperZIndex.toFloat())
            .onSizeChanged { size ->
                boxSize.value = Size(size.width.toFloat(), size.height.toFloat())
            }
            .hoverable(interactionSource = interactionSource, enabled = true)
        ) {

        if (isHovered.value) {
            // show custom tooltip if provided
            if (tooltipComposable != null) {
                tooltipComposable()
            }
            else if (!text.isNullOrEmpty()) {
                // auto position the tooltip if no offset is provided
                val tooltipOffset =
                    offset ?: DpOffset((boxSize.value.width / 4).dp, ((boxSize.value.height / 2) + 5f).dp)
                Tooltip(text, tooltipOffset)
            }
        }
        content()
    }
}

// UI element to show the tooltip
@Composable
private fun Tooltip(
    text: String,
    offset: DpOffset = DpOffset(0.dp, 0.dp),
) {
    Box(
        modifier =  standardTooltipModifier
            .offset(x = offset.x, y = offset.y)
            .clip(RoundedCornerShape(8.dp))
            .background(Color.LightGray)
            .padding(10.dp)
    ) {
        Text(
            text,
            color = Color.Black,
            fontSize = TextUnit(1f, TextUnitType.Em),
            lineHeight = TextUnit(1.5f, TextUnitType.Em)
        )
    }
}

