package de.jplag.rlang;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;

import org.antlr.v4.runtime.CharStreams;
import org.antlr.v4.runtime.CommonTokenStream;
import org.antlr.v4.runtime.ParserRuleContext;
import org.antlr.v4.runtime.tree.ParseTree;
import org.antlr.v4.runtime.tree.ParseTreeWalker;

import de.jplag.AbstractParser;
import de.jplag.ParsingException;
import de.jplag.Token;
import de.jplag.TokenType;
import de.jplag.rlang.grammar.RFilter;
import de.jplag.rlang.grammar.RLexer;
import de.jplag.rlang.grammar.RParser;

/**
 * This class sets up the lexer and parser generated by ANTLR4, feeds the submissions through them and passes the
 * selected tokens on to the main program.
 */
public class RParserAdapter extends AbstractParser {

    private File currentFile;
    private List<Token> tokens;

    /**
     * Creates the RParserAdapter
     */
    public RParserAdapter() {
        super();
    }

    /**
     * Parsers a set of files into a single token list of {@link Token}s.
     * @param files the set of files.
     * @return a list containing all tokens of all files.
     */
    public List<Token> parse(Set<File> files) throws ParsingException {
        tokens = new ArrayList<>();
        for (File file : files) {
            parseFile(file);
            tokens.add(Token.fileEnd(file));
        }
        return tokens;
    }

    private void parseFile(File file) throws ParsingException {
        try (FileInputStream inputStream = new FileInputStream(file)) {
            currentFile = file;

            // create a lexer, a parser and a buffer between them.
            RLexer lexer = new RLexer(CharStreams.fromStream(inputStream));
            CommonTokenStream tokens = new CommonTokenStream(lexer);

            RFilter filter = new RFilter(tokens);
            filter.stream();
            tokens.seek(0);

            RParser parser = new RParser(tokens);

            // Create a tree walker and the entry context defined by the parser grammar
            ParserRuleContext entryContext = parser.prog();
            ParseTreeWalker treeWalker = new ParseTreeWalker();

            // Walk over the parse tree:
            for (int i = 0; i < entryContext.getChildCount(); i++) {
                ParseTree parseTree = entryContext.getChild(i);
                treeWalker.walk(new JPlagRListener(this), parseTree);
            }
        } catch (IOException exception) {
            throw new ParsingException(file, exception.getMessage(), exception);
        }
    }

    /**
     * Adds a new {@link Token} to the current token list.
     * @param type the type of the new {@link Token}
     * @param line the line of the Token in the current file
     * @param start the start column of the Token in the line
     * @param length the length of the Token
     */
    /* package-private */ void addToken(TokenType type, int line, int start, int length) {
        tokens.add(new Token(type, currentFile, line, start, length));

    }
}
