/*
 * Decompiled with CFR 0.152.
 */
package de.jplag;

import de.jplag.SharedTokenType;
import de.jplag.Token;
import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.NoSuchFileException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import java.util.stream.Stream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class TokenPrinter {
    private static final Logger logger = LoggerFactory.getLogger(TokenPrinter.class);
    private static final String BAR = "|";
    private static final String TAB = "\t";
    private static final String SPACE = " ";
    private static final String NON_WHITESPACE = "\\S";
    private static final int MIN_PADDING = 1;
    private static final int TAB_LENGTH = 8;
    private static final String TAB_REPLACEMENT = " ".repeat(8);
    private static final boolean INDICATE_TINY_TOKEN = true;
    private static final boolean REPLACE_TABS = false;
    private static final boolean PRINT_EMPTY_LINES = true;
    private static final boolean SPACIOUS = true;

    private TokenPrinter() {
    }

    public static String printTokens(List<Token> tokens) {
        return TokenPrinter.printTokens(tokens, null);
    }

    public static String printTokens(List<Token> tokens, File rootDirectory) {
        return TokenPrinter.printTokens(tokens, rootDirectory, Optional.empty());
    }

    public static String printTokens(List<Token> tokenList, File rootDirectory, Optional<String> suffix) {
        PrinterOutputBuilder builder = new PrinterOutputBuilder();
        Map<File, List<Token>> fileToTokens = TokenPrinter.groupTokensByFile(tokenList);
        fileToTokens.forEach((file, fileTokens) -> {
            if (rootDirectory != null) {
                builder.append(rootDirectory.toPath().relativize(file.toPath()).toString());
            } else {
                builder.append("<unknown path>");
            }
            List<LineData> lineDatas = TokenPrinter.getLineData(fileTokens, suffix);
            lineDatas.forEach(lineData -> {
                builder.setLine(lineData.lineNumber());
                String currentLine = lineData.text();
                builder.appendCodeLine(currentLine);
                List<Token> tokens = lineData.tokens();
                if (tokens.isEmpty()) {
                    return;
                }
                builder.appendTokenLinePrefix();
                for (Token token : tokens) {
                    builder.advanceToTokenPosition(currentLine, token.getColumn(), true);
                    String stringRepresentation = TokenPrinter.getStringRepresentation(token);
                    builder.append(BAR).append(stringRepresentation);
                    int tokenEndIndex = token.getColumn() + token.getLength() - 1;
                    builder.advanceToTokenPosition(currentLine, tokenEndIndex, false);
                    if (!builder.positionBeforeOrEqualTo(tokenEndIndex)) continue;
                    builder.append(BAR);
                }
                builder.appendTokenLineSuffix();
                builder.advanceToNextLine();
            });
            builder.advanceToNextLine();
        });
        return builder.toString();
    }

    private static List<LineData> getLineData(List<Token> fileTokens, Optional<String> suffix) {
        File file = fileTokens.get(0).getFile();
        if (suffix.isPresent()) {
            file = new File(file.getPath() + suffix.get());
        }
        List<String> lines = TokenPrinter.linesFromFile(file);
        int currentLine = -1;
        HashMap lineNumbersToTokens = new HashMap(fileTokens.size());
        for (Token token : fileTokens) {
            if (token.getLine() != -1) {
                currentLine = token.getLine();
            }
            int line = token.getType() == SharedTokenType.FILE_END ? lines.size() : currentLine;
            ArrayList<Token> tokens = lineNumbersToTokens.containsKey(line) ? (List)lineNumbersToTokens.get(line) : new ArrayList<Token>();
            tokens.add(token);
            lineNumbersToTokens.put(line, tokens);
        }
        Stream<Integer> lineNumbers = IntStream.range(1, lines.size() + 1).boxed();
        return lineNumbers.map(lineIndex -> new LineData((Integer)lineIndex, (String)lines.get(lineIndex - 1), lineNumbersToTokens.getOrDefault(lineIndex, List.of()))).toList();
    }

    private static Map<File, List<Token>> groupTokensByFile(List<Token> tokens) {
        return tokens.stream().collect(Collectors.groupingBy(Token::getFile));
    }

    private static String getStringRepresentation(Token token) {
        String description = token.getType().getDescription();
        return token.getLength() <= 1 ? description.toLowerCase() : description;
    }

    private static List<String> linesFromFile(File file) {
        try {
            return Files.readAllLines(file.toPath());
        }
        catch (NoSuchFileException exception) {
            logger.error("File does not exist, thus no tokens are printed: " + file.getAbsolutePath());
        }
        catch (IOException exception) {
            logger.error("Cannot read " + file.getAbsolutePath() + ":", (Throwable)exception);
        }
        return Collections.emptyList();
    }

    private static final class PrinterOutputBuilder {
        public static final String LINE_SEPARATOR = System.lineSeparator();
        private final StringBuilder builder = new StringBuilder();
        private int columnIndex = 1;
        private int lineNumber;
        private int trailingLineSeparators = 0;

        private PrinterOutputBuilder() {
        }

        private static int digitCount(int number) {
            if (number == 0) {
                return 1;
            }
            int minusLength = number < 0 ? 1 : 0;
            return (int)Math.log10(Math.abs(number)) + minusLength + 1;
        }

        private void resetLinePosition() {
            this.columnIndex = 1;
        }

        public PrinterOutputBuilder append(String str) {
            int n = this.trailingLineSeparators = str.equals(LINE_SEPARATOR) ? this.trailingLineSeparators + 1 : 0;
            if (this.trailingLineSeparators >= 3) {
                return this;
            }
            this.builder.append(str);
            this.columnIndex += str.length();
            return this;
        }

        public PrinterOutputBuilder append(int i) {
            return this.append(Integer.toString(i));
        }

        private PrinterOutputBuilder appendTokenLineSuffix() {
            return this.advanceToNextLine();
        }

        private PrinterOutputBuilder appendCodeLine(String currentLine) {
            this.advanceToNextLine().append(this.lineNumber);
            int paddingLength = Math.max(8 - PrinterOutputBuilder.digitCount(this.lineNumber), 1);
            this.appendPadding(paddingLength);
            return this.append(currentLine);
        }

        PrinterOutputBuilder advanceToTokenPosition(String currentLine, int targetPosition, boolean breakLine) {
            if (!this.positionBeforeOrEqualTo(targetPosition = Math.min(targetPosition, currentLine.length()))) {
                if (!breakLine) {
                    return this;
                }
                this.appendTokenLinePrefix();
                if (!this.positionBeforeOrEqualTo(targetPosition)) {
                    return this;
                }
            }
            String padding = currentLine.substring(this.columnIndex - 1, targetPosition - 1).replace(TokenPrinter.TAB, TokenPrinter.TAB).replaceAll(TokenPrinter.NON_WHITESPACE, TokenPrinter.SPACE);
            this.append(padding);
            return this;
        }

        private boolean positionBeforeOrEqualTo(int targetPosition) {
            return this.columnIndex <= targetPosition;
        }

        PrinterOutputBuilder appendTokenLinePrefix() {
            int paddingLength = Math.max(PrinterOutputBuilder.digitCount(this.lineNumber) + 1, 8);
            this.advanceToNextLine().appendPadding(paddingLength);
            this.resetLinePosition();
            return this;
        }

        private PrinterOutputBuilder appendPadding(int paddingLength) {
            String padding = TokenPrinter.SPACE.repeat(paddingLength);
            this.append(padding);
            return this;
        }

        private PrinterOutputBuilder advanceToNextLine() {
            this.append(LINE_SEPARATOR);
            this.resetLinePosition();
            return this;
        }

        public void setLine(int lineNumber) {
            this.lineNumber = lineNumber;
        }

        public String toString() {
            return this.builder.toString();
        }
    }

    private record LineData(Integer lineNumber, String text, List<Token> tokens) {
    }
}

