/**
 * I18nMultiFile.java
 *
 * Copyright 2016 Joshua Schnabel
 * 
 * Created: 05.12.2016 14:00:10
 * Part of: jsI18n
 * 
 * For licence informations check the LICENCE file!
 */
package de.joshuaschnabel.l18n;

import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.charset.Charset;
import java.util.Locale;
import java.util.Set;

import com.eclipsesource.json.Json;

import de.joshuaschnabel.l18n.exception.LanguageNotSupportedException;
import de.joshuaschnabel.l18n.json.JsonTranslationDeserializer;
import de.joshuaschnabel.l18n.scanner.ResourceScanner;

/**
 * @author Joshua Schnabel
 *
 *         TODO Description
 *
 */
public class I18nMultiFile extends I18n
{

	/**
	 * Description: {@link I18nMultiFile}
	 *
	 * @param defaultLanguage
	 *           - Locale of the default language
	 */
	public I18nMultiFile(Locale defaultLanguage)
	{
		super(defaultLanguage);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see de.joshuaschnabel.l18n.I18n#checkIfLanguageIsLoaded(java.util.Locale)
	 */
	@Override
	protected void checkIfLanguageIsLoaded(Locale language)
	{
		if (!this.getSupportedLanguages().contains(language))
			throw new LanguageNotSupportedException("Language " + language.getDisplayName() + " is not supported");

		this.getTranslations().put(language, loadLanguageFromMultiFile(this.getLanguages().get(language)));
	}

	private static Translation loadLanguageFromMultiFile(String filePath)
	{
		Translation translation;

		InputStream is = I18n.class.getClass().getResourceAsStream("/" + filePath);

		try (InputStreamReader reader = new InputStreamReader(is, Charset.forName("UTF-8")))
		{
			translation = JsonTranslationDeserializer.deserialiseTranslation(Json.parse(reader).asObject());
		}
		catch (IOException e)
		{
			throw new IllegalStateException("Error while loading '" + filePath + "'! (" + e.toString() + ")");
		}
		return translation;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see de.joshuaschnabel.l18n.I18n#loadLanguages()
	 */
	@Override
	protected void loadLanguages()
	{
		Set<String> filelist = ResourceScanner.getResources("jsI18n.translations", ".*\\.tns");
		for (String filePath : filelist)
		{
			Translation translation = loadLanguageFromMultiFile(filePath);

			for (Locale locale : translation.getLanguages())
			{
				this.addSupportedLanguages(locale, filePath);
			}
		}
	}

}
