/**
 * I18n.java
 *
 * Copyright 2016 Joshua Schnabel
 * 
 * Created: 02.12.2016 18:21:35
 * Part of: jsI18n
 * 
 * For licence informations check the LICENCE file!
 */
package de.joshuaschnabel.l18n;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;

import de.joshuaschnabel.l18n.exception.LanguageNotSupportedException;

/**
 * @author Joshua Schnabel
 *
 *         TODO Description
 *
 */
public abstract class I18n
{
	private Locale							defaultLanguage;
	private Locale							currentLanguage;
	private Map<Locale, String>		supportedLanguages	= new HashMap<>();
	private Map<Locale, Translation>	translations			= new HashMap<>();

	/**
	 * See {@link I18n} for description. The languages will be loaded from
	 * /jsI18n/i18n.tns <br>
	 * 
	 * @param defaultLanguage
	 *           - default Language if system language is not supported
	 */
	public I18n(Locale defaultLanguage)
	{
		loadLanguages();
		this.defaultLanguage = defaultLanguage;
		selectLanguage();
	}

	private static Locale getSystemLanguage()
	{
		return Locale.getDefault(Locale.Category.DISPLAY);
	}

	private void selectLanguage() throws IllegalStateException
	{
		try
		{
			setLanguage(getSystemLanguage());
		}
		catch (@SuppressWarnings("unused") LanguageNotSupportedException e)
		{
			try
			{
				setLanguage(this.defaultLanguage);
			}
			catch (LanguageNotSupportedException e1)
			{
				throw new IllegalStateException("Default-language not supported (" + e1.getMessage() + ")");
			}
		}
	}

	/**
	 * @param language
	 *           - language which should be set
	 * @throws LanguageNotSupportedException
	 *            if the language is not supported
	 */
	public void setLanguage(Locale language) throws LanguageNotSupportedException
	{
		if (this.getSupportedLanguages().contains(language))
		{
			this.currentLanguage = language;
		}
		else
		{
			throw new LanguageNotSupportedException(language + " not supported");
		}
	}

	/**
	 * @return the defaultLanguage
	 */
	public Locale getDefaultLanguage()
	{
		return this.defaultLanguage;
	}

	/**
	 * @return The current language
	 */
	public Locale getCurrentLanguage()
	{
		return this.currentLanguage;
	}

	/**
	 * @param identifier
	 *           - the identifier of the language string
	 * @return the language string
	 */
	public String getTranslationString(String identifier)
	{
		if (getTranslationString(identifier, this.currentLanguage) == null)
		{
			return getTranslationString(identifier, this.defaultLanguage);
		}
		return getTranslationString(identifier, this.currentLanguage);
	}

	/**
	 * @param identifier
	 *           - the identifier of the language string
	 * @param language
	 *           - Locale of the wanted language
	 * @return the language string
	 * @throws LanguageNotSupportedException
	 *            if the language is not supported
	 */
	public String getTranslationString(String identifier, Locale language) throws LanguageNotSupportedException
	{
		checkIfLanguageIsLoaded(language);
		return this.getTranslations().get(language).getString(identifier);
	}

	/**
	 * @param identifier
	 *           - the identifier of the language string
	 * @param args
	 *           - arguments referenced by the format specifiers in the language
	 *           string. {@link String#format(String, Object...)}
	 * @return the formatted language string
	 */
	public String getTranslationString(String identifier, Object... args)
	{
		return String.format(this.currentLanguage, getTranslationString(identifier), args);
	}

	protected abstract void checkIfLanguageIsLoaded(Locale language);

	protected abstract void loadLanguages();

	/**
	 * @param translations
	 *           the translations to set
	 */
	protected void setTranslations(Map<Locale, Translation> translations)
	{
		this.translations = translations;
	}

	/**
	 * @param supportedLanguages
	 *           the supportedLanguages to set
	 */
	protected void setSupportedLanguages(Map<Locale, String> supportedLanguages)
	{
		this.supportedLanguages = supportedLanguages;
	}

	protected Map<Locale, String> getLanguages()
	{
		return this.supportedLanguages;
	}

	/**
	 * @return the supportedLanguages
	 */
	public List<Locale> getSupportedLanguages()
	{
		return Collections.unmodifiableList(new ArrayList<>(this.supportedLanguages.keySet()));
	}

	protected void addSupportedLanguages(Locale locale, String path)
	{
		this.supportedLanguages.put(locale, path);
	}

	/**
	 * @return the translations
	 */
	protected Map<Locale, Translation> getTranslations()
	{
		return this.translations;
	}
}
