package de.jensklingenberg.ktorfit

import de.jensklingenberg.ktorfit.Strings.Companion.EXPECTED_URL_SCHEME
import de.jensklingenberg.ktorfit.converter.ResponseConverter
import de.jensklingenberg.ktorfit.converter.request.RequestConverter
import io.ktor.client.*
import io.ktor.client.engine.*


/**
 * Main class for Ktorfit, create the class than use the [create<T>()] function.
 */
class Ktorfit private constructor(
    val baseUrl: String,
    val httpClient: HttpClient = HttpClient(),
    val requestConverters: Set<RequestConverter>,
    val responseConverters: Set<ResponseConverter>,
) {

    /**
     * Builder class for Ktorfit.
     *
     * @see baseUrl
     * @see httpClient
     */
    class Builder {
        private var _baseUrl: String = ""
        private var _httpClient = HttpClient()
        private var _responseConverter: MutableSet<ResponseConverter> = mutableSetOf()
        private var _requestConverter: MutableSet<RequestConverter> = mutableSetOf()

        /**
         * That will be used for every request with object
         */
        fun baseUrl(url: String) = apply {
            if (url.isEmpty()) {
                throw IllegalStateException("Base URL required")
            }

            if (!url.endsWith("/")) {
                throw IllegalStateException("Base URL needs to end with /")
            }
            if (!url.startsWith("http") && !url.startsWith("https")) {
                throw IllegalStateException(EXPECTED_URL_SCHEME)
            }
            this._baseUrl = url
        }

        /**
         * Client that will be used for every request with object
         */
        fun httpClient(client: HttpClient) = apply {
            this._httpClient = client
        }

        /**
         * Build HttpClient by just passing an engine
         */
        fun httpClient(engine: HttpClientEngine) = apply {
            this._httpClient = HttpClient(engine)
        }

        /**
         * Build HttpClient by just passing an engine factory
         */
        fun <T : HttpClientEngineConfig> httpClient(engineFactory: HttpClientEngineFactory<T>) = apply {
            this._httpClient = HttpClient(engineFactory)
        }

        /**
         * Client-Builder that will be used for every request with object
         */
        fun httpClient(config: HttpClientConfig<*>.() -> Unit) = apply {
            this._httpClient = HttpClient(this._httpClient.engine, config)
        }

        /**
         * Client-Builder with engine that will be used for every request with object
         */
        fun httpClient(engine: HttpClientEngine, config: HttpClientConfig<*>.() -> Unit) = apply {
            this._httpClient = HttpClient(engine, config)
        }

        /**
         * Client-Builder with engine factory that will be used for every request with object
         */
        fun <T : HttpClientEngineConfig> httpClient(
            engineFactory: HttpClientEngineFactory<T>,
            config: HttpClientConfig<T>.() -> Unit
        ) = apply {
            this._httpClient = HttpClient(engineFactory, config)
        }

        /**
         * Use this to add [RequestConverter] for unsupported return types of requests
         */
        fun requestConverter(vararg converters: RequestConverter) = apply {
            converters.forEach { converter ->
                this._requestConverter.add(converter)
            }
        }

        /**
         * Use this to add [ResponseConverter] for unsupported return types of response
         */
        fun responseConverter(vararg converters: ResponseConverter) = apply {
            converters.forEach { converter ->
                this._responseConverter.add(converter)
            }
        }


        /**
         * Apply changes to builder and get the Ktorfit instance without the need of calling [build] afterwards.
         */
        fun build(builder: Builder.() -> Unit) = this.apply(builder).build()

        /**
         * Creates an instance of Ktorfit with specified baseUrl and HttpClient.
         */
        fun build(): Ktorfit {
            return Ktorfit(_baseUrl, _httpClient, _requestConverter, _responseConverter)
        }
    }
}

/**
 * Create a Ktorfit instance using Kotlin-DSL.
 */
fun ktorfit(builder: Ktorfit.Builder.() -> Unit) = Ktorfit.Builder().apply(builder).build()

/**
 * Creates a Ktorfit Builder instance using Kotlin-DSL.
 */
fun ktorfitBuilder(builder: Ktorfit.Builder.() -> Unit) = Ktorfit.Builder().apply(builder)

/**
 * This will make IntelliJ think that this function exists.
 * The real implementation will be generated by the KSP plugin
 * Ktorfit will return an implementation of type [T] for a requested interface
 *
 * val ktorfit = Ktorfit("example.com")
 * val testApi = ktorfit.create<TestApi>()
 */

inline fun <reified T> Ktorfit.create(): T {
    throw NotImplementedError("Ktorfit didn't generate Code for " + T::class.simpleName + " You need to apply the KSP Plugin")
}
