package de.jensklingenberg.ktorfit

import de.jensklingenberg.ktorfit.adapter.CallAdapter
import io.ktor.client.call.*
import io.ktor.client.request.*
import io.ktor.client.request.forms.*
import io.ktor.client.statement.*
import io.ktor.http.*
import io.ktor.http.content.*
import io.ktor.util.*

@InternalKtorfitApi
class KtorfitClient(val ktorfit: Ktorfit){

    val httpClient = ktorfit.httpClient

    /**
     * Converts [value] to an URL encoded value
     */
    fun encode(value: Any): String {
        return value.toString().encodeURLParameter()
    }

    /**
     * Converts [value] to an Map with URL encoded values
     */
    fun encodeMap(values: Map<String, Any>): Map<String, String> {
        return values.mapKeys { it.key.encodeURLParameter() }.mapValues { it.value.toString().encodeURLParameter() }
    }

    /**
     * This will handle all requests for functions with suspend modifier
     */
    suspend inline fun <reified TReturn, reified PRequest> suspendRequest(
        requestData: RequestData
    ): TReturn {

        val request = httpRequest(requestData)

        ktorfit.getAdapterList().firstOrNull {
            it.getSupported().contains(CallAdapter.SupportType.CONVERTRESPONSE) && it.supportedType(
                requestData.qualifiedRawTypeName
            )
        }?.let {
            return it.convertResponse(request.body<PRequest>() as Any) as TReturn
        }

        return request.body()

    }


//T is return type
    //P is requested type
    /**
     * This will handle all requests for functions without suspend modifier
     */

    inline fun <reified TReturn, reified PRequest> request(
        requestData: RequestData
    ): TReturn {
        ktorfit.getAdapterList().firstOrNull {
            it.getSupported().contains(CallAdapter.SupportType.CONVERTSUSPENDCALL) && it.supportedType(
                requestData.qualifiedRawTypeName
            )
        }?.let {
            return it.convertSuspendCall(requestData.qualifiedRawTypeName) {
                val response = httpRequest(requestData)
                val data = response.body<PRequest>()!!
                Pair(data, response)
            } as TReturn
        }

        throw IllegalArgumentException("Add a CallAdapter for " + requestData.qualifiedRawTypeName + " or make function suspend")
    }

    suspend inline fun httpRequest(
        requestData: RequestData
    ): HttpResponse {
        return httpClient.request {
            requestBuilder(requestData)
        }
    }

    /**
     * This is used for requests with @Streaming annotation
     */

    suspend inline fun prepareRequest(
        requestData: RequestData
    ): HttpStatement {
        return httpClient.prepareRequest {
            requestBuilder(requestData)
        }
    }

    @OptIn(InternalAPI::class)
    fun HttpRequestBuilder.requestBuilder(
        requestData: RequestData
    ) {
        requestData.headers.forEach {
            header(it.key, it.value)
        }

        if (requestData.fields.isNotEmpty()) {
            val formParameters = Parameters.build {
                requestData.fields.forEach {
                    append(it.key, it.value)
                }
            }
            if (requestData.method == HttpMethod.Post.value ||
                requestData.method == HttpMethod.Put.value ||
                requestData.method == HttpMethod.Patch.value
            ) {
                setBody(FormDataContent(formParameters))
            } else {
                this.url.parameters.appendAll(formParameters)
            }
        }

        if (requestData.parts.isNotEmpty()) {

            val partData =
                requestData.parts.values.filterIsInstance<List<*>>().map { it as List<PartData> }.flatten()


            val formData = formData {
                requestData.parts.filter { it.value is String }.forEach {
                    this@formData.append(it.key, it.value)
                }
            }
            val partDataList = formData + partData
            setBody(MultiPartFormDataContent(partDataList))
        }
        this.method = HttpMethod.parse(requestData.method)

        requestData.bodyData?.let {
            setBody(it)
        }

        requestData.queries.forEach {
            parameter(it.key, it.value)
        }

        url(ktorfit.baseUrl + requestData.relativeUrl)
        requestData.requestBuilder(this)
    }

}