package de.jensklingenberg.ktorfit

import de.jensklingenberg.ktorfit.converter.CoreResponseConverter
import de.jensklingenberg.ktorfit.converter.ResponseConverter
import de.jensklingenberg.ktorfit.converter.SuspendResponseConverter
import io.ktor.client.*
import io.ktor.client.engine.*


/**
 * Main class for Ktorfit, create the class than use the [create<T>()] function.
 */
class Ktorfit private constructor(
    val baseUrl: String,
    val httpClient: HttpClient = HttpClient(),
    val responseConverters: Set<ResponseConverter>,
    val suspendResponseConverters: Set<SuspendResponseConverter>
) {

    @Deprecated(
        "Use Ktorfit.Builder()",
        replaceWith = ReplaceWith("Ktorfit.Builder().baseUrl(baseUrl).httpClient(httpClient).build()")
    )
    constructor(
        baseUrl: String,
        httpClient: HttpClient = HttpClient()
    ) : this(baseUrl, httpClient, emptySet(), emptySet())

    /**
     * Builder class for Ktorfit.
     *
     * @see baseUrl
     * @see httpClient
     */
    class Builder {
        private lateinit var _baseUrl: String
        private var _httpClient = HttpClient()
        private var _responseConverter: MutableSet<ResponseConverter> = mutableSetOf()
        private var _suspendResponseConverter: MutableSet<SuspendResponseConverter> = mutableSetOf()

        /**
         * That will be used for every request with object
         */
        fun baseUrl(url: String) = apply {
            this._baseUrl = url
        }

        /**
         * Client that will be used for every request with object
         */
        fun httpClient(client: HttpClient) = apply {
            this._httpClient = client
        }

        /**
         * Build HttpClient by just passing an engine
         */
        fun httpClient(engine: HttpClientEngine) = apply {
            this._httpClient = HttpClient(engine)
        }

        /**
         * Client-Builder that will be used for every request with object
         */
        fun httpClient(config: HttpClientConfig<*>.() -> Unit) = apply {
            this._httpClient = HttpClient(this._httpClient.engine, config)
        }

        /**
         * Client-Builder with engine that will be used for every request with object
         */
        fun httpClient(engine: HttpClientEngine, config: HttpClientConfig<*>.() -> Unit) = apply {
            this._httpClient = HttpClient(engine, config)
        }

        /**
         * Use this to add [ResponseConverter] or [SuspendResponseConverter] for unsupported return types of requests
         */
        fun responseConverter(vararg converters: CoreResponseConverter) = apply {
            converters.forEach { converter ->
                when (converter) {
                    is ResponseConverter -> this._responseConverter.add(converter)
                    is SuspendResponseConverter -> this._suspendResponseConverter.add(converter)
                    else -> throw IllegalArgumentException("Your response converter must be either of type ResponseConverter or SuspendRespondConverter")
                }
            }
        }

        /**
         * Apply changes to builder and get the Ktorfit instance without the need of calling [build] afterwards.
         */
        fun build(builder: Builder.() -> Unit) = this.apply(builder).build()

        /**
         * Creates an instance of Ktorfit with specified baseUrl and HttpClient.
         */
        fun build(): Ktorfit {
            if (!this::_baseUrl.isInitialized || this._baseUrl.isEmpty()) {
                throw IllegalStateException("Base URL required")
            }

            if (!_baseUrl.endsWith("/")) {
                throw IllegalStateException("Base URL needs to end with /")
            }

            return Ktorfit(_baseUrl, _httpClient, _responseConverter, _suspendResponseConverter)
        }
    }
}

/**
 * Create a Ktorfit instance using Kotlin-DSL.
 */
fun ktorfit(builder: Ktorfit.Builder.() -> Unit) = Ktorfit.Builder().apply(builder).build()

/**
 * Creates a Ktorfit Builder instance using Kotlin-DSL.
 */
fun ktorfitBuilder(builder: Ktorfit.Builder.() -> Unit) = Ktorfit.Builder().apply(builder)

/**
 * This will make IntelliJ think that this function exists.
 * The real implementation will be generated by the KSP plugin
 * Ktorfit will return an implementation of type [T] for a requested interface
 *
 * val ktorfit = Ktorfit("example.com")
 * val testApi = ktorfit.create<TestApi>()
 */

inline fun <reified T> Ktorfit.create(): T {
    throw NotImplementedError("Ktorfit didn't generate Code for " + T::class.simpleName + " You need to apply the KSP Plugin")
}
