package de.jensklingenberg.ktorfit.parser

import com.google.devtools.ksp.processing.KSPLogger
import com.google.devtools.ksp.symbol.KSValueParameter
import de.jensklingenberg.ktorfit.getBodyAnnotation
import de.jensklingenberg.ktorfit.getFieldAnnotation
import de.jensklingenberg.ktorfit.getFieldMapAnnotation
import de.jensklingenberg.ktorfit.getHeaderMapAnnotation
import de.jensklingenberg.ktorfit.getHeadersAnnotation
import de.jensklingenberg.ktorfit.getPartAnnotation
import de.jensklingenberg.ktorfit.getPartMapAnnotation
import de.jensklingenberg.ktorfit.getPathAnnotation
import de.jensklingenberg.ktorfit.getQueryAnnotation
import de.jensklingenberg.ktorfit.getQueryMapAnnotation
import de.jensklingenberg.ktorfit.getQueryNameAnnotation
import de.jensklingenberg.ktorfit.getRequestBuilderAnnotation
import de.jensklingenberg.ktorfit.getUrlAnnotation
import de.jensklingenberg.ktorfit.model.MyParam
import de.jensklingenberg.ktorfit.model.MyType
import de.jensklingenberg.ktorfit.model.annotations.ParamAnnotation
import de.jensklingenberg.ktorfit.resolveTypeName
import ktorfitError

fun getMyParamList(ksValueParameterList: List<KSValueParameter>, logger: KSPLogger): List<MyParam> {
    return ksValueParameterList.map {
        val pararamAnnos = getParamAnnotationList(it, logger)

        val reqBuilderAnno = it.getRequestBuilderAnnotation()
        val parameterName = it.name?.asString() ?: ""

        if (pararamAnnos.isEmpty() && reqBuilderAnno == null) {
            logger.error("No Ktorfit Annotation found at " + parameterName + " " + it.parent.toString(), it)
        }

        val type = if (reqBuilderAnno != null) {
            if (it.type.resolve()
                    .resolveTypeName() != "[@kotlin.ExtensionFunctionType] Function1<HttpRequestBuilder, Unit>"
            ) {
                logger.ktorfitError("@ReqBuilder parameter type needs to be HttpRequestBuilder.()->Unit",it)
                throw java.lang.IllegalStateException()
            } else {
                MyType(
                    "HttpRequestBuilder.()->Unit",
                    "HttpRequestBuilder.()->Unit"
                )
            }

        } else {
            MyType(
                it.type.resolve().resolveTypeName(),
                it.type.resolve().declaration.qualifiedName?.asString() ?: ""
            )
        }

        MyParam(parameterName, type, pararamAnnos, reqBuilderAnno)
    }
}


fun getParamAnnotationList(ksValueParameter: KSValueParameter, logger: KSPLogger): List<ParamAnnotation> {
    val pararamAnnos = mutableListOf<ParamAnnotation>()
    ksValueParameter.getBodyAnnotation()?.let {
        pararamAnnos.add(it)
    }

    ksValueParameter.getPathAnnotation()?.let {
        if (ksValueParameter.type.resolve().isMarkedNullable) {
            logger.ktorfitError("Path parameter type may not be nullable", ksValueParameter.type)
        }
        pararamAnnos.add(it)
    }

    ksValueParameter.getHeadersAnnotation()?.let {
        pararamAnnos.add(it)
    }

    ksValueParameter.getHeaderMapAnnotation()?.let {
        //TODO: Find out how isAssignableFrom works
        if (!ksValueParameter.type.toString().endsWith("Map")) {
            logger.ktorfitError("@HeaderMap parameter type must be Map.", ksValueParameter)
        }
        pararamAnnos.add(it)
    }

    ksValueParameter.getQueryAnnotation()?.let {
        pararamAnnos.add(it)
    }

    ksValueParameter.getQueryNameAnnotation()?.let {
        pararamAnnos.add(it)
    }

    ksValueParameter.getQueryMapAnnotation()?.let {
        if (!ksValueParameter.type.toString().endsWith("Map")) {
            logger.error("@QueryMap parameter type must be Map.", ksValueParameter)
        }
        if (ksValueParameter.type.resolve().arguments.first().type.toString() != "String") {
            logger.error("@QueryMap keys must be of type String:", ksValueParameter)
        }
        pararamAnnos.add(it)
    }

    ksValueParameter.getFieldAnnotation()?.let {
        pararamAnnos.add(it)
    }

    ksValueParameter.getFieldMapAnnotation()?.let {
        if (!ksValueParameter.type.toString().endsWith("Map")) {
            logger.error("@FieldMap parameter type must be Map.", ksValueParameter)
        }
        pararamAnnos.add(it)
    }

    ksValueParameter.getPartAnnotation()?.let {
        if (ksValueParameter.type.resolve().isMarkedNullable) {
            logger.ktorfitError("Part parameter type may not be nullable", ksValueParameter.type)
        }
        pararamAnnos.add(it)
    }

    ksValueParameter.getPartMapAnnotation()?.let {
        if (!ksValueParameter.type.toString().endsWith("Map")) {
            logger.error("@PartMap parameter type must be Map.", ksValueParameter)
        }
        pararamAnnos.add(it)
    }

    ksValueParameter.getUrlAnnotation()?.let {
        pararamAnnos.add(it)
    }
    return pararamAnnos
}




