import com.google.devtools.ksp.KspExperimental
import com.google.devtools.ksp.closestClassDeclaration
import com.google.devtools.ksp.processing.CodeGenerator
import com.google.devtools.ksp.processing.Dependencies.Companion.ALL_FILES
import com.google.devtools.ksp.processing.KSPLogger
import com.google.devtools.ksp.processing.Resolver
import com.google.devtools.ksp.processing.SymbolProcessor
import com.google.devtools.ksp.processing.SymbolProcessorEnvironment
import com.google.devtools.ksp.processing.SymbolProcessorProvider
import com.google.devtools.ksp.symbol.KSAnnotated
import com.google.devtools.ksp.symbol.KSFunctionDeclaration
import com.google.devtools.ksp.symbol.KSNode
import de.jensklingenberg.ktorfit.generator.generateClassImpl
import de.jensklingenberg.ktorfit.generator.generateHttpExtSource
import de.jensklingenberg.ktorfit.http.DELETE
import de.jensklingenberg.ktorfit.http.GET
import de.jensklingenberg.ktorfit.http.HEAD
import de.jensklingenberg.ktorfit.http.HTTP
import de.jensklingenberg.ktorfit.http.OPTIONS
import de.jensklingenberg.ktorfit.http.PATCH
import de.jensklingenberg.ktorfit.http.POST
import de.jensklingenberg.ktorfit.http.PUT
import de.jensklingenberg.ktorfit.parser.readClass
import java.io.OutputStreamWriter


public class KtorfitProcessorProvider : SymbolProcessorProvider {
    override fun create(environment: SymbolProcessorEnvironment): SymbolProcessor {
        return KtorfitProcessor(environment)
    }
}

public class KtorfitProcessor(private val env: SymbolProcessorEnvironment) : SymbolProcessor {
    private val codeGenerator: CodeGenerator = env.codeGenerator
    private val logger: KSPLogger = env.logger
    private var invoked = false

    private lateinit var myResolver: Resolver

    override fun process(resolver: Resolver): List<KSAnnotated> {
        myResolver = resolver
        val allFiles = resolver.getAllFiles().map { it.fileName }
        logger.warn(allFiles.toList().toString())
        if (invoked) {
            return emptyList()
        }
        invoked = true

        return emptyList()
    }

    @OptIn(KspExperimental::class)
    private fun getAnnotatedFunctions(): List<KSFunctionDeclaration> {
        val getAnnotated = myResolver.getSymbolsWithAnnotation(GET::class.java.name).toList()
        val postAnnotated = myResolver.getSymbolsWithAnnotation(POST::class.java.name).toList()
        val putAnnotated = myResolver.getSymbolsWithAnnotation(PUT::class.java.name).toList()
        val deleteAnnotated = myResolver.getSymbolsWithAnnotation(DELETE::class.java.name).toList()
        val headAnnotated = myResolver.getSymbolsWithAnnotation(HEAD::class.java.name).toList()
        val optionsAnnotated = myResolver.getSymbolsWithAnnotation(OPTIONS::class.java.name).toList()
        val patchAnnotated = myResolver.getSymbolsWithAnnotation(PATCH::class.java.name).toList()
        val httpAnnotated = myResolver.getSymbolsWithAnnotation(HTTP::class.java.name).toList()

        val ksAnnotatedList =
            getAnnotated + postAnnotated + putAnnotated + deleteAnnotated + headAnnotated + optionsAnnotated + patchAnnotated + httpAnnotated
        return ksAnnotatedList.map { it as KSFunctionDeclaration }
    }


    override fun finish() {

        val myClasses = getAnnotatedFunctions().groupBy { it.closestClassDeclaration()!! }
            .map { (classDec) ->
                if(classDec.origin.name=="JAVA"){
                    logger.ktorfitError("Java Interfaces are not supported", classDec)
                }
                readClass(classDec, logger)
            }

        generateClassImpl(myClasses, codeGenerator, logger)


        val source = generateHttpExtSource(myClasses, env.platforms.any { it.platformName == "JS" })
        codeGenerator.createNewFile(ALL_FILES, "de.jensklingenberg.ktorfit", "KtorfitExt", "kt").use { output ->
            OutputStreamWriter(output).use { writer ->
                writer.write(source)
            }
        }

    }
}

public fun KSPLogger.ktorfitError(s: String, classDec: KSNode) {
    this.error("Ktorfit: $s",classDec)
}
