package com.inet.pdfc.junit;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.nio.file.Path;
import java.util.Properties;

import org.junit.Assert;

import com.inet.logging.Logger;
import com.inet.pdfc.remote.client.PdfcServerInfo;
import com.inet.pdfc.remote.client.RPCClient;
import com.inet.pdfc.remote.client.RPCClient.PdfData;

import junit.framework.AssertionFailedError;

/**
 * A set of assertion methods to check two PDF files for equality.
 */
public class PdfcAssert {

    private static Properties     configuration;
    private static String     configurationName;

    private static PdfcServerInfo server = new PdfcServerInfo();

    /**
     * Defines the configuration for all subsequent calls of the assert API
     * 
     * @param configName the name/path of the configuration file. A <code>null</code> value will reset the internal
     *            configuration to default.
     * @throws IOException in case the designated file is not readable or not a valid XML file
     */
    public static void setPdfcConfiguration( String configName ) throws IOException {
        if( configName == null ) {
            configuration = null;
        } else {
            try( FileInputStream fileInputStream = new FileInputStream( configName ) ) {
                Properties props = new Properties();
                props.loadFromXML( fileInputStream );
                String val = props.getProperty( "LOG_LEVEL" );
                if( val != null ) {
                    if( "OFF".equals( val ) ) {
                        val = "STATUS";
                    }
                    if( "ALL".equals( val ) ) {
                        val = "DEBUG";
                    }
                    props.setProperty( "LOG_LEVEL", val );
                }
                configuration = props;
            }
        }
        configurationName = null;
    }

    /**
     * Set the compare profile with the name or id.
     * This profile should be for the login user available. The available names can you see at http://&lt;pdfcserver:port&gt;/pdfcapi/listconfig
     *
     * If it didn't found the profile it throw an exception
     * with all user available configuration
     *
     * @param nameOrID name or id, case insensitive
     */
    public static void setPdfcConfigurationWithName(final String nameOrID){
        configurationName = nameOrID;
        configuration = null;
    }

    /**
     * Defines the configuration for all subsequent calls of the assert API
     * 
     * @param settings the configuration properties to be used
     */
    public static void setPdfcConfiguration( Properties settings ) {
        configuration = settings;
    }

    /**
     * Set the path to the PDFC server RPC application. If not set multicast DNS is used to search the server. Possible values are:
     * <ul>
     * <li>myhost:9900
     * <li>myhost:9900/pdfcrpcws
     * <li>http://myhost:9900
     * <li>http://myhost:9900/pdfcrpcws
     * </ul>
     * 
     * @param servicePath the path to the service
     */
    public static void setServicePath( String servicePath ) {
        server.setServicePath( servicePath );
    }

    /**
     * Set the log level.
     * 
     * @param level new level, once of the constant in {@link Logger}
     */
    public static void setLogLevel( int level ) {
        RPCClient.getLogger().setLogLevel( level );
    }

    /**
     * Set the login credentials
     * 
     * @param user the user name
     * @param password the password
     */
    public static void setCredentials( String user, String password ) {
        server.setCredentials( user, password );
    }

    /**
     * Asserts that two pdf documents contain no/some differences. Submits an optional message in case of
     * {@link AssertionFailedError}.
     * 
     * @param message an optional message in case of fail
     * @param expected the pdf file holding the expected content
     * @param actual the pdf file holding the actual content
     * @param expectEquality whether zero or nonzero differences are expected
     * @throws AssertionError if the test fail
     */
    private static void testEquality( String message, PdfData expected, PdfData actual, boolean expectEquality ) {

        if( expected == null ) {
            Assert.fail( "The 'expected' file is not set." );
        }
        if( actual == null ) {
            Assert.fail( "The 'actual' file is not set." );
        }
        try (RPCClient client = new RPCClient() ) {
            client.setServerInfo( server );
            if(configuration != null) {
                client.setConfiguration( configuration );
            }else{
                client.setConfigName( configurationName );
            }
            client.setActual( actual );
            client.setExpected( expected );
            int diffCount;
            try {
                diffCount = client.runComparison();
                Throwable error = client.getError();
                if( diffCount < 0 && error != null ) {
                    throw new AssertionError( error );
                }
                if( expectEquality ) {
                    Assert.assertEquals( message, 0, diffCount );
                } else {
                    Assert.assertNotEquals( message, 0, diffCount );
                }
            } catch( IOException ex ) {
                client.cancel();
                throw ex;
            }
        } catch( IOException ex ) {
            throw new AssertionError( ex );
        }
    }

    /**
     * Asserts that two pdf documents contain no differences
     * 
     * @param expected the pdf file with the expected content
     * @param actual the pdf file with the actual content
     */
    public static void assertPdfEquals( File expected, File actual ) {
        assertPdfEquals( null, expected, actual );
    }

    /**
     * Asserts that two pdf documents contain no differences an AssertionFailedError is thrown with the given message
     * 
     * @param message the message in case of a fail
     * @param expected the pdf file with the expected content
     * @param actual the pdf file with the actual content
     */
    public static void assertPdfEquals( String message, File expected, File actual ) {
        testEquality( message, new PdfData( expected ), new PdfData( actual ), true );
    }

    /**
     * Asserts that two pdf documents given by their path contain no differences
     * 
     * @param expected the path of the pdf file with the expected content
     * @param actual the path of the pdf file with the actual content
     */
    public static void assertPdfEquals( String expected, String actual ) {
        assertPdfEquals( null, expected, actual );
    }

    /**
     * Asserts that two pdf documents given by their path contain no differences an AssertionFailedError is thrown with
     * the given message
     * 
     * @param message the message in case of a fail
     * @param expected the path of the pdf file with the expected content
     * @param actual the path of the pdf file with the actual content
     */
    public static void assertPdfEquals( String message, String expected, String actual ) {
        assertPdfEquals( message, new File( expected ), new File( actual ) );
    }

    /**
     * Asserts that two pdf documents given by their path contain no differences
     * 
     * @param expected the path of the pdf file with the expected content
     * @param actual the path of the pdf file with the actual content
     */
    public static void assertPdfEquals( Path expected, Path actual ) {
        assertPdfEquals( null, expected, actual );
    }

    /**
     * Asserts that two pdf documents given by their path contain no differences an AssertionFailedError is thrown with
     * the given message
     * 
     * @param message the message in case of a fail
     * @param expected the path of the pdf file with the expected content
     * @param actual the path of the pdf file with the actual content
     */
    public static void assertPdfEquals( String message, Path expected, Path actual ) {
        assertPdfEquals( message, expected.toFile(), actual.toFile() );
    }

    /**
     * Asserts that two pdf documents given by their path contain no differences an AssertionFailedError is thrown with
     * the given message
     * 
     * @param expected the URL of the pdf file with the expected content
     * @param actual the URL of the pdf file with the actual content
     */
    public static void assertPdfEquals( URL expected, URL actual ) {
        assertPdfEquals( null, expected, actual );
    }

    /**
     * Asserts that two pdf documents given by their path contain no differences an AssertionFailedError is thrown with
     * the given message
     * 
     * @param message the message in case of a fail
     * @param expected the InputStream of the pdf file with the expected content
     * @param actual the InputStream of the pdf file with the actual content
     */
    public static void assertPdfEquals( String message, InputStream expected, InputStream actual ) {
        // find the name of the caller to make a specific name for the ErrorHandler
        StackTraceElement[] stack = new Exception().getStackTrace();
        String baseName = "";
        for( StackTraceElement el : stack ) {
            String className = el.getClassName();
            if( !PdfcAssert.class.getName().equals( className ) ) {
                if( className.contains( "." ) ) {
                    String shortName = className.substring( className.lastIndexOf( '.' ) );
                    if( shortName.length() > 3 ) { // length > 3 so we assume it's not obfuscated                       
                        baseName = shortName + '-';
                        break;
                    }
                }
                baseName = className + '-';
                break;
            }
        }
        testEquality( message, new PdfData( baseName + "expected", expected ), new PdfData( baseName + "actual", actual ), true );
    }

    /**
     * Asserts that two pdf documents given by their path contain no differences an AssertionFailedError is thrown with
     * the given message
     * 
     * @param expected the InputStream of the pdf file with the expected content
     * @param actual the InputStream of the pdf file with the actual content
     */
    public static void assertPdfEquals( InputStream expected, InputStream actual ) {
        assertPdfEquals( null, expected, actual );
    }

    /**
     * Asserts that two pdf documents given by their path contain no differences an AssertionFailedError is thrown with
     * the given message
     * 
     * @param message the message in case of a fail
     * @param expected the URL of the pdf file with the expected content
     * @param actual the URL of the pdf file with the actual content
     */
    public static void assertPdfEquals( String message, URL expected, URL actual ) {
        testEquality( message, new PdfData( expected ), new PdfData( actual ), true );
    }

    /**
     * Asserts that two pdf documents contain differences
     * 
     * @param expected the pdf file with the expected content
     * @param actual the pdf file with the actual content
     */
    public static void assertPdfNotEquals( File expected, File actual ) {
        assertPdfNotEquals( null, expected, actual );
    }

    /**
     * Asserts that two pdf documents contain differences an AssertionFailedError is thrown with the given message
     * 
     * @param message the message in case of a fail
     * @param expected the pdf file with the expected content
     * @param actual the pdf file with the actual content
     */
    public static void assertPdfNotEquals( String message, File expected, File actual ) {
        testEquality( message, new PdfData( expected ), new PdfData( actual ), false );
    }

    /**
     * Asserts that two pdf documents given by their path contain differences
     * 
     * @param expected the path of the pdf file with the expected content
     * @param actual the path of the pdf file with the actual content
     */
    public static void assertPdfNotEquals( String expected, String actual ) {
        assertPdfNotEquals( new File( expected ), new File( actual ) );
    }

    /**
     * Asserts that two pdf documents given by their path contain differences an AssertionFailedError is thrown with the
     * given message
     * 
     * @param message the message in case of a fail
     * @param expected the path of the pdf file with the expected content
     * @param actual the path of the pdf file with the actual content
     */
    public static void assertPdfNotEquals( String message, String expected, String actual ) {
        assertPdfNotEquals( message, new File( expected ), new File( actual ) );
    }

    /**
     * Asserts that two pdf documents given by their path contain differences
     * 
     * @param expected the {@link Path} of the pdf file with the expected content
     * @param actual the path of the pdf file with the actual content
     */
    public static void assertPdfNotEquals( Path expected, Path actual ) {
        assertPdfNotEquals( expected.toFile(), actual.toFile() );
    }

    /**
     * Asserts that two pdf documents given by their {@link Path} contain differences an {@link AssertionFailedError} is
     * thrown with the given message
     * 
     * @param message the message in case of a fail
     * @param expected the {@link Path} of the pdf file with the expected content
     * @param actual the path of the pdf file with the actual content
     */
    public static void assertPdfNotEquals( String message, Path expected, Path actual ) {
        assertPdfNotEquals( message, expected.toFile(), actual.toFile() );
    }
}
