package de.hirola.sportslibrary.model;

import de.hirola.kintojava.model.Persisted;
import de.hirola.sportslibrary.PersistentObject;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.time.LocalDate;
import java.time.ZoneId;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

/**
 * Copyright 2021 by Michael Schmidt, Hirola Consulting
 * This software us licensed unter the AGPL-3.0 or later.
 *
 * A track for trainings. Tracks can be imported or recorded.
 *
 * @author Michael Schmidt (Hirola)
 * @since 1.1.1
 */
public class Track extends PersistentObject {

    private Track.Id trackId; // a temporary id while recording tracks
    @Persisted
    private String name = ""; // name of track
    @Persisted
    private String description = ""; // short description
    @Persisted
    private LocalDate importDate = LocalDate.now();
    @Persisted
    private long startTime = -1;
    @Persisted
    private long stopTime = -1;
    @Persisted
    private double distance = -1.0;
    @Persisted
    private double averageSpeed = -1.0;
    @Persisted
    private double altitudeDifference = -1;
    @Persisted
    private List<LocationData> locations; // list of tracking data

    /**
     * Default constructor for reflection.
     */
    public Track() {
        locations = new ArrayList<>();
    }

    /**
     * Create a track to start recording.
     *
     * @param name of track
     * @param description of track
     * @param startTime of track
     */
    public Track(String name, String description, long startTime) {
        this.name = name;
        this.description = description;
        this.startTime = startTime;
        locations = new ArrayList<>();
    }

    /**
     * Create a recorded and completed track. The location of the tack can be empty.
     *
     * @param name of track
     * @param description of track
     * @param startTime of track
     * @param stopTime of track
     * @param locations of track, can be null
     */
    public Track(String name, String description, long startTime, long stopTime, @Nullable List<LocationData> locations) {
        this.name = name;
        this.description = description;
        this.startTime = startTime;
        this.stopTime = stopTime;
        this.locations = Objects.requireNonNullElseGet(locations, ArrayList::new);
        //TODO: start, stop, avg, duration if locations not null
    }

    /**
     * Create an imported track. The start and end time is determined from the locations.
     * If no import date given, the current date will be set.
     *
     * @param name of track
     * @param description of track
     * @param importDate of track
     * @param locations of track
     */
    public Track(@NotNull String name, @Nullable String description, @Nullable LocalDate importDate,
                 @NotNull List<LocationData> locations) {
        this.name = name;
        this.description = Objects.requireNonNullElse(description, "");
        this.importDate = Objects.requireNonNullElseGet(importDate, () -> LocalDate.now(ZoneId.systemDefault()));
        this.locations = locations;
        //TODO: start and stop time, avg, speed
    }

    /**
     * Create an imported track.
     * If no import date given, the current date will be set.
     *
     * @param name of track
     * @param description of track
     * @param importDate of track
     * @param stopTime of track
     * @param avg of track
     * @param distance of track
     * @param locations of track
     */
    public Track(@NotNull String name, @Nullable String description, @Nullable LocalDate importDate,
                 long startTime, long stopTime, double avg, double distance, @NotNull List<LocationData> locations) {
        this.name = name;
        this.description = Objects.requireNonNullElse(description, "");
        this.importDate = Objects.requireNonNullElseGet(importDate, () -> LocalDate.now(ZoneId.systemDefault()));
        this.startTime = startTime;
        this.stopTime = stopTime;
        this.averageSpeed = avg;
        this.distance = distance;
        this.locations = locations;
    }

    /**
     * Get the temporary id of the track.
     * This id is using while recording and temporary saving in local datastore.
     *
     * @return The id for the track while recording and temporary saving.
     */
    @Nullable
    public Track.Id getTrackId() {
        return trackId;
    }

    /**
     * Set the temporary track id for recording a track.
     * The id will not be saved permanently.
     *
     * @param trackId The temporary id
     */
    public void setTrackId(Id trackId) {
        this.trackId = trackId;
    }

    /**
     * Get the name the track.
     *
     * @return The name of the track
     */
    public String getName() {
        return name;
    }

    /**
     * Set the name of the track.
     *
     * @param name of the track.
     */
    public void setName(String name) {
        this.name = name;
    }

    /**
     * Get the name the track.
     *
     * @return The name of the track
     */
    public String getDescription() {
        return description;
    }

    /**
     * Set the description of the track.
     *
     * @param description of the track.
     */
    public void setDescription(String description) {
        this.description = description;
    }

    /**
     * Get the start time the track in milliseconds to UTC Time
     * or -1 if no time is set.
     *
     * @return The start time of the track in milliseconds to UTC Time
     */
    public long getStartTime() {
        //TODO: if start time equal -1, get the time form last location
        return startTime;
    }

    /**
     * Set the start time of the track in milliseconds to UTC Time.
     *
     * @param startTime of the track
     */
    public void setStartTime(long startTime) {
        this.startTime = startTime;
    }

    /**
     * Get the stop time the track in milliseconds to UTC Time
     * or -1 if no time is set.
     *
     * @return The stop time of the track
     */
    public long getStopTime() {
        //TODO: if stop time equal -1, get the time form last location
        return stopTime;
    }

    /**
     * Set the stop time of the track in milliseconds to UTC Time.
     *
     * @param stopTime of the track.
     */
    public void setStopTime(long stopTime) {
        this.stopTime = stopTime;
    }

    /**
     * Get the distance of the track in meters.
     * If not set (-1), the duration will calculate from the locations.
     *
     * @return The distance of the track in meters
     */
    public double getDistance() {
        //TODO: if -1, calculate from locations
        return distance;
    }

    /**
     * Set the distance of the track in meters.
     *
     * @param distance of the track in meters
     */
    public void setDistance(double distance) {
        this.distance = distance;
    }

    /**
     * Get the average speed of the track in km/h.
     * If not set (-1), the speed will calculate from
     * the individual speeds of the locations.
     *
     * @return The average speed of the track in km/h
     */
    public double getAverageSpeed() {
        //TODO: if 0, calculate from locations
        return averageSpeed;
    }

    /**
     * Set the average speed of the track in km/h.
     *
     * @param speed of the track in km/h
     */
    public void setAverageSpeed(double speed) {
        this.averageSpeed = speed;
    }

    /**
     * Get the altitudeDifference of the track in meter.
     * If not set (-1), the altitude difference will calculate
     * from the individual altitude difference of the locations.
     *
     * @return The average speed of the track in km/h
     */
    public double getAltitudeDifference() {
        //TODO: if -1, calculate from locations
        return altitudeDifference;
    }

    /**
     * Set the altitudeDifference of the track in meter.
     *
     * @param altitudeDifference of the track in meter
     */
    public void setAltitudeDifference(double altitudeDifference) {
        this.altitudeDifference = altitudeDifference;
    }

    /**
     * Get the import date of the track.
     *
     * @return The import date of the track
     */
    public LocalDate getImportDate() {
        return importDate;
    }

    /**
     * Get the locations of the track.
     *
     * @return The locations of the track
     */
    public List<LocationData> getLocations() {
        if (locations == null) {
            locations = new ArrayList<>();
        }
        return locations;
    }

    /**
     * The temporary id for the track.
     */
    public static class Id {

        private final long id;

        /**
         * Create a new id object with a given number.
         *
         * @param id The number for the id.
         */
        public Id(long id) {
            this.id = id;
        }

        /**
         * Get the id.
         *
         * @return The id
         */
        public long getId() {
            return id;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) return true;
            if (o == null || getClass() != o.getClass()) return false;
            Id id1 = (Id) o;
            return id == id1.id;
        }

        @Override
        public int hashCode() {
            return Objects.hash(id);
        }

        @Override
        public String toString() {
            return String.valueOf(id);
        }
    }

    @Override
    public boolean equals(Object o) {
        // gleicher Name = gleiches Objekt
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        if (!super.equals(o)) return false;
        Track track = (Track) o;
        return name.equals(track.name)
                && description.equals(track.description)
                && startTime == track.startTime
                && stopTime == track.stopTime;
    }

    @Override
    public int hashCode() {
        return Objects.hash(super.hashCode(), name, description, startTime, stopTime);
    }

    @Override
    public List<String> getIdentityAttributeNames() {
        List<String> identityAttributeNames = new ArrayList<>();
        identityAttributeNames.add("name");
        return identityAttributeNames;
    }
}
