package de.hirola.sportslibrary.model;

import de.hirola.kintojava.model.Persisted;
import de.hirola.sportslibrary.PersistentObject;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

/**
 * Copyright 2021 by Michael Schmidt, Hirola Consulting
 * This software us licensed under the AGPL-3.0 or later.
 *
 * Object for managing movement types. The user can make his own settings.
 * When you start the app for the first time, some movement types are imported
 * into the data store using JSON.
 *
 * @author Michael Schmidt (Hirola)
 * @since 1.1.1
 *
 */
public class MovementType extends PersistentObject {

    @Persisted
    private final String key;  // token for the type of moving, e.g. L is running
    @Persisted
    private final String stringForKey; // the text for the token
    private String colorKeyString; // the color for the token, dynamically on different platforms
    @Persisted
    private double speed; // speed of movement type in km/h
    @Persisted
    private double pace; // pace of movement type, user defined in relation to the speed

    /**
     * Default constructor for reflection.
     */
    public MovementType() {
        super();
        key = "";
        stringForKey = "";
        colorKeyString = "green";
        speed = 0.0;
        pace = 0.0;
    }

    /**
     * Creates a movement type object.
     *
     * @param key of the type
     * @param stringForKey token for the type
     * @param colorKeyString color for the token
     * @param speed for the type
     * @param pace for the type
     */
    public MovementType(String key, String stringForKey, String colorKeyString, double speed, double pace) {
        super();
        this.key = key;
        this.stringForKey = stringForKey;
        this.colorKeyString = colorKeyString;
        this.speed = speed;
        this.pace = pace;
    }

    /**
     * Get the token for the movement type.
     *
     * @return Token for the movement type
     */
    public String getKey() {
        return key;
    }

    /**
     * Get the token for the movement type.
     *
     * @return Token for the movement type
     */
    public String getStringForKey() {
        return stringForKey;
    }


    /**
     * Get the color string for the token of the movement type.
     *
     * @return Color string for the token
     */
    public String getColorKeyString() {
        return colorKeyString;
    }

    /**
     * Set the color for the token.
     *
     * @param colorKeyString of token
     */
    public void setColorKeyString(String colorKeyString) {
        this.colorKeyString = colorKeyString;
    }

    /**
     * Get the speed for the movement type.
     *
     * @return Token for the movement type
     */
    public double getSpeed() {
        return speed;
    }

    /**
     * Set the speed for the movement type.
     *
     * @param speed of movement type
     */
    public void setSpeed(double speed) {
        this.speed = speed;
    }

    /**
     * Get the pace for the movement type.
     *
     * @return Pace for the movement type
     */
    public double getPace() {
        return pace;
    }

    /**
     * Set the pace for the speed of the movement type.
     *
     * @param pace of movement type
     */
    public void setPace(double pace) {
        this.pace = pace;
    }

    @Override
    public boolean equals(Object o) {
        // gleicher Schlüssel = gleiches Objekt
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        if (!super.equals(o)) return false;
        MovementType that = (MovementType) o;
        return key.equals(that.key);
    }

    @Override
    public int hashCode() {
        return Objects.hash(super.hashCode(), key);
    }

    @Override
    public List<String> getIdentityAttributeNames() {
        List<String> identityAttributeNames = new ArrayList<>();
        identityAttributeNames.add("key");
        return identityAttributeNames;
    }
}
