package de.hirola.sportslibrary.model;

import de.hirola.kintojava.model.Persisted;
import de.hirola.sportslibrary.PersistentObject;

import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

/**
 * Copyright 2021 by Michael Schmidt, Hirola Consulting
 * This software us licensed under the AGPL-3.0 or later.
 *
 * An object to store location data.
 *
 * @author Michael Schmidt (Hirola)
 * @since 1.1.1
 */
public class LocationData extends PersistentObject {

    @Persisted
    private final long timeStamp; // UTC time of this location, in milliseconds since epoch (January 1, 1970).
    @Persisted
    private final String provider;
    @Persisted
    private final double latitude;
    @Persisted
    private final double longitude;
    @Persisted
    private final double altitude;
    @Persisted
    private final double speed;

    /**
     * Default constructor for reflection.
     */
    public LocationData() {
        timeStamp = LocalDateTime.now().toInstant(ZoneOffset.UTC).toEpochMilli();
        provider = "https://www.countrycoordinate.com"; // Neustadt in Sachsen
        latitude = 51.023639;
        longitude = 14.213444;
        altitude = 0.0;
        speed = 0.0;
    }

    /**
     * Creates a location object.
     *
     * @param latitude coordinate for the location
     * @param longitude coordinate for the location
     */
    public LocationData(double latitude, double longitude) {
        timeStamp = LocalDateTime.now().toInstant(ZoneOffset.UTC).toEpochMilli();
        provider = "Unknown";
        this.latitude = latitude;
        this.longitude = longitude;
        this.altitude = 0.0;
        this.speed = 0.0;
    }

    /**
     * Creates a location object.
     *
     * @param timeStamp of the location
     * @param provider source of the location
     * @param latitude coordinate for the location
     * @param longitude coordinate for the location
     * @param altitude of the location
     * @param speed of the location
     */
    public LocationData(long timeStamp, String provider, double latitude, double longitude, double altitude, double speed) {
        this.timeStamp = timeStamp;
        this.provider = provider;
        this.latitude = latitude;
        this.longitude = longitude;
        this.altitude = altitude;
        this.speed = speed;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        if (!super.equals(o)) return false;
        LocationData that = (LocationData) o;
        return that.timeStamp == timeStamp
                && that.provider.equals(provider)
                && Double.compare(that.latitude, latitude) == 0
                && Double.compare(that.longitude, longitude) == 0
                && Double.compare(that.altitude, altitude) == 0
                && Double.compare(that.speed, speed) == 0;
    }

    @Override
    public int hashCode() {
        return Objects.hash(super.hashCode(), timeStamp, provider, latitude, longitude, altitude, speed);
    }

    @Override
    public List<String> getIdentityAttributeNames() {
        List<String> identityAttributeNames = new ArrayList<>();
        identityAttributeNames.add("latitude");
        identityAttributeNames.add("longitude");
        identityAttributeNames.add("altitude");
        identityAttributeNames.add("speed");
        identityAttributeNames.add("timeStamp");
        return identityAttributeNames;
    }
}