package de.hirola.sportslibrary;

import de.hirola.sportslibrary.model.*;

import de.hirola.kintojava.Kinto;
import de.hirola.kintojava.KintoConfiguration;
import de.hirola.kintojava.KintoException;
import de.hirola.kintojava.model.KintoObject;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.time.LocalDate;
import java.util.ArrayList;
import java.util.List;

/**
 * Copyright 2021 by Michael Schmidt, Hirola Consulting
 * This software us licensed under the AGPL-3.0 or later.
 *
 * Adds a persistence layer, encapsulating the actual data storage technology used.
 *
 * @author Michael Schmidt (Hirola)
 * @since 1.1.1
 */
public final class DataRepository {

    private final Kinto kinto; // we use the kinto data layer
    private final DatastoreDelegate delegate; // the delegate for data store events

    /**
     * Create the local datastore access layer.
     *
     * @param packageName of the app using this library
     * @param delegate to inform about datastore events, can be null
     * @throws SportsLibraryException if the datalayer could not initialize
     */
    public DataRepository(@NotNull String packageName, @Nullable DatastoreDelegate delegate) throws SportsLibraryException {
        // set the default database name
        this.delegate = delegate;
        // add all types for managing by kinto java
        ArrayList<Class<? extends KintoObject>> typeList = new ArrayList<>();
        typeList.add(User.class);
        typeList.add(LocationData.class);
        typeList.add(Track.class);
        typeList.add(MovementType.class);
        typeList.add(TrainingType.class);
        typeList.add(Training.class);
        typeList.add(RunningUnit.class);
        typeList.add(RunningPlanEntry.class);
        typeList.add(RunningPlan.class);

        // create a kinto java configuration
        try {
            KintoConfiguration configuration = new KintoConfiguration.Builder(packageName)
                    .objectTypes(typeList)
                    .build();
            // create the kinto java instance
            kinto = new Kinto(configuration);
        } catch (KintoException exception) {
            throw new SportsLibraryException(exception.getMessage());
        }
    }

    /**
     * Get a flag to determine if the datastore is empty.
     * Some templates are required, which must be imported at the first start.
     *
     * @return A flag to determine if the datastore is empty
     * @throws SportsLibraryException if an error occurred
     */
    public boolean isEmpty() throws SportsLibraryException {
        // the datastore is "empty" if there are no movement and training types or running plans (yet)
        int emptyValueCounter = 0;
        try {
            // keine Bewegungsarten
            List<? extends KintoObject> movementTypes =  kinto.findAll(MovementType.class);
            if (movementTypes.isEmpty()) {
                emptyValueCounter++;
            }
            // keine Trainingsarten
            List<? extends KintoObject> trainingTypes =  kinto.findAll(TrainingType.class);
            if (trainingTypes.isEmpty()) {
                emptyValueCounter++;
            }
            // keine Laufpläne
            List<? extends KintoObject> runningPlans =  kinto.findAll(RunningPlan.class);
            if (runningPlans.isEmpty()) {
                emptyValueCounter++;
            }
        } catch (KintoException exception) {
            throw new SportsLibraryException("The content of the local datastore couldn't determined : "
            + exception.getMessage());
        }
        return emptyValueCounter == 3;
    }

    /**
     * Get a flag to determine if the datastore is open.
     *
     * @return A flag to determine if the datastore is open.
     */
    public boolean isOpen() {
        return kinto.isOpen();
    }

    /**
     * Adds a new object to the local datastore.
     *
     * @param object to be added
     * @throws SportsLibraryException if an error occurred while adding
     */
    public void add(PersistentObject object) throws SportsLibraryException {
        try {
            kinto.add(object);
            if (delegate != null) {
                delegate.didObjectAdded(object);
            }
        } catch (KintoException exception) {
            throw new SportsLibraryException(exception);
        }
    }

    /**
     * Update an existing object on the local datastore.
     *
     * @param object to be updated
     * @throws SportsLibraryException if an error occurred while updating
     */
    public void update(PersistentObject object) throws SportsLibraryException{
        try {
            kinto.update(object);
            if (delegate != null) {
                delegate.didObjectUpdated(object);
            }
        } catch (KintoException exception) {
            throw new SportsLibraryException(exception);
        }
    }

    /**
     * Removes an existing object from the local datastore.
     *
     * @param object to be removed
     * @throws SportsLibraryException if an error occurred while removing
     */
    public void remove(PersistentObject object) throws SportsLibraryException {
        try {
            kinto.remove(object);
            if (delegate != null) {
                delegate.didObjectRemoved(object);
            }
        } catch (KintoException exception) {
            throw new SportsLibraryException(exception);
        }
    }

    /**
     * Get all objects with a given type.
     *
     * @param fromType of object to get
     * @return A list of objects with the given type.
     * @throws SportsLibraryException if an error occurred while getting
     */
    public List<? extends PersistentObject> findAll(Class<? extends PersistentObject> fromType) throws SportsLibraryException {
        List<? extends PersistentObject> persistentObjects =  new ArrayList<>();
        try {
            List<? extends KintoObject> kintoObjects =  kinto.findAll(fromType);
            if (!kintoObjects.isEmpty()) {
                // check if all objects extends PersistentObject
                if (!kintoObjects.stream().allMatch( p -> p instanceof PersistentObject)) {
                    throw new SportsLibraryException("All objects from result must be extends PersistentObject.");
                }
                // all objects extend PersistentObject!
                // noinspection unchecked
                persistentObjects = (List<? extends PersistentObject>) kintoObjects;
                // check and correct the start date if running plan not active
                for (PersistentObject persistentObject : persistentObjects) {
                    if (persistentObject instanceof RunningPlan) {
                        RunningPlan runningPlan = (RunningPlan) persistentObject;
                        if (!runningPlan.isCompleted() || !runningPlan.isActive()) {
                            LocalDate startDate = runningPlan.getStartDate();
                            LocalDate today = LocalDate.now();
                            if (startDate.isBefore(today) || startDate.isEqual(today)) {
                                // the method adjust the start day automatically
                                runningPlan.setStartDate(today);
                                try {
                                    // save the corrected start date to local data store
                                    kinto.update(runningPlan);
                                } catch (KintoException exception) {
                                    if (Global.APP_DEBUG_MODE) {
                                        exception.printStackTrace();
                                    }
                                }
                            }
                        }
                    }
                }
            }
            return persistentObjects;
        } catch (KintoException exception) {
            throw new SportsLibraryException(exception);
        }
    }
}
