/*
 * Copyright 2016 Heiko Seeberger
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package de.heikoseeberger.commons.akka.stream

import akka.stream.{ Attributes, FlowShape, Inlet, Outlet }
import akka.stream.stage.{ GraphStage, GraphStageLogic, InHandler, OutHandler }

/**
 * This companion defines a factory for [[ScanAhead]] instances, see [[ScanAhead.apply]].
 */
object ScanAhead {
  def apply[A, B](zero: B)(f: (B, A) => B): ScanAhead[A, B] = new ScanAhead(zero)(f)
}

/**
 * This stage is like `scan`, but the first element emitted is the result of applying the given function to the given
 * zero value and the first pushed element.
 *
 * @param zero zero value for folding
 * @param f binary operation for folding
 * @tparam A input type
 * @tparam B output type
 */
final class ScanAhead[A, B] private (zero: B)(f: (B, A) => B) extends GraphStage[FlowShape[A, B]] {

  override val shape = FlowShape(Inlet[A]("scanAhead.in"), Outlet[B]("scanAhead.out"))

  override def createLogic(attributes: Attributes) = new GraphStageLogic(shape) {
    import shape._

    private var acc = zero

    setHandler(in, new InHandler {
      override def onPush() = {
        acc = f(acc, grab(in))
        push(out, acc)
      }
    })

    setHandler(out, new OutHandler {
      override def onPull() = pull(in)
    })
  }
}
