/*
 * Copyright 2016 Heiko Seeberger
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package de.heikoseeberger.commons.akka.stream

import akka.stream.stage.{ GraphStage, GraphStageLogic, InHandler, OutHandler }
import akka.stream.{ Attributes, FlowShape, Inlet, Outlet }

/**
 * This companion defines a factory for [[LastElement]] instances, see [[LastElement.apply]].
 */
object LastElement {

  /**
   * Creates [[LastElement]] instances.
   *
   * @tparam A element type
   * @return new [[LastElement]] instance
   */
  def apply[A](): LastElement[A] = new LastElement[A]
}

/**
 * This stage emits the last element, if any, pushed from upstream before normal or abnormal completion and completes
 * normally.
 *
 * @tparam A element type
 */
final class LastElement[A] private extends GraphStage[FlowShape[A, Option[A]]] {

  override val shape = FlowShape(Inlet[A]("lastElement.in"), Outlet[Option[A]]("lastElement.out"))

  override def createLogic(attributes: Attributes) = new GraphStageLogic(shape) {
    import shape._

    private var last = Option.empty[A]

    setHandler(in, new InHandler {

      override def onPush() = {
        last = Some(grab(in))
        pull(in)
      }

      override def onUpstreamFinish() = {
        if (isAvailable(out)) push(out, last)
        super.onUpstreamFinish()
      }

      override def onUpstreamFailure(t: Throwable) = {
        if (isAvailable(out)) push(out, last)
        super.onUpstreamFinish()
      }
    })

    setHandler(out, new OutHandler {
      override def onPull() = pull(in)
    })
  }
}
