/*
 * Copyright 2015 Heiko Seeberger
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package de.heikoseeberger.akkamacrologging

import akka.actor.{ Actor, ActorLogging => AkkaActorLogging }
import akka.event.{ Logging, LoggingAdapter => AkkaLoggingAdapter }

trait ActorLogging extends AkkaActorLogging { this: Actor =>

  private var _log: LoggingAdapter = _

  override def log: LoggingAdapter = {
    if (_log eq null) _log = LoggingAdapter(Logging(context.system, this))
    _log
  }
}

object LoggingAdapter {
  def apply(underlying: AkkaLoggingAdapter) = new LoggingAdapter(underlying)
}

final class LoggingAdapter private (val underlying: AkkaLoggingAdapter) extends AkkaLoggingAdapter {
  import LoggingAdapterMacro._

  override def error(cause: Throwable, message: String): Unit = macro errorC0

  override def error(cause: Throwable, template: String, arg1: Any): Unit = macro errorC1

  override def error(cause: Throwable, template: String, arg1: Any, arg2: Any): Unit = macro errorC2

  override def error(cause: Throwable, template: String, arg1: Any, arg2: Any, arg3: Any): Unit = macro errorC3

  override def error(cause: Throwable, template: String, arg1: Any, arg2: Any, arg3: Any, arg4: Any): Unit = macro errorC4

  override def error(message: String): Unit = macro error0

  override def error(template: String, arg1: Any): Unit = macro error1

  override def error(template: String, arg1: Any, arg2: Any): Unit = macro error2

  override def error(template: String, arg1: Any, arg2: Any, arg3: Any): Unit = macro error3

  override def error(template: String, arg1: Any, arg2: Any, arg3: Any, arg4: Any): Unit = macro error4

  override def warning(message: String): Unit = macro warning0

  override def warning(template: String, arg1: Any): Unit = macro warning1

  override def warning(template: String, arg1: Any, arg2: Any): Unit = macro warning2

  override def warning(template: String, arg1: Any, arg2: Any, arg3: Any): Unit = macro warning3

  override def warning(template: String, arg1: Any, arg2: Any, arg3: Any, arg4: Any): Unit = macro warning4

  override def info(message: String): Unit = macro info0

  override def info(template: String, arg1: Any): Unit = macro info1

  override def info(template: String, arg1: Any, arg2: Any): Unit = macro info2

  override def info(template: String, arg1: Any, arg2: Any, arg3: Any): Unit = macro info3

  override def info(template: String, arg1: Any, arg2: Any, arg3: Any, arg4: Any): Unit = macro info4

  override def debug(message: String): Unit = macro debug0

  override def debug(template: String, arg1: Any): Unit = macro debug1

  override def debug(template: String, arg1: Any, arg2: Any): Unit = macro debug2

  override def debug(template: String, arg1: Any, arg2: Any, arg3: Any): Unit = macro debug3

  override def debug(template: String, arg1: Any, arg2: Any, arg3: Any, arg4: Any): Unit = macro debug4

  override def isErrorEnabled = underlying.isErrorEnabled

  override def isWarningEnabled = underlying.isWarningEnabled

  override def isInfoEnabled = underlying.isInfoEnabled

  override def isDebugEnabled = underlying.isDebugEnabled

  override protected def notifyError(cause: Throwable, message: String) = underlying.error(cause, message)

  override protected def notifyError(message: String) = underlying.error(message)

  override protected def notifyWarning(message: String) = underlying.warning(message)

  override protected def notifyInfo(message: String) = underlying.info(message)

  override protected def notifyDebug(message: String) = underlying.debug(message)
}
