/*
 * Decompiled with CFR 0.152.
 */
package de.hdi.mongobumblebee.dao;

import com.mongodb.client.MongoClient;
import com.mongodb.client.MongoCollection;
import com.mongodb.client.MongoDatabase;
import com.mongodb.client.model.Filters;
import de.hdi.mongobumblebee.changeset.ChangeEntry;
import de.hdi.mongobumblebee.dao.ChangeEntryIndexDao;
import de.hdi.mongobumblebee.dao.LockDao;
import de.hdi.mongobumblebee.exception.MongoBumblebeeConfigurationException;
import de.hdi.mongobumblebee.exception.MongoBumblebeeConnectionException;
import de.hdi.mongobumblebee.exception.MongoBumblebeeLockException;
import java.time.LocalDateTime;
import java.util.Date;
import org.bson.Document;
import org.bson.conversions.Bson;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.StringUtils;

public class ChangeEntryDao {
    private static final Logger log = LoggerFactory.getLogger(ChangeEntryDao.class);
    private MongoDatabase mongoDatabase;
    private MongoClient mongoClient;
    private ChangeEntryIndexDao indexDao = new ChangeEntryIndexDao();
    private String changelogCollectionName;
    private boolean waitForLock;
    private long changeLogLockWaitTime;
    private long changeLogLockPollRate;
    private boolean throwExceptionIfCannotObtainLock;
    private LockDao lockDao;

    public ChangeEntryDao(String changelogCollectionName, String lockCollectionName, boolean waitForLock, long changeLogLockWaitTime, long changeLogLockPollRate, boolean throwExceptionIfCannotObtainLock) {
        this.lockDao = new LockDao(lockCollectionName);
        this.changelogCollectionName = changelogCollectionName;
        this.waitForLock = waitForLock;
        this.changeLogLockWaitTime = changeLogLockWaitTime;
        this.changeLogLockPollRate = changeLogLockPollRate;
        this.throwExceptionIfCannotObtainLock = throwExceptionIfCannotObtainLock;
    }

    public MongoDatabase connectMongoDb(MongoClient mongoClient, String dbName) throws MongoBumblebeeConfigurationException {
        if (!StringUtils.hasText((String)dbName)) {
            throw new MongoBumblebeeConfigurationException("DB name is not set. Should be defined in MongoDB URI or via setter");
        }
        this.mongoClient = mongoClient;
        this.mongoDatabase = mongoClient.getDatabase(dbName);
        this.ensureChangeLogCollectionIndex((MongoCollection<Document>)this.mongoDatabase.getCollection(this.changelogCollectionName));
        this.initializeLock();
        return this.mongoDatabase;
    }

    public boolean acquireProcessLock() throws MongoBumblebeeConnectionException, MongoBumblebeeLockException {
        this.verifyDbConnection();
        boolean acquired = this.lockDao.acquireLock(this.getMongoDatabase());
        if (!acquired && this.waitForLock) {
            long timeToGiveUp = new Date().getTime() + this.changeLogLockWaitTime * 1000L * 60L;
            while (!acquired && new Date().getTime() < timeToGiveUp) {
                acquired = this.lockDao.acquireLock(this.getMongoDatabase());
                if (acquired) continue;
                log.info("Waiting for process lock....");
                try {
                    Thread.sleep(this.changeLogLockPollRate * 1000L);
                }
                catch (InterruptedException interruptedException) {}
            }
        }
        LocalDateTime lastAccess = this.lockDao.getLastAccess(this.getMongoDatabase());
        if (!acquired && lastAccess != null && lastAccess.plusMinutes(this.changeLogLockWaitTime).isBefore(LocalDateTime.now())) {
            log.info("Process lock released because it wasn't updated for " + this.changeLogLockWaitTime + " minutes.");
            this.lockDao.releaseLock(this.getMongoDatabase());
            acquired = this.lockDao.acquireLock(this.getMongoDatabase());
        }
        if (!acquired && this.throwExceptionIfCannotObtainLock) {
            log.info("MongoBumblebee did not acquire process lock. Throwing exception.");
            throw new MongoBumblebeeLockException("Could not acquire process lock");
        }
        return acquired;
    }

    public void updateLock() {
        this.lockDao.updateLock(this.getMongoDatabase());
    }

    public void releaseProcessLock() throws MongoBumblebeeConnectionException {
        this.verifyDbConnection();
        this.lockDao.releaseLock(this.getMongoDatabase());
    }

    public boolean isProccessLockHeld() throws MongoBumblebeeConnectionException {
        this.verifyDbConnection();
        return this.lockDao.isLockHeld(this.getMongoDatabase());
    }

    public boolean isNewChange(ChangeEntry changeEntry) throws MongoBumblebeeConnectionException {
        this.verifyDbConnection();
        MongoCollection changeLogCollection = this.getMongoDatabase().getCollection(this.changelogCollectionName);
        Document entry = (Document)changeLogCollection.find((Bson)changeEntry.buildSearchQueryDBObject()).first();
        return entry == null;
    }

    public void save(ChangeEntry changeEntry) throws MongoBumblebeeConnectionException {
        this.verifyDbConnection();
        MongoCollection changeLogCollection = this.getMongoDatabase().getCollection(this.changelogCollectionName);
        if (this.isNewChange(changeEntry)) {
            changeLogCollection.insertOne((Object)changeEntry.buildFullDBObject());
        } else {
            Bson filter = Filters.and((Bson[])new Bson[]{Filters.eq((String)"changeId", (Object)changeEntry.getChangeId()), Filters.eq((String)"author", (Object)changeEntry.getAuthor())});
            Document document = changeEntry.buildFullDBObject();
            changeLogCollection.findOneAndReplace(filter, (Object)document);
        }
    }

    private void verifyDbConnection() throws MongoBumblebeeConnectionException {
        if (this.getMongoDatabase() == null) {
            throw new MongoBumblebeeConnectionException("Database is not connected. MongoBumblebee has thrown an unexpected error", new NullPointerException());
        }
    }

    private void ensureChangeLogCollectionIndex(MongoCollection<Document> collection) {
        Document index = this.indexDao.findRequiredChangeAndAuthorIndex(collection);
        if (index == null) {
            this.indexDao.createRequiredUniqueIndex(collection);
            log.debug("Index in collection " + this.changelogCollectionName + " was created");
        } else if (!this.indexDao.isUnique(index)) {
            this.indexDao.dropIndex(collection, index);
            this.indexDao.createRequiredUniqueIndex(collection);
            log.debug("Index in collection " + this.changelogCollectionName + " was recreated");
        }
    }

    public void close() {
        this.mongoClient.close();
    }

    private void initializeLock() {
        this.lockDao.intitializeLock(this.mongoDatabase);
    }

    public void setIndexDao(ChangeEntryIndexDao changeEntryIndexDao) {
        this.indexDao = changeEntryIndexDao;
    }

    void setLockDao(LockDao lockDao) {
        this.lockDao = lockDao;
    }

    public void setChangelogCollectionName(String changelogCollectionName) {
        this.changelogCollectionName = changelogCollectionName;
    }

    public void setLockCollectionName(String lockCollectionName) {
        this.lockDao.setLockCollectionName(lockCollectionName);
    }

    public boolean isThrowExceptionIfCannotObtainLock() {
        return this.throwExceptionIfCannotObtainLock;
    }

    public void setThrowExceptionIfCannotObtainLock(boolean throwExceptionIfCannotObtainLock) {
        this.throwExceptionIfCannotObtainLock = throwExceptionIfCannotObtainLock;
    }

    public MongoDatabase getMongoDatabase() {
        return this.mongoDatabase;
    }

    public boolean isWaitForLock() {
        return this.waitForLock;
    }

    public void setWaitForLock(boolean waitForLock) {
        this.waitForLock = waitForLock;
    }

    public long getChangeLogLockWaitTime() {
        return this.changeLogLockWaitTime;
    }

    public void setChangeLogLockWaitTime(long changeLogLockWaitTime) {
        this.changeLogLockWaitTime = changeLogLockWaitTime;
    }

    public long getChangeLogLockPollRate() {
        return this.changeLogLockPollRate;
    }

    public void setChangeLogLockPollRate(long changeLogLockPollRate) {
        this.changeLogLockPollRate = changeLogLockPollRate;
    }
}

