package de.gwdg.metadataqa.marc.utils;

/**
 * This class heavily based on the class published in
 * Jamie Carlstone (2017) Scoring the Quality of E-Serials MARC Records Using Java,
 * Serials Review, 43:3-4, pp. 271-277, DOI: 10.1080/00987913.2017.1350525
 * https://www.tandfonline.com/doi/full/10.1080/00987913.2017.1350525
 */
public class SerialOriginal {
  private String issn022;
  private String frequency310;
  private String datesOfPublication362;
  private String sourceOfDescription588;
  private String oclcNumber001;
  private String fixedFields008;
  private String authenticationCode042;
  private String encodingLevelLDR;
  private String fixedLengthData006;
  private String title245;
  private String publication260;
  private String publication264;
  private String rdaContentType336;
  private String subject650;
  private String notes936;

  public SerialOriginal() {
    super();
  }

  public SerialOriginal(String issn022, String frequency310, String datesOfPublication362, String sourceOfDescription588,
                        String oclcNumber001, String fixedFields008, String authenticationCode042, String encodingLevelLDR,
                        String fixedLengthData006, String title245, String publication260, String publication264,
                        String rdaContentType336, String subject650, String notes936) {
    this.oclcNumber001 = oclcNumber001;
    this.fixedFields008 = fixedFields008;
    this.authenticationCode042 = authenticationCode042;
    this.encodingLevelLDR = encodingLevelLDR;
    this.fixedLengthData006 = fixedLengthData006;
    this.title245 = title245;
    this.publication260 = publication260;
    this.publication264 = publication264;
    this.rdaContentType336 = rdaContentType336;
    this.subject650 = subject650;
    this.notes936 = notes936;
    setIssn022(issn022);
    setFrequency310(frequency310);
    setDatesOfPublication362(datesOfPublication362);
    setSourceOfDescription588(sourceOfDescription588);
  }

  @Override
  public int hashCode() {
    final int prime = 31;
    int result = super.hashCode();
    result = prime * result + ((datesOfPublication362 == null) ? 0 :
      datesOfPublication362.hashCode());
    result = prime * result + ((frequency310 == null) ? 0 :
      frequency310.hashCode());
    result = prime * result + ((issn022 == null) ? 0 :
      issn022.hashCode());
    result = prime * result + ((sourceOfDescription588 == null) ? 0 :
      sourceOfDescription588.hashCode());
    return result;
  }
  @Override
  public boolean equals(Object obj) {
    if (this == obj)
      return true;
    if (!super.equals(obj))
      return false;
    if (getClass() != obj.getClass())
      return false;
    SerialOriginal other = (SerialOriginal) obj;
    if (datesOfPublication362 == null) {
      if (other.datesOfPublication362 != null)
        return false;
    } else if (!datesOfPublication362.equals(other.datesOfPublication362))
      return false;
    if (frequency310 == null) {
      if (other.frequency310 != null)
        return false;
    } else if (!frequency310.equals(other.frequency310))
      return false;
    if (issn022 == null) {
      if (other.issn022 != null)
        return false;
    } else if (!issn022.equals(other.issn022))
      return false;

    if (sourceOfDescription588 == null) {
      if (other.sourceOfDescription588 != null)
        return false;
    } else if (!sourceOfDescription588.equals(other.sourceOfDescription588))
      return false;
    return true;
  }

  @Override
  public String toString() {
    return "Serial [issn022=" + issn022
      + ", frequency310=" + frequency310
      + ", datesOfPublication362=" + datesOfPublication362
      + ", sourceOfDescription588=" + sourceOfDescription588
      + ", toString()=" + super.toString() + "]";
  }

  public String getIssn022() {
    return issn022;
  }

  public void setIssn022(String issn022) {
    this.issn022 = issn022;
  }

  public String getFrequency310() {
    return frequency310;
  }

  public void setFrequency310(String frequency310) {
    this.frequency310 = frequency310;
  }

  public String getDatesOfPublication362() {
    return datesOfPublication362;
  }

  public void setDatesOfPublication362(String datesOfPublication362) {
    this.datesOfPublication362 = datesOfPublication362;
  }

  public String getSourceOfDescription588() {
    return sourceOfDescription588;
  }

  public void setSourceOfDescription588(String sourceOfDescription588) {
    this.sourceOfDescription588 = sourceOfDescription588;
  }

  public int determineRecordQualityScore() {
    int score = 0;
    // Date 1 is totally unknown
    if (fixedFields008.matches(".{7}uuuu.+")) {
      score = score - 3;
    }
    // Country of publication is totally unknown
    if (fixedFields008.matches(".{15}xx.+")) {
      score = score - 1;
    }
    // Publication language is totally unknown
    if (fixedFields008.matches(".{35}xxx.+")) {
      score = score - 1;
    }
    // Authentication code (from the 042) is empty (the record is not pcc or nsdp)
    if (authenticationCode042 != "") {
      score = score + 7;
    }
    // Encoding level is blank or I (fully cataloged)
    if (encodingLevelLDR.matches(".{17} .+")
        || encodingLevelLDR.matches(".{17}I.+")) {
      score = score + 5;
    }
    // Encoding level is M or L (not so fully cataloged, more likely to be a good record than K or 7)
    if (encodingLevelLDR.matches(".{17}M.+")
        || encodingLevelLDR.matches(".{17}L.+")
        || encodingLevelLDR.matches(".{17}K.+")
        || encodingLevelLDR.matches(".{17}7.+")) {
      score = score + 1;
    }
    // 006 is present
    if (fixedLengthData006 != "") {
      score = score + 1;
    }
    // Record has publisher AACR2
    if (publication260 != "") {
      score = score + 1;
    }
    // Record has publisher RDA
    if (publication264 != "") {
      score = score + 1;
    }
    // Publication frequency
    if (frequency310 != "") {
      score = score + 1;
    }
    // RDA fields
    if (rdaContentType336 != "") {
      score = score + 1;
    }
    // Begins with...
    if (datesOfPublication362 != "") {
      score = score + 1;
    }
    // Description based on/ Latest issue consulted notes
    if (sourceOfDescription588 != "") {
      score = score + 1;
    }
    /*
    // Has a Library of Congress subject heading (6XX_0)
    if (getSubject6xx().contains("0$a")) {
      score = score + 2;
    }
    if (getSubject6xx() == "") {
      score = score - 5;
    }
    */

    // Any PCC record should automatically be kept unless it is not online and/or a ceased title
    if (authenticationCode042.contains("pcc")) {
      score = score + 100;
    }

    // Automatic Discards:
    // Discard any that are not "o" for electronic
    if (fixedFields008.matches(".{23}[abcdfqrs].+")
        || fixedFields008.matches(".{23}\\..+")) {
      score = score * 0 - 100;
    }

    // Discard any that are not active titles
    if (fixedFields008.matches(".{11}[0-8].+") ||
      fixedFields008.matches(".{11}u.+")) {
      score = score * 0 - 100;
    }

    // Discard any that are RECORD REPORTED FOR DELETION
    if (notes936.contains("DELETION")) {
      score = score * 0 - 100;
    }

    // Discard any with a first date of "0"
    if (fixedFields008.matches(".{7}0.+")) {
      score = score * 0 - 100;
    }

    // Discard any with an encoding level of "3"
    if (encodingLevelLDR.matches(".{17}3.+")) {
      score = score * 0 - 100;
    }

    System.out.print(
      // Record quality ranking
      // Record score
      // OCLC number
      oclcNumber001
      + ", " + fixedFields008.charAt(23)
      // ISSNs
      + ", " + issn022
      // Dates
      + ", " + fixedFields008.charAt(7) + fixedFields008.charAt(8) + fixedFields008.charAt(9) + fixedFields008.charAt(10)
      + ", " + fixedFields008.charAt(11) + fixedFields008.charAt(12) + fixedFields008.charAt(13) + fixedFields008.charAt(14)
      // Encoding level
      + ", " + encodingLevelLDR.charAt(17)
      // Title
      + ", " + title245 + ", "
    );
    return score;
  }
}
