/*
 * Decompiled with CFR 0.152.
 */
package de.gsi.chart.samples.utils;

import de.gsi.chart.samples.utils.TestDataSetSource;
import de.gsi.dataset.utils.DoubleCircularBuffer;
import de.gsi.math.spectra.Apodization;
import de.gsi.math.spectra.SpectrumTools;
import java.io.BufferedInputStream;
import java.io.IOException;
import javax.sound.midi.InvalidMidiDataException;
import javax.sound.midi.MetaMessage;
import javax.sound.midi.MidiChannel;
import javax.sound.midi.MidiEvent;
import javax.sound.midi.MidiMessage;
import javax.sound.midi.MidiSystem;
import javax.sound.midi.MidiUnavailableException;
import javax.sound.midi.Sequence;
import javax.sound.midi.Sequencer;
import javax.sound.midi.ShortMessage;
import javax.sound.midi.Synthesizer;
import javax.sound.midi.Track;
import net.jafama.FastMath;
import org.jtransforms.fft.FloatFFT_1D;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MidiWaveformSynthesizer {
    private static final Logger LOGGER = LoggerFactory.getLogger(MidiWaveformSynthesizer.class);
    private static final String[] KEY_NAMES = new String[]{"C", "C#", "D", "D#", "E", "F", "F#", "G", "G#", "A", "A#", "B"};
    private static final int LOCAL_NOTE_ON = 1;
    private static final int LOCAL_NOTE_OFF = 2;
    private static final int N_NOTES = 128;
    protected transient Sequencer sequencer;
    protected transient Sequence sequence;
    protected transient Synthesizer synthesizer;
    protected transient MidiChannel synthesizerChannel;
    private final float[] noteAmplitude = new float[128];
    private final float[] noteFrequency = new float[128];
    private boolean muteOutput = true;
    private float noteAmplitudeDecay = 0.1f;
    private final Object lockObject = new Object();
    private DoubleCircularBuffer buffer;
    private int counter = 0;

    public MidiWaveformSynthesizer(String midiFile, int bufferSize) {
        this.buffer = new DoubleCircularBuffer(bufferSize);
        for (int i = 0; i < 128; ++i) {
            this.noteFrequency[i] = (float)(Math.PI * 2 * (double)(440.0f + 36.666668f * (float)(i - 69)));
        }
        try (BufferedInputStream is = new BufferedInputStream(TestDataSetSource.class.getResourceAsStream(midiFile));){
            MidiChannel[] channels;
            this.sequencer = MidiSystem.getSequencer(false);
            this.sequencer.open();
            this.sequence = MidiSystem.getSequence(is);
            this.sequencer.setLoopCount(-1);
            Track[] tracks = this.sequence.getTracks();
            Track trk = this.sequence.createTrack();
            for (Track track : tracks) {
                MidiWaveformSynthesizer.addNotesToTrack(track, trk);
            }
            this.sequencer.setSequence(this.sequence);
            this.synthesizer = MidiSystem.getSynthesizer();
            this.synthesizer.open();
            for (MidiChannel channel : channels = this.synthesizer.getChannels()) {
                if (channel == null) continue;
                this.synthesizerChannel = channel;
                break;
            }
            this.sequencer.addMetaEventListener(evt -> {
                int command = evt.getType();
                byte[] data = evt.getData();
                if (data.length < 2 || command != 1 && command != 2 && command != 176) {
                    return;
                }
                int note = evt.getData()[1] & 0xFF;
                int velocity = evt.getData()[2] & 0xFF;
                if (command == 1) {
                    this.noteAmplitude[note] = velocity;
                    if (!this.muteOutput) {
                        this.synthesizerChannel.noteOn(note, velocity);
                    }
                    if (LOGGER.isDebugEnabled()) {
                        LOGGER.atDebug().addArgument((Object)evt).addArgument((Object)note).addArgument((Object)velocity).log("note on event = {}  note = {}  velocity {}");
                    }
                } else if (command == 2) {
                    this.noteAmplitude[note] = 0.0f;
                    this.synthesizerChannel.noteOff(note, 0);
                    if (LOGGER.isDebugEnabled()) {
                        LOGGER.atDebug().addArgument((Object)evt).addArgument((Object)note).addArgument((Object)velocity).log("note off event = {}  note = {}  velocity {}");
                    }
                } else if (command == 176 && LOGGER.isDebugEnabled()) {
                    LOGGER.atDebug().addArgument((Object)evt).addArgument((Object)note).addArgument((Object)velocity).log("generic CONTROL_CHANGE evt = {} bytes = {} {}");
                }
            });
        }
        catch (MidiUnavailableException e) {
            LOGGER.atError().setCause((Throwable)e).log("could not initialise MidiSystem");
        }
        catch (IOException e) {
            LOGGER.atError().setCause((Throwable)e).addArgument((Object)TestDataSetSource.class.getResource(midiFile)).log("could not open file '{}'");
        }
        catch (InvalidMidiDataException e) {
            LOGGER.atError().setCause((Throwable)e).addArgument((Object)midiFile).log("'{}' does not seem to be recognised as a Midi file");
        }
    }

    public void decode(float[] data, int frameSize, int updatePeriod, int samplingRate, int nBits) {
        if (frameSize <= 0) {
            throw new IllegalArgumentException("Frame size must be greater than zero");
        }
        Track track = this.mergeShortMessageEvent(this.sequence.getTracks());
        float length = 1.0E-6f * (float)this.sequence.getMicrosecondLength();
        float ts = 1.0f / (float)samplingRate;
        long trackTicks = track.ticks();
        float tickLength = trackTicks <= 0L ? 0.0f : length / (float)trackTicks;
        int frameCount = data.length / frameSize;
        float scale = 2 << Math.max(1, nBits + 1);
        int fftSize = 2 * frameSize;
        FloatFFT_1D fft = new FloatFFT_1D((long)fftSize);
        float[] apodization = new float[fftSize];
        for (int i = 0; i < fftSize; ++i) {
            apodization[i] = (float)Apodization.Hann.getIndex(i, apodization.length);
        }
        int frameCounter = 0;
        int tickIndex = 0;
        float[] waveForm = new float[2 * frameSize];
        int nUpdateDistance = (int)((double)updatePeriod / 1000.0 * (double)samplingRate);
        int i = 0;
        while (frameCounter < frameCount) {
            float t = (float)i * ts;
            MidiEvent tickEvt = track.get(tickIndex);
            float tickTimeStamp = (float)tickEvt.getTick() * tickLength;
            this.update(samplingRate, nBits);
            if (t > tickTimeStamp && tickIndex < track.size() - 1) {
                if (tickEvt.getMessage() instanceof ShortMessage) {
                    ShortMessage sm = (ShortMessage)tickEvt.getMessage();
                    int note = sm.getData1() & 0xFF;
                    int velocity = sm.getData2() & 0xFF;
                    int command = sm.getCommand();
                    if (command == 144 || command == 1) {
                        this.noteAmplitude[note] = velocity;
                    } else if (command == 128 || command == 2) {
                        this.noteAmplitude[note] = 0.0f;
                    }
                }
                ++tickIndex;
            }
            if (i > 0 && i % nUpdateDistance == 0) {
                for (int j = 0; j < waveForm.length; ++j) {
                    float noise = 0.001f * (float)System.nanoTime() % 2.0f;
                    waveForm[j] = apodization[j] * this.getSample(j) + noise / scale;
                }
                MidiWaveformSynthesizer.decodeFrame(fft, waveForm, data, frameCounter * frameSize % data.length);
                ++frameCounter;
            }
            ++i;
        }
        for (int note = 0; note < 128; ++note) {
            this.noteAmplitude[note] = 0.0f;
            this.synthesizerChannel.noteOff(note, 0);
        }
    }

    public void finalize() {
        this.sequencer.close();
        this.synthesizer.close();
    }

    public DoubleCircularBuffer getBuffer() {
        return this.buffer;
    }

    public float getNoteAmplitudeDecay() {
        return this.noteAmplitudeDecay;
    }

    public float getSample(int readPos) {
        return (float)this.buffer.get(readPos);
    }

    public boolean isOutputMuted() {
        return this.muteOutput;
    }

    public final Track mergeShortMessageEvent(Track[] tracks) {
        Track trk = this.sequence.createTrack();
        for (Track track : tracks) {
            for (int i = 0; i < track.size(); ++i) {
                MidiEvent evt = track.get(i);
                MidiMessage mm = evt.getMessage();
                if (!(mm instanceof ShortMessage)) continue;
                trk.add(evt);
            }
        }
        return trk;
    }

    public void pause() {
        this.sequencer.stop();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void reset() {
        Object object = this.lockObject;
        synchronized (object) {
            this.counter = 0;
            for (int note = 0; note < 128; ++note) {
                this.noteAmplitude[note] = 0.0f;
                this.synthesizerChannel.noteOff(note, 0);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void setBufferLength(int bufferSize) {
        Object object = this.lockObject;
        synchronized (object) {
            this.buffer = new DoubleCircularBuffer(bufferSize);
            this.reset();
        }
    }

    public void setNoteAmplitudeDecay(float noteAmplitudeDecay) {
        this.noteAmplitudeDecay = noteAmplitudeDecay;
    }

    public void setOutputMuted(boolean state) {
        this.muteOutput = state;
        if (this.muteOutput) {
            this.reset();
        }
    }

    public void start() {
        this.reset();
        this.sequencer.start();
    }

    public void stop() {
        this.sequencer.stop();
        this.sequencer.setTickPosition(0L);
        this.reset();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void update(int samplingRate, int nBits) {
        Object object = this.lockObject;
        synchronized (object) {
            double val = 0.0;
            double scale = 2 << Math.max(1, nBits - 8);
            double ts = 1.0 / (double)samplingRate;
            float alpha = (float)Math.exp(-ts / (double)this.noteAmplitudeDecay);
            for (int i = 0; i < 128; ++i) {
                if (!(this.noteAmplitude[i] > 0.0f)) continue;
                val += scale * (double)this.noteAmplitude[i] * FastMath.sinQuick((double)(this.noteFrequency[i] * (float)this.counter / (float)samplingRate));
                int n = i;
                this.noteAmplitude[n] = this.noteAmplitude[n] * alpha;
                if (!(this.noteAmplitude[i] < 1.0f)) continue;
                this.noteAmplitude[i] = 0.0f;
            }
            ++this.counter;
            this.buffer.put(val);
        }
    }

    public static final void addNotesToTrack(Track track, Track trk) throws InvalidMidiDataException {
        for (int ii = 0; ii < track.size(); ++ii) {
            MidiEvent me = track.get(ii);
            MidiMessage mm = me.getMessage();
            if (!(mm instanceof ShortMessage)) continue;
            ShortMessage sm = (ShortMessage)mm;
            int command = sm.getCommand();
            int com = -1;
            if (command == 144) {
                com = 1;
            } else if (command == 128) {
                com = 2;
            }
            if (com <= 0) continue;
            byte[] b = sm.getMessage();
            int l = b == null ? 0 : b.length;
            MetaMessage metaMessage = new MetaMessage(com, b, l);
            MidiEvent me2 = new MidiEvent(metaMessage, me.getTick());
            trk.add(me2);
        }
    }

    private static void decodeFrame(FloatFFT_1D fft, float[] in, float[] out, int offset) {
        fft.realForward(in);
        float[] mag = SpectrumTools.computeMagnitudeSpectrum_dB((float[])in, (boolean)true);
        System.arraycopy(mag, 0, out, offset, mag.length);
    }

    public static String keyName(int nKeyNumber) {
        if (nKeyNumber > 127) {
            return "illegal value";
        }
        int nNote = nKeyNumber % 12;
        int nOctave = nKeyNumber / 12;
        return KEY_NAMES[nNote] + (nOctave - 1);
    }
}

