/*
 * Decompiled with CFR 0.152.
 */
package de.griefed.serverpackcreator.utilities.common;

import de.griefed.serverpackcreator.utilities.common.InvalidFileTypeException;
import de.griefed.serverpackcreator.utilities.common.InvalidLinkException;
import java.awt.Desktop;
import java.awt.GraphicsEnvironment;
import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.InvalidPathException;
import java.nio.file.Path;
import java.nio.file.Paths;
import mslinks.ShellLink;
import mslinks.ShellLinkException;
import net.lingala.zip4j.ZipFile;
import org.apache.commons.io.FileUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

public class FileUtilities {
    private static final Logger LOG = LogManager.getLogger(FileUtilities.class);

    public boolean replaceFile(File sourceFile, File destinationFile) throws IOException {
        if (sourceFile.exists() && destinationFile.delete()) {
            FileUtils.moveFile(sourceFile, destinationFile);
            return true;
        }
        LOG.error("Source file not found.");
        return false;
    }

    public void unzipArchive(String zipFile, String destinationDirectory) {
        LOG.info("Extracting ZIP-file: " + zipFile);
        try (ZipFile zip = new ZipFile(zipFile);){
            zip.extractAll(destinationDirectory);
        }
        catch (IOException ex) {
            LOG.error("Error: There was an error extracting the archive " + zipFile, (Throwable)ex);
        }
    }

    public FileType checkFileType(String file) {
        if (file.length() == 0) {
            return FileType.INVALID;
        }
        return this.checkFileType(new File(file));
    }

    public FileType checkFileType(File file) {
        if (file.getName().endsWith("lnk")) {
            return FileType.LINK;
        }
        if (file.isDirectory()) {
            return FileType.DIRECTORY;
        }
        if (FileUtils.isSymlink(file)) {
            return FileType.SYMLINK;
        }
        if (file.isFile()) {
            return FileType.FILE;
        }
        return FileType.INVALID;
    }

    public boolean isLink(String file) {
        return this.isLink(new File(file));
    }

    public boolean isLink(File file) {
        if (file.getName().endsWith("lnk")) {
            return true;
        }
        return !file.toString().matches("[A-Za-z]:.*") && FileUtils.isSymlink(file);
    }

    public String resolveLink(String link) throws InvalidFileTypeException, IOException {
        return this.resolveLink(new File(link));
    }

    public String resolveLink(File link) throws IOException, InvalidFileTypeException {
        FileType type = this.checkFileType(link);
        switch (type) {
            case LINK: 
            case SYMLINK: {
                try {
                    return this.resolveLink(link, type);
                }
                catch (InvalidFileTypeException | InvalidLinkException | ShellLinkException ex) {
                    LOG.error("Somehow an invalid FileType was specified. Please report this on GitHub!", (Throwable)ex);
                }
            }
            case FILE: 
            case DIRECTORY: {
                return link.toString();
            }
        }
        throw new InvalidFileTypeException("FileType must be either LINK or SYMLINK");
    }

    private String resolveLink(File file, FileType fileType) throws InvalidFileTypeException, IOException, InvalidLinkException, ShellLinkException {
        switch (fileType) {
            case SYMLINK: {
                return file.getCanonicalPath();
            }
            case LINK: {
                return new ShellLink(file).resolveTarget();
            }
        }
        throw new InvalidFileTypeException("FileType must be either LINK or SYMLINK");
    }

    public boolean checkPermissions(String fileOrDirectory) throws InvalidPathException {
        return this.checkPermissions(Paths.get(fileOrDirectory, new String[0]));
    }

    public boolean checkPermissions(File fileOrDirectory) throws InvalidPathException {
        return this.checkPermissions(fileOrDirectory.toPath());
    }

    public boolean checkPermissions(Path fileOrDirectory) {
        return this.checkReadPermission(fileOrDirectory) && this.checkWritePermission(fileOrDirectory);
    }

    public boolean checkReadPermission(String fileOrDirectory) {
        return this.checkReadPermission(Paths.get(fileOrDirectory, new String[0]));
    }

    public boolean checkReadPermission(File fileOrDirectory) {
        return this.checkReadPermission(fileOrDirectory.toPath());
    }

    public boolean checkReadPermission(Path fileOrDirectory) {
        try {
            if (!Files.isReadable(fileOrDirectory)) {
                LOG.error(String.format("No read-permission for %s", fileOrDirectory));
                return false;
            }
        }
        catch (SecurityException ex) {
            LOG.error(String.format("No read-permission for %s", fileOrDirectory), (Throwable)ex);
            return false;
        }
        return true;
    }

    public boolean checkWritePermission(String fileOrDirectory) {
        return this.checkReadPermission(Paths.get(fileOrDirectory, new String[0]));
    }

    public boolean checkWritePermission(File fileOrDirectory) {
        return this.checkReadPermission(fileOrDirectory.toPath());
    }

    public boolean checkWritePermission(Path fileOrDirectory) {
        try {
            if (!Files.isWritable(fileOrDirectory)) {
                LOG.error(String.format("No write-permission for %s", fileOrDirectory));
                return false;
            }
        }
        catch (SecurityException ex) {
            LOG.error(String.format("No write-permission for %s", fileOrDirectory), (Throwable)ex);
            return false;
        }
        return true;
    }

    public void openFolder(String folder) {
        if (GraphicsEnvironment.isHeadless()) {
            LOG.error("Graphics environment not supported.");
        } else {
            try {
                Desktop.getDesktop().open(new File(folder));
            }
            catch (IOException ex) {
                LOG.error("Error opening file explorer for " + folder + ".", (Throwable)ex);
            }
        }
    }

    public void openFile(String fileToOpen) {
        if (GraphicsEnvironment.isHeadless()) {
            LOG.error("Graphics environment not supported.");
        } else {
            try {
                if (Desktop.getDesktop().isSupported(Desktop.Action.EDIT)) {
                    Desktop.getDesktop().open(new File(fileToOpen));
                }
            }
            catch (IOException ex) {
                LOG.error("Error opening custom server-icon.png.", (Throwable)ex);
            }
        }
    }

    public static enum FileType {
        FILE,
        DIRECTORY,
        LINK,
        SYMLINK,
        INVALID;

    }
}

