/* Copyright (C) 2022  Griefed
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301
 * USA
 *
 * The full license can be found at https:github.com/Griefed/ServerPackCreator/blob/main/LICENSE
 */
package de.griefed.serverpackcreator.swing;

import de.griefed.serverpackcreator.ApplicationProperties;
import de.griefed.serverpackcreator.i18n.LocalizationManager;
import de.griefed.serverpackcreator.swing.themes.DarkTheme;
import de.griefed.serverpackcreator.swing.themes.LightTheme;
import de.griefed.serverpackcreator.utilities.UpdateChecker;
import de.griefed.serverpackcreator.utilities.common.InvalidFileTypeException;
import de.griefed.serverpackcreator.utilities.misc.Generated;
import de.griefed.versionchecker.Update;
import java.awt.Component;
import java.awt.Desktop;
import java.awt.Dialog;
import java.awt.Dimension;
import java.awt.Toolkit;
import java.awt.Window;
import java.awt.datatransfer.Clipboard;
import java.awt.datatransfer.StringSelection;
import java.awt.event.ActionEvent;
import java.awt.event.WindowEvent;
import java.io.BufferedReader;
import java.io.DataOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.net.ProtocolException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.util.Objects;
import java.util.Optional;
import java.util.Properties;
import javax.net.ssl.HttpsURLConnection;
import javax.swing.ImageIcon;
import javax.swing.JFileChooser;
import javax.swing.JFrame;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.JScrollPane;
import javax.swing.JSeparator;
import javax.swing.JTabbedPane;
import javax.swing.JTextPane;
import javax.swing.SwingUtilities;
import javax.swing.UIManager;
import javax.swing.UnsupportedLookAndFeelException;
import javax.swing.filechooser.FileNameExtensionFilter;
import javax.swing.text.BadLocationException;
import javax.swing.text.DefaultStyledDocument;
import javax.swing.text.SimpleAttributeSet;
import javax.swing.text.StyleConstants;
import javax.swing.text.StyledDocument;
import javax.swing.text.Utilities;
import mdlaf.MaterialLookAndFeel;
import mdlaf.components.textpane.MaterialTextPaneUI;
import org.apache.commons.io.FileUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

/**
 * This class creates our menubar which will be displayed at the top of the ServerPackCreator frame.
 * It contains various menus and menuitems to execute, change, open and edit various different
 * aspects of ServerPackCreator.
 *
 * @author Griefed
 */
@Generated
public class MainMenuBar extends Component {

  private static final Logger LOG = LogManager.getLogger(MainMenuBar.class);

  private final Clipboard CLIPBOARD = Toolkit.getDefaultToolkit().getSystemClipboard();

  private final LocalizationManager LOCALIZATIONMANAGER;
  private final ApplicationProperties APPLICATIONPROPERTIES;
  private final UpdateChecker UPDATECHECKER;
  private final de.griefed.serverpackcreator.utilities.common.Utilities UTILITIES;

  private final LightTheme LIGHTTHEME;
  private final DarkTheme DARKTHEME;

  private final JFrame FRAME_SERVERPACKCREATOR;

  private final TabCreateServerPack TAB_CREATESERVERPACK;

  private final JTabbedPane TABBEDPANE;

  private final WindowEvent CLOSEEVENT;

  private final MaterialLookAndFeel LAF_DARK;
  private final MaterialLookAndFeel LAF_LIGHT;

  private final Dimension CHOOSERDIMENSION = new Dimension(750, 450);
  private final Dimension ABOUTDIMENSION = new Dimension(925, 520);

  private final ImageIcon HELPICON =
      new ImageIcon(
          Objects.requireNonNull(
              ServerPackCreatorGui.class.getResource("/de/griefed/resources/gui/help.png")));
  private final ImageIcon ICON_HASTEBIN =
      new ImageIcon(
          Objects.requireNonNull(
              ServerPackCreatorGui.class.getResource("/de/griefed/resources/gui/hastebin.png")));

  private final JMenuBar MENUBAR = new JMenuBar();

  private final String[] HASTEOPTIONS = new String[3];

  private final StyledDocument ABOUT_DOCUMENT = new DefaultStyledDocument();
  private final StyledDocument CONFIG_DOCUMENT = new DefaultStyledDocument();
  private final StyledDocument SPCLOG_DOCUMENT = new DefaultStyledDocument();

  private final SimpleAttributeSet CONFIG_ATTRIBUTESET = new SimpleAttributeSet();
  private final SimpleAttributeSet SPCLOG_ATTRIBUTESET = new SimpleAttributeSet();

  private final JTextPane ABOUT_WINDOW_TEXTPANE = new JTextPane(ABOUT_DOCUMENT);
  private final JTextPane CONFIG_WINDOW_TEXTPANE = new JTextPane(CONFIG_DOCUMENT);
  private final JTextPane SPCLOG_WINDOW_TEXTPANE = new JTextPane(SPCLOG_DOCUMENT);
  private final JTextPane FILETOOLARGE_WINDOW_TEXTPANE = new JTextPane();

  private final MaterialTextPaneUI MATERIALTEXTPANEUI = new MaterialTextPaneUI();

  private boolean isDarkTheme;

  private JFileChooser configChooser;

  private File lastLoadedConfigurationFile = null;

  /**
   * Constructor for our MainMenuBar. Prepares various Strings, Arrays, Panels and windows.
   *
   * @param injectedLocalizationManager Instance of {@link LocalizationManager} required for
   *     localized log messages.
   * @param injectedLightTheme Instance of {@link LightTheme} required for theme switching.
   * @param injectedDarkTheme Instance of {@link DarkTheme} required for theme switching.
   * @param injectedJFrame The parent from in which everything ServerPackCreator is displayed in.
   * @param injectedLAF_Light Instance of {@link MaterialLookAndFeel} with our {@link LightTheme}.
   * @param injectedLAF_Dark Instance of {@link MaterialLookAndFeel} with our {@link DarkTheme}.
   * @param injectedTabCreateServerPack Our tab for configuring ServerPackCreator.
   * @param injectedTabbedPane The tabbed pane which holds all our tabs.
   * @param injectedApplicationProperties Instance of {@link Properties} required for various
   *     different things.
   * @param injectedUpdateChecker Instance of {@link UpdateChecker} to check for
   *     update-availability.
   * @param injectedUtilities Instance of {@link Utilities} for various things.
   * @author Griefed
   */
  public MainMenuBar(
      LocalizationManager injectedLocalizationManager,
      LightTheme injectedLightTheme,
      DarkTheme injectedDarkTheme,
      JFrame injectedJFrame,
      MaterialLookAndFeel injectedLAF_Light,
      MaterialLookAndFeel injectedLAF_Dark,
      TabCreateServerPack injectedTabCreateServerPack,
      JTabbedPane injectedTabbedPane,
      ApplicationProperties injectedApplicationProperties,
      UpdateChecker injectedUpdateChecker,
      de.griefed.serverpackcreator.utilities.common.Utilities injectedUtilities) {

    if (injectedApplicationProperties == null) {
      this.APPLICATIONPROPERTIES = new ApplicationProperties();
    } else {
      this.APPLICATIONPROPERTIES = injectedApplicationProperties;
    }

    if (injectedLocalizationManager == null) {
      this.LOCALIZATIONMANAGER = new LocalizationManager(APPLICATIONPROPERTIES);
    } else {
      this.LOCALIZATIONMANAGER = injectedLocalizationManager;
    }

    if (injectedUpdateChecker == null) {
      this.UPDATECHECKER = new UpdateChecker(LOCALIZATIONMANAGER, APPLICATIONPROPERTIES);
    } else {
      this.UPDATECHECKER = injectedUpdateChecker;
    }

    if (injectedUtilities == null) {
      this.UTILITIES =
          new de.griefed.serverpackcreator.utilities.common.Utilities(
              LOCALIZATIONMANAGER, APPLICATIONPROPERTIES);
    } else {
      this.UTILITIES = injectedUtilities;
    }

    this.LIGHTTHEME = injectedLightTheme;
    this.DARKTHEME = injectedDarkTheme;
    this.FRAME_SERVERPACKCREATOR = injectedJFrame;
    this.LAF_LIGHT = injectedLAF_Light;
    this.LAF_DARK = injectedLAF_Dark;
    this.TAB_CREATESERVERPACK = injectedTabCreateServerPack;
    this.TABBEDPANE = injectedTabbedPane;

    try {
      isDarkTheme =
          Boolean.parseBoolean(
              APPLICATIONPROPERTIES.getProperty("de.griefed.serverpackcreator.gui.darkmode"));
    } catch (NullPointerException ex) {
      LOG.error("No setting for darkmode found in properties-file. Using true.");
      isDarkTheme = true;
      APPLICATIONPROPERTIES.put("de.griefed.serverpackcreator.gui.darkmode", "true");
    }

    CLOSEEVENT = new WindowEvent(FRAME_SERVERPACKCREATOR, WindowEvent.WINDOW_CLOSING);

    String ABOUTWINDOWTEXT =
        LOCALIZATIONMANAGER.getLocalizedString("createserverpack.gui.about.text");
    ABOUT_WINDOW_TEXTPANE.setEditable(false);
    ABOUT_WINDOW_TEXTPANE.setOpaque(false);
    ABOUT_WINDOW_TEXTPANE.setMinimumSize(ABOUTDIMENSION);
    ABOUT_WINDOW_TEXTPANE.setPreferredSize(ABOUTDIMENSION);
    ABOUT_WINDOW_TEXTPANE.setMaximumSize(ABOUTDIMENSION);
    SimpleAttributeSet ABOUTATTRIBUTESET = new SimpleAttributeSet();
    StyleConstants.setBold(ABOUTATTRIBUTESET, true);
    StyleConstants.setFontSize(ABOUTATTRIBUTESET, 14);
    ABOUT_WINDOW_TEXTPANE.setCharacterAttributes(ABOUTATTRIBUTESET, true);
    StyleConstants.setAlignment(ABOUTATTRIBUTESET, StyleConstants.ALIGN_CENTER);
    ABOUT_DOCUMENT.setParagraphAttributes(0, ABOUT_DOCUMENT.getLength(), ABOUTATTRIBUTESET, false);
    try {
      ABOUT_DOCUMENT.insertString(0, ABOUTWINDOWTEXT, ABOUTATTRIBUTESET);
    } catch (BadLocationException ex) {
      LOG.error("Error inserting text into aboutDocument.", ex);
    }
    ABOUT_WINDOW_TEXTPANE.addHierarchyListener(
        e1 -> {
          Window window = SwingUtilities.getWindowAncestor(ABOUT_WINDOW_TEXTPANE);
          if (window instanceof Dialog) {
            Dialog dialog = (Dialog) window;
            if (!dialog.isResizable()) {
              dialog.setResizable(true);
            }
          }
        });

    HASTEOPTIONS[0] =
        LOCALIZATIONMANAGER.getLocalizedString("createserverpack.gui.about.hastebin.dialog.yes");
    HASTEOPTIONS[1] =
        LOCALIZATIONMANAGER.getLocalizedString(
            "createserverpack.gui.about.hastebin.dialog.clipboard");
    HASTEOPTIONS[2] =
        LOCALIZATIONMANAGER.getLocalizedString("createserverpack.gui.about.hastebin.dialog.no");

    CONFIG_WINDOW_TEXTPANE.setOpaque(false);
    CONFIG_WINDOW_TEXTPANE.setEditable(false);
    StyleConstants.setBold(CONFIG_ATTRIBUTESET, true);
    StyleConstants.setFontSize(CONFIG_ATTRIBUTESET, 14);
    CONFIG_WINDOW_TEXTPANE.setCharacterAttributes(CONFIG_ATTRIBUTESET, true);
    StyleConstants.setAlignment(CONFIG_ATTRIBUTESET, StyleConstants.ALIGN_LEFT);
    CONFIG_DOCUMENT.setParagraphAttributes(
        0, CONFIG_DOCUMENT.getLength(), CONFIG_ATTRIBUTESET, false);
    CONFIG_WINDOW_TEXTPANE.addHierarchyListener(
        e1 -> {
          Window window = SwingUtilities.getWindowAncestor(CONFIG_WINDOW_TEXTPANE);
          if (window instanceof Dialog) {
            Dialog dialog = (Dialog) window;
            if (!dialog.isResizable()) {
              dialog.setResizable(true);
            }
          }
        });

    SPCLOG_WINDOW_TEXTPANE.setOpaque(false);
    SPCLOG_WINDOW_TEXTPANE.setEditable(false);
    StyleConstants.setBold(SPCLOG_ATTRIBUTESET, true);
    StyleConstants.setFontSize(SPCLOG_ATTRIBUTESET, 14);
    SPCLOG_WINDOW_TEXTPANE.setCharacterAttributes(SPCLOG_ATTRIBUTESET, true);
    StyleConstants.setAlignment(SPCLOG_ATTRIBUTESET, StyleConstants.ALIGN_LEFT);
    SPCLOG_DOCUMENT.setParagraphAttributes(
        0, SPCLOG_DOCUMENT.getLength(), SPCLOG_ATTRIBUTESET, false);
    SPCLOG_WINDOW_TEXTPANE.addHierarchyListener(
        e1 -> {
          Window window = SwingUtilities.getWindowAncestor(SPCLOG_WINDOW_TEXTPANE);
          if (window instanceof Dialog) {
            Dialog dialog = (Dialog) window;
            if (!dialog.isResizable()) {
              dialog.setResizable(true);
            }
          }
        });
  }

  /**
   * Create the menubar, add all menus, add all menuitems and add actionlisteners for our menuitems.
   *
   * @return JMenuBar. Returns the menubar containing all elements we need to control various
   *     aspects of our app.
   * @author Griefed
   */
  public JMenuBar createMenuBar() {

    // create menus
    JMenu fileMenu = new JMenu(LOCALIZATIONMANAGER.getLocalizedString("menubar.gui.menu.file"));
    JMenu editMenu = new JMenu(LOCALIZATIONMANAGER.getLocalizedString("menubar.gui.menu.edit"));
    JMenu viewMenu = new JMenu(LOCALIZATIONMANAGER.getLocalizedString("menubar.gui.menu.view"));
    JMenu aboutMenu = new JMenu(LOCALIZATIONMANAGER.getLocalizedString("menubar.gui.menu.about"));

    // create menu items
    JMenuItem file_NewConfigurationMenuItem = new JMenuItem("New configuration");
    JMenuItem file_LoadConfigMenuItem =
        new JMenuItem(LOCALIZATIONMANAGER.getLocalizedString("menubar.gui.menuitem.loadconfig"));
    JMenuItem file_SaveConfigMenuItem =
        new JMenuItem(LOCALIZATIONMANAGER.getLocalizedString("menubar.gui.menuitem.saveconfig"));
    JMenuItem file_SaveAsConfigMenuItem =
        new JMenuItem(LOCALIZATIONMANAGER.getLocalizedString("menubar.gui.menuitem.saveas"));
    JMenuItem file_UploadConfigurationToHasteBin =
        new JMenuItem(LOCALIZATIONMANAGER.getLocalizedString("menubar.gui.menuitem.uploadconfig"));
    JMenuItem file_UploadServerPackCreatorLogToHasteBin =
        new JMenuItem(LOCALIZATIONMANAGER.getLocalizedString("menubar.gui.menuitem.uploadlog"));
    JMenuItem file_UpdateFallbackModslist =
        new JMenuItem(
            LOCALIZATIONMANAGER.getLocalizedString("menubar.gui.menuitem.updatefallback"));
    JMenuItem file_ExitConfigMenuItem =
        new JMenuItem(LOCALIZATIONMANAGER.getLocalizedString("menubar.gui.menuitem.exit"));

    JMenuItem edit_SwitchTheme =
        new JMenuItem(LOCALIZATIONMANAGER.getLocalizedString("menubar.gui.menuitem.theme"));
    JMenuItem edit_OpenInEditorServerProperties =
        new JMenuItem(
            LOCALIZATIONMANAGER.getLocalizedString("menubar.gui.menuitem.serverproperties"));
    JMenuItem edit_OpenInEditorServerIcon =
        new JMenuItem(LOCALIZATIONMANAGER.getLocalizedString("menubar.gui.menuitem.servericon"));

    JMenuItem view_OpenAddonsDirectoryMenuItem =
        new JMenuItem(LOCALIZATIONMANAGER.getLocalizedString("menubar.gui.menuitem.addonsdir"));
    JMenuItem view_ExampleAddonRepositoryMenuItem =
        new JMenuItem(
            LOCALIZATIONMANAGER.getLocalizedString("menubar.gui.menuitem.exampleaddonrepo"));
    JMenuItem view_OpenServerPackCreatorDirectoryMenuItem =
        new JMenuItem(LOCALIZATIONMANAGER.getLocalizedString("menubar.gui.menuitem.spcdir"));
    JMenuItem view_OpenServerPacksDirectoryMenuItem =
        new JMenuItem(
            LOCALIZATIONMANAGER.getLocalizedString("menubar.gui.menuitem.serverpacksdir"));
    JMenuItem view_OpenServerFilesDirectoryMenuItem =
        new JMenuItem(
            LOCALIZATIONMANAGER.getLocalizedString("menubar.gui.menuitem.serverfilesdir"));
    JMenuItem view_OpenSPCLog =
        new JMenuItem(LOCALIZATIONMANAGER.getLocalizedString("menubar.gui.menuitem.spclog"));
    JMenuItem view_OpenModloaderInstallerLog =
        new JMenuItem(LOCALIZATIONMANAGER.getLocalizedString("menubar.gui.menuitem.modloaderlog"));
    JMenuItem view_OpenAddonLog =
        new JMenuItem(LOCALIZATIONMANAGER.getLocalizedString("menubar.gui.menuitem.addonlog"));

    JMenuItem about_OpenAboutWindowMenuItem =
        new JMenuItem(LOCALIZATIONMANAGER.getLocalizedString("menubar.gui.menuitem.about"));
    JMenuItem about_OpenGitHubPageMenuItem =
        new JMenuItem(LOCALIZATIONMANAGER.getLocalizedString("menubar.gui.menuitem.repository"));
    JMenuItem about_OpenGitHubIssuesPageMenuItem =
        new JMenuItem(LOCALIZATIONMANAGER.getLocalizedString("menubar.gui.menuitem.issues"));
    JMenuItem about_OpenReleasesPageMenuItem =
        new JMenuItem(LOCALIZATIONMANAGER.getLocalizedString("menubar.gui.menuitem.releases"));
    JMenuItem about_OpenDiscordLinkMenuItem =
        new JMenuItem(LOCALIZATIONMANAGER.getLocalizedString("menubar.gui.menuitem.discord"));
    JMenuItem about_OpenDonationsPageMenuItem =
        new JMenuItem(LOCALIZATIONMANAGER.getLocalizedString("menubar.gui.menuitem.donate"));
    JMenuItem about_OpenWikiHelpMenuItem =
        new JMenuItem(LOCALIZATIONMANAGER.getLocalizedString("menubar.gui.menuitem.wiki.help"));
    JMenuItem about_OpenWikiHowToMenuItem =
        new JMenuItem(LOCALIZATIONMANAGER.getLocalizedString("menubar.gui.menuitem.wiki.howto"));
    JMenuItem about_CheckForUpdates =
        new JMenuItem(LOCALIZATIONMANAGER.getLocalizedString("menubar.gui.menuitem.updates"));

    // create action listeners for items
    file_NewConfigurationMenuItem.addActionListener(this::newConfiguration);
    file_LoadConfigMenuItem.addActionListener(this::loadConfigurationFromFileMenuItem);
    file_SaveConfigMenuItem.addActionListener(this::saveConfigToFileMenuItem);
    file_SaveAsConfigMenuItem.addActionListener(this::saveAsConfigToFileMenuItem);
    file_UploadConfigurationToHasteBin.addActionListener(
        this::uploadConfigurationToHasteBinMenuItem);
    file_UploadServerPackCreatorLogToHasteBin.addActionListener(
        this::uploadServerPackCreatorLogToHasteBinMenuItem);
    file_UpdateFallbackModslist.addActionListener(this::updateFallbackModslist);
    file_ExitConfigMenuItem.addActionListener(this::exitMenuItem);

    edit_SwitchTheme.addActionListener(this::switchThemeMenuItem);
    edit_OpenInEditorServerProperties.addActionListener(this::openInEditorServerProperties);
    edit_OpenInEditorServerIcon.addActionListener(this::openServerIcon);

    view_OpenServerPackCreatorDirectoryMenuItem.addActionListener(this::openSPCDirectoryMenuItem);
    view_OpenServerPacksDirectoryMenuItem.addActionListener(this::openServerPacksDirectoryMenuItem);
    view_OpenServerFilesDirectoryMenuItem.addActionListener(this::openServerFilesDirectoryMenuItem);
    view_OpenAddonsDirectoryMenuItem.addActionListener(this::openPluginsDirectoryMenuItem);
    view_ExampleAddonRepositoryMenuItem.addActionListener(this::viewExampleAddonMenuItem);
    view_OpenSPCLog.addActionListener(this::openSPClog);
    view_OpenModloaderInstallerLog.addActionListener(this::openModloaderInstallerLog);
    view_OpenAddonLog.addActionListener(this::openAddonsLog);

    about_OpenAboutWindowMenuItem.addActionListener(this::openAboutSPCMenuItem);
    about_OpenGitHubPageMenuItem.addActionListener(this::openGitHubMenuItem);
    about_OpenGitHubIssuesPageMenuItem.addActionListener(this::openIssuesMenuItem);
    about_OpenReleasesPageMenuItem.addActionListener(this::openReleaseMenuItem);
    about_OpenDiscordLinkMenuItem.addActionListener(this::openDiscordLinkMenuItem);
    about_OpenDonationsPageMenuItem.addActionListener(this::openDonateMenuItem);
    about_OpenWikiHelpMenuItem.addActionListener(this::openWikiHelpMenuItem);
    about_OpenWikiHowToMenuItem.addActionListener(this::openWikiHowToMenuItem);
    about_CheckForUpdates.addActionListener(this::checkForUpdates);

    // add items to menus
    fileMenu.add(file_NewConfigurationMenuItem);
    fileMenu.add(file_LoadConfigMenuItem);
    fileMenu.add(new JSeparator());
    fileMenu.add(file_SaveConfigMenuItem);
    fileMenu.add(file_SaveAsConfigMenuItem);
    fileMenu.add(new JSeparator());
    fileMenu.add(file_UploadConfigurationToHasteBin);
    fileMenu.add(file_UploadServerPackCreatorLogToHasteBin);
    fileMenu.add(new JSeparator());
    fileMenu.add(file_UpdateFallbackModslist);
    fileMenu.add(new JSeparator());
    fileMenu.add(file_ExitConfigMenuItem);

    editMenu.add(edit_OpenInEditorServerProperties);
    editMenu.add(edit_OpenInEditorServerIcon);
    editMenu.add(new JSeparator());
    editMenu.add(edit_SwitchTheme);

    viewMenu.add(view_OpenServerPackCreatorDirectoryMenuItem);
    viewMenu.add(view_OpenServerPacksDirectoryMenuItem);
    viewMenu.add(view_OpenServerFilesDirectoryMenuItem);
    viewMenu.add(view_OpenAddonsDirectoryMenuItem);
    viewMenu.add(new JSeparator());
    viewMenu.add(view_ExampleAddonRepositoryMenuItem);
    viewMenu.add(new JSeparator());
    viewMenu.add(view_OpenSPCLog);
    viewMenu.add(view_OpenModloaderInstallerLog);
    viewMenu.add(view_OpenAddonLog);

    aboutMenu.add(about_OpenAboutWindowMenuItem);
    aboutMenu.add(about_CheckForUpdates);
    aboutMenu.add(new JSeparator());
    aboutMenu.add(about_OpenWikiHelpMenuItem);
    aboutMenu.add(about_OpenWikiHowToMenuItem);
    aboutMenu.add(new JSeparator());
    aboutMenu.add(about_OpenGitHubPageMenuItem);
    aboutMenu.add(about_OpenGitHubIssuesPageMenuItem);
    aboutMenu.add(about_OpenReleasesPageMenuItem);
    aboutMenu.add(new JSeparator());
    aboutMenu.add(about_OpenDiscordLinkMenuItem);
    aboutMenu.add(new JSeparator());
    aboutMenu.add(about_OpenDonationsPageMenuItem);

    // add menus
    MENUBAR.add(fileMenu);
    MENUBAR.add(editMenu);
    MENUBAR.add(viewMenu);
    MENUBAR.add(aboutMenu);

    return MENUBAR;
  }

  private void openAddonsLog(ActionEvent actionEvent) {
    LOG.debug("Clicked open Addons-log.");

    UTILITIES.FileUtils().openFile("logs/addons.log");
  }

  private void openModloaderInstallerLog(ActionEvent actionEvent) {
    LOG.debug("Clicked open Modloader-Installer-log.");

    UTILITIES.FileUtils().openFile("logs/modloader_installer.log");
  }

  private void openSPClog(ActionEvent actionEvent) {
    LOG.debug("Clicked open ServerPackCreator-log.");

    UTILITIES.FileUtils().openFile("logs/serverpackcreator.log");
  }

  private void checkForUpdates(ActionEvent actionEvent) {
    LOG.debug("Clicked Check for Updates");

    if (!displayUpdateDialog()) {
      JOptionPane.showMessageDialog(
          FRAME_SERVERPACKCREATOR,
          LOCALIZATIONMANAGER.getLocalizedString("menubar.gui.menuitem.updates.none"),
          LOCALIZATIONMANAGER.getLocalizedString("menubar.gui.menuitem.updates.none.title"),
          JOptionPane.INFORMATION_MESSAGE,
          UIManager.getIcon("OptionPane.informationIcon"));
    }
  }

  /**
   * If an initialize is available for ServerPackCreator, display a dialog asking the user whether
   *
   * @return {@link Boolean} <code>true</code> if an update was found and the dialog displayed.
   * @author Griefed
   */
  protected boolean displayUpdateDialog() {

    Optional<Update> update =
        UPDATECHECKER.checkForUpdate(
            APPLICATIONPROPERTIES.SERVERPACKCREATOR_VERSION(),
            APPLICATIONPROPERTIES.checkForAvailablePreReleases());

    if (update.isPresent()) {
      String textContent =
          String.format(
              LOCALIZATIONMANAGER.getLocalizedString("update.dialog.new"), update.get().url());

      StyledDocument styledDocument = new DefaultStyledDocument();
      SimpleAttributeSet simpleAttributeSet = new SimpleAttributeSet();
      MaterialTextPaneUI materialTextPaneUI = new MaterialTextPaneUI();
      JTextPane jTextPane = new JTextPane(styledDocument);
      StyleConstants.setBold(simpleAttributeSet, true);
      StyleConstants.setFontSize(simpleAttributeSet, 14);
      jTextPane.setCharacterAttributes(simpleAttributeSet, true);
      StyleConstants.setAlignment(simpleAttributeSet, StyleConstants.ALIGN_LEFT);
      styledDocument.setParagraphAttributes(
          0, styledDocument.getLength(), simpleAttributeSet, false);
      jTextPane.addHierarchyListener(
          e1 -> {
            Window window = SwingUtilities.getWindowAncestor(jTextPane);
            if (window instanceof Dialog) {
              Dialog dialog = (Dialog) window;
              if (!dialog.isResizable()) {
                dialog.setResizable(true);
              }
            }
          });
      jTextPane.setOpaque(false);
      jTextPane.setEditable(false);
      Clipboard clipboard = Toolkit.getDefaultToolkit().getSystemClipboard();
      String[] options = new String[3];

      options[0] = LOCALIZATIONMANAGER.getLocalizedString("update.dialog.yes");
      options[1] = LOCALIZATIONMANAGER.getLocalizedString("update.dialog.no");
      options[2] = LOCALIZATIONMANAGER.getLocalizedString("update.dialog.clipboard");

      try {
        styledDocument.insertString(0, textContent, simpleAttributeSet);
      } catch (BadLocationException ex) {
        LOG.error("Error inserting text into aboutDocument.", ex);
      }

      materialTextPaneUI.installUI(jTextPane);

      switch (JOptionPane.showOptionDialog(
          FRAME_SERVERPACKCREATOR,
          jTextPane,
          LOCALIZATIONMANAGER.getLocalizedString("update.dialog.available"),
          JOptionPane.DEFAULT_OPTION,
          JOptionPane.INFORMATION_MESSAGE,
          UIManager.getIcon("OptionPane.informationIcon"),
          options,
          options[0])) {
        case 0:
          try {
            if (Desktop.getDesktop().isSupported(Desktop.Action.BROWSE)) {
              Desktop.getDesktop().browse(update.get().url().toURI());
            }
          } catch (IOException | RuntimeException | URISyntaxException ex) {
            LOG.error("Error opening browser.", ex);
          }
          break;

        case 1:
          clipboard.setContents(new StringSelection(update.get().url().toString()), null);
          break;

        default:
          break;
      }
      return true;
    } else {
      return false;
    }
  }

  /**
   * Update the fallback clientside-only mods-list from the repositories.
   *
   * @param actionEvent The event which triggers this method.
   * @author Grefed
   */
  private void updateFallbackModslist(ActionEvent actionEvent) {
    LOG.debug("Running update check for fallback modslist...");
    if (APPLICATIONPROPERTIES.updateFallback()) {
      JOptionPane.showMessageDialog(
          FRAME_SERVERPACKCREATOR,
          LOCALIZATIONMANAGER.getLocalizedString("menubar.gui.menuitem.updatefallback.updated"),
          LOCALIZATIONMANAGER.getLocalizedString("menubar.gui.menuitem.updatefallback.title"),
          JOptionPane.INFORMATION_MESSAGE,
          UIManager.getIcon("OptionPane.informationIcon"));
    } else {
      JOptionPane.showMessageDialog(
          FRAME_SERVERPACKCREATOR,
          LOCALIZATIONMANAGER.getLocalizedString("menubar.gui.menuitem.updatefallback.nochange"),
          LOCALIZATIONMANAGER.getLocalizedString("menubar.gui.menuitem.updatefallback.title"),
          JOptionPane.INFORMATION_MESSAGE,
          UIManager.getIcon("OptionPane.informationIcon"));
    }
  }

  /**
   * Open the given url in a browser.
   *
   * @param uri {@link URI} the URI to the website you want to open.
   * @author Griefed
   */
  private void openLinkInBrowser(URI uri) {
    try {
      if (Desktop.getDesktop().isSupported(Desktop.Action.BROWSE)) {
        Desktop.getDesktop().browse(uri);
      }
    } catch (IOException ex) {
      LOG.error("Error opening browser with link " + uri + ".", ex);
    }
  }

  /**
   * Open the Help-section of the wiki in a browser.
   *
   * @param actionEvent The event which triggers this method.
   * @author Griefed
   */
  private void openWikiHelpMenuItem(ActionEvent actionEvent) {
    LOG.debug("Clicked Help.");

    openLinkInBrowser(
        URI.create(
            "https://wiki.griefed.de/en/Documentation/ServerPackCreator/ServerPackCreator-Help"));
  }

  /**
   * Open the HowTo-section of the wiki in a browser.
   *
   * @param actionEvent The event which triggers this method.
   * @author Griefed
   */
  private void openWikiHowToMenuItem(ActionEvent actionEvent) {
    LOG.debug("Clicked Getting started.");

    openLinkInBrowser(
        URI.create(
            "https://wiki.griefed.de/en/Documentation/ServerPackCreator/ServerPackCreator-HowTo"));
  }

  /**
   * Upon button-press, load default values for textfields so the user can start with a new
   * configuration. Just as if ServerPackCreator was started without a serverpackcreator.conf being
   * present.
   *
   * @param actionEvent The event which triggers this method.
   * @author Griefed
   */
  private void newConfiguration(ActionEvent actionEvent) {
    LOG.debug("Clearing GUI...");
    TAB_CREATESERVERPACK.clearInterface();
    lastLoadedConfigurationFile = null;
  }

  /**
   * Upon button-press, open the Discord invite-link to Griefed's Discord server in the users
   * default browser.
   *
   * @param actionEvent The event which triggers this method.
   * @author Griefed
   */
  private void openDiscordLinkMenuItem(ActionEvent actionEvent) {
    LOG.debug("Clicked Join Discord.");

    openLinkInBrowser(URI.create("https://discord.griefed.de"));
  }

  /**
   * Upon button-press, open ServerPackCreators issue-page on GitHub in the users default browser.
   *
   * @param actionEvent The event which triggers this method.
   * @author Griefed
   */
  private void openIssuesMenuItem(ActionEvent actionEvent) {
    LOG.debug("Clicked Open Issues page on GitHub.");

    openLinkInBrowser(URI.create("https://github.com/Griefed/ServerPackCreator/issues"));
  }

  /**
   * Upon button-press, uploads the serverpackcreator.log-file to HasteBin and display a dialog
   * asking the user whether they want to open the URL in their default browser or copy the link to
   * their clipboard. If the filesize exceeds 10 MB, a warning is displayed, telling the user about
   * filesize limitations of HasteBin.
   *
   * @param actionEvent The event which triggers this method.
   * @author Griefed
   */
  private void uploadServerPackCreatorLogToHasteBinMenuItem(ActionEvent actionEvent) {
    LOG.debug("Clicked Upload ServerPackCreator Log to HasteBin.");

    if (hasteBinPreChecks(new File("logs/serverpackcreator.log"))) {
      String urltoHasteBin = createHasteBinFromFile(new File("logs/serverpackcreator.log"));
      String textContent = String.format("URL: %s", urltoHasteBin);

      try {
        SPCLOG_DOCUMENT.insertString(0, textContent, SPCLOG_ATTRIBUTESET);
      } catch (BadLocationException ex) {
        LOG.error("Error inserting text into aboutDocument.", ex);
      }

      MATERIALTEXTPANEUI.installUI(SPCLOG_WINDOW_TEXTPANE);

      switch (JOptionPane.showOptionDialog(
          FRAME_SERVERPACKCREATOR,
          SPCLOG_WINDOW_TEXTPANE,
          LOCALIZATIONMANAGER.getLocalizedString("createserverpack.gui.about.hastebin.dialog"),
          JOptionPane.DEFAULT_OPTION,
          JOptionPane.INFORMATION_MESSAGE,
          ICON_HASTEBIN,
          HASTEOPTIONS,
          HASTEOPTIONS[0])) {
        case 0:
          openLinkInBrowser(URI.create(urltoHasteBin));

          break;

        case 1:
          CLIPBOARD.setContents(new StringSelection(urltoHasteBin), null);
          break;

        default:
          break;
      }
    } else {
      fileTooLargeDialog();
    }
  }

  /**
   * Upon button-press, uploads the serverpackcreator.conf-file to HasteBin and display a dialog
   * asking the user whether they want to open the URL in their default browser or copy the link to
   * their clipboard.
   *
   * @param actionEvent The event which triggers this method.
   * @author Griefed
   */
  private void uploadConfigurationToHasteBinMenuItem(ActionEvent actionEvent) {
    LOG.debug("Clicked Upload Configuration to HasteBin.");

    if (hasteBinPreChecks(new File("serverpackcreator.conf"))) {

      String urltoHasteBin = createHasteBinFromFile(new File("serverpackcreator.conf"));
      String textContent = String.format("URL: %s", urltoHasteBin);

      try {
        CONFIG_DOCUMENT.insertString(0, textContent, CONFIG_ATTRIBUTESET);
      } catch (BadLocationException ex) {
        LOG.error("Error inserting text into aboutDocument.", ex);
      }

      MATERIALTEXTPANEUI.installUI(CONFIG_WINDOW_TEXTPANE);

      switch (JOptionPane.showOptionDialog(
          FRAME_SERVERPACKCREATOR,
          CONFIG_WINDOW_TEXTPANE,
          LOCALIZATIONMANAGER.getLocalizedString("createserverpack.gui.about.hastebin.dialog"),
          JOptionPane.DEFAULT_OPTION,
          JOptionPane.INFORMATION_MESSAGE,
          ICON_HASTEBIN,
          HASTEOPTIONS,
          HASTEOPTIONS[0])) {
        case 0:
          openLinkInBrowser(URI.create(urltoHasteBin));

          break;

        case 1:
          CLIPBOARD.setContents(new StringSelection(urltoHasteBin), null);
          break;

        default:
          break;
      }
    } else {
      fileTooLargeDialog();
    }
  }

  /**
   * Opens a dialog informing the user that a file exceeds 10 MB in size.
   *
   * @author Griefed
   */
  private void fileTooLargeDialog() {
    MATERIALTEXTPANEUI.installUI(FILETOOLARGE_WINDOW_TEXTPANE);
    JOptionPane.showConfirmDialog(
        FRAME_SERVERPACKCREATOR,
        LOCALIZATIONMANAGER.getLocalizedString("menubar.gui.filetoolarge"),
        LOCALIZATIONMANAGER.getLocalizedString("menubar.gui.filetoolargetitle"),
        JOptionPane.DEFAULT_OPTION,
        JOptionPane.WARNING_MESSAGE,
        ICON_HASTEBIN);
  }

  /**
   * Upon button-press, open the server.properties-file, in the server-files directory, in the users
   * default text-editor.
   *
   * @param actionEvent The event which triggers this method.
   * @author Griefed
   */
  private void openInEditorServerProperties(ActionEvent actionEvent) {
    LOG.debug("Clicked Open server.properties in Editor.");

    if (new File(TAB_CREATESERVERPACK.getServerPropertiesPath()).isFile()) {
      UTILITIES.FileUtils().openFile(TAB_CREATESERVERPACK.getServerPropertiesPath());
    } else {
      UTILITIES.FileUtils().openFile("./server_files/server.properties");
    }
  }

  /**
   * Upon button-press, open the server-icon.png-file, in the server-files directory, in the users
   * default picture-viewer.
   *
   * @param actionEvent The event which triggers this method.
   * @author Griefed
   */
  private void openServerIcon(ActionEvent actionEvent) {
    LOG.debug("Clicked Open server-icon.png in Editor.");

    if (new File(TAB_CREATESERVERPACK.getServerIconPath()).isFile()) {
      UTILITIES.FileUtils().openFile(TAB_CREATESERVERPACK.getServerIconPath());
    } else {
      UTILITIES.FileUtils().openFile("./server_files/server-icon.png");
    }
  }

  /**
   * Upon button-press, close ServerPackCreator gracefully.
   *
   * @param actionEvent The event which triggers this method.
   * @author Griefed
   */
  private void exitMenuItem(ActionEvent actionEvent) {
    LOG.debug("Clicked Exit.");
    FRAME_SERVERPACKCREATOR.dispatchEvent(CLOSEEVENT);
  }

  /**
   * Upon button-press, open a Filechooser dialog which allows the user to specify a file in which
   * the current configuration in the GUI will be saved to.
   *
   * @param actionEvent The event which triggers this method.
   * @author Griefed
   */
  private void saveAsConfigToFileMenuItem(ActionEvent actionEvent) {
    LOG.debug("Clicked Save As...");

    configChooser = new JFileChooser();
    configChooser.setCurrentDirectory(new File("."));
    configChooser.setDialogTitle("Store current configuration");
    configChooser.setFileSelectionMode(JFileChooser.FILES_ONLY);
    configChooser.setFileFilter(
        new FileNameExtensionFilter(
            LOCALIZATIONMANAGER.getLocalizedString("createserverpack.gui.buttonloadconfig.filter"),
            "conf"));
    configChooser.setAcceptAllFileFilterUsed(false);
    configChooser.setMultiSelectionEnabled(false);
    configChooser.setPreferredSize(CHOOSERDIMENSION);

    if (configChooser.showOpenDialog(FRAME_SERVERPACKCREATOR) == JFileChooser.APPROVE_OPTION) {

      try {

        if (configChooser.getSelectedFile().getCanonicalPath().endsWith(".conf")) {

          TAB_CREATESERVERPACK.saveConfig(
              new File(configChooser.getSelectedFile().getCanonicalPath()));
          LOG.debug(
              "Saved configuration to: " + configChooser.getSelectedFile().getCanonicalPath());

        } else {

          TAB_CREATESERVERPACK.saveConfig(
              new File(configChooser.getSelectedFile().getCanonicalPath() + ".conf"));
          LOG.debug(
              "Saved configuration to: "
                  + configChooser.getSelectedFile().getCanonicalPath()
                  + ".conf");
        }

      } catch (IOException ex) {
        LOG.error(
            LOCALIZATIONMANAGER.getLocalizedString(
                "createserverpack.log.error.buttonloadconfigfromfile"),
            ex);
      }
    }
  }

  /**
   * Upon button-press, save the current configuration in the GUI to the serverpackcreator.conf-file
   * in ServerPackCreators base directory. if <code>
   * de.griefed.serverpackcreator.configuration.saveloadedconfig</code> is set to <code>true</code>
   * and the field <code>lastLoadedConfigurationFile</code> is not null, the last loaded
   * configuration-file is also saved to.
   *
   * @param actionEvent The event which triggers this method.
   * @author Griefed
   */
  private void saveConfigToFileMenuItem(ActionEvent actionEvent) {
    LOG.debug("Clicked Save.");
    LOG.debug("Saving serverpackcreator.conf");
    TAB_CREATESERVERPACK.saveConfig(new File("./serverpackcreator.conf"));

    if (lastLoadedConfigurationFile != null && APPLICATIONPROPERTIES.getSaveLoadedConfiguration()) {
      LOG.debug("Saving " + lastLoadedConfigurationFile.getName());
      TAB_CREATESERVERPACK.saveConfig(lastLoadedConfigurationFile);
    }
  }

  /**
   * Upon button-press, change the current theme to either light or dark-mode, depending on which
   * theme is currently active.
   *
   * @param actionEvent The event which triggers this method.
   * @author Griefed
   */
  private void switchThemeMenuItem(ActionEvent actionEvent) {
    LOG.debug("Clicked Toggle light/dark-mode.");

    if (!isDarkTheme) {
      try {
        UIManager.setLookAndFeel(LAF_DARK);
        MaterialLookAndFeel.changeTheme(DARKTHEME);

        isDarkTheme = true;

        try (OutputStream outputStream =
            Files.newOutputStream(APPLICATIONPROPERTIES.SERVERPACKCREATOR_PROPERTIES().toPath())) {

          APPLICATIONPROPERTIES.setProperty(
              "de.griefed.serverpackcreator.gui.darkmode", String.valueOf(true));
          APPLICATIONPROPERTIES.store(outputStream, null);

        } catch (IOException ex) {
          LOG.error("Couldn't write properties-file.", ex);
        }

      } catch (UnsupportedLookAndFeelException ex) {
        LOG.error("Couldn't change theme.", ex);
      }
    } else {
      try {
        UIManager.setLookAndFeel(LAF_LIGHT);
        MaterialLookAndFeel.changeTheme(LIGHTTHEME);

        isDarkTheme = false;

        try (OutputStream outputStream =
            Files.newOutputStream(APPLICATIONPROPERTIES.SERVERPACKCREATOR_PROPERTIES().toPath())) {

          APPLICATIONPROPERTIES.setProperty(
              "de.griefed.serverpackcreator.gui.darkmode", String.valueOf(false));
          APPLICATIONPROPERTIES.store(outputStream, null);

        } catch (IOException ex) {
          LOG.error("Couldn't write properties-file.", ex);
        }

      } catch (UnsupportedLookAndFeelException ex) {
        LOG.error("Couldn't change theme.", ex);
      }
    }

    SwingUtilities.updateComponentTreeUI(FRAME_SERVERPACKCREATOR);
    TABBEDPANE.setOpaque(true);
    TAB_CREATESERVERPACK.validateInputFields();
    TAB_CREATESERVERPACK.updatePanelTheme();
  }

  /**
   * Upon button-press, open a file-selector to load a serverpackcreator.conf-file into
   * ServerPackCreator.
   *
   * @param actionEvent The event which triggers this method.
   * @author Griefed
   */
  private void loadConfigurationFromFileMenuItem(ActionEvent actionEvent) {
    LOG.debug("Clicked load configuration from file.");

    configChooser = new JFileChooser();
    configChooser.setCurrentDirectory(new File("."));
    configChooser.setDialogTitle(
        LOCALIZATIONMANAGER.getLocalizedString("createserverpack.gui.buttonloadconfig.title"));
    configChooser.setFileSelectionMode(JFileChooser.FILES_ONLY);
    configChooser.setFileFilter(
        new FileNameExtensionFilter(
            LOCALIZATIONMANAGER.getLocalizedString("createserverpack.gui.buttonloadconfig.filter"),
            "conf"));
    configChooser.setAcceptAllFileFilterUsed(false);
    configChooser.setMultiSelectionEnabled(false);
    configChooser.setPreferredSize(CHOOSERDIMENSION);

    if (configChooser.showOpenDialog(FRAME_SERVERPACKCREATOR) == JFileChooser.APPROVE_OPTION) {

      try {

        /* This log is meant to be read by the user, therefore we allow translation. */
        LOG.info(
            String.format(
                LOCALIZATIONMANAGER.getLocalizedString(
                    "createserverpack.log.info.buttonloadconfigfromfile"),
                configChooser.getSelectedFile().getCanonicalPath()));

        File specifiedConfigFile;
        try {
          specifiedConfigFile =
              new File(UTILITIES.FileUtils().resolveLink(configChooser.getSelectedFile()));
        } catch (InvalidFileTypeException ex) {
          LOG.error("Could not resolve link/symlink. Using entry from user input for checks.", ex);
          specifiedConfigFile =
              new File(configChooser.getSelectedFile().getCanonicalPath().replace("\\", "/"));
        }

        TAB_CREATESERVERPACK.loadConfig(specifiedConfigFile);
        lastLoadedConfigurationFile = specifiedConfigFile;

      } catch (IOException ex) {
        LOG.error("Error loading configuration from selected file.", ex);
      }

      LOG.debug("Configuration successfully loaded.");
    }
  }

  /**
   * Upon button-press, open the folder containing installed plugins for ServerPackCreator in the
   * users file-explorer.
   *
   * @param actionEvent The event which triggers this method.
   * @author Griefed
   */
  private void openPluginsDirectoryMenuItem(ActionEvent actionEvent) {
    LOG.debug("Clicked open plugins directory.");
    UTILITIES.FileUtils().openFolder(APPLICATIONPROPERTIES.DIRECTORY_PLUGINS());
  }

  /**
   * Upon button-press, open the example plugins repository-page on GitHub in the users default
   * browser.
   *
   * @param actionEvent The event which triggers this method.
   * @author Griefed
   */
  private void viewExampleAddonMenuItem(ActionEvent actionEvent) {
    LOG.debug("Clicked view example addon");

    openLinkInBrowser(URI.create("https://github.com/Griefed/ServerPackCreatorExampleAddon"));
  }

  /**
   * Upon button-press, open the base directory of ServerPackCreator in the users file-explorer.
   *
   * @param actionEvent The event which triggers this method.
   * @author Griefed
   */
  private void openSPCDirectoryMenuItem(ActionEvent actionEvent) {
    LOG.debug("Clicked open installation directory.");
    UTILITIES.FileUtils().openFolder(".");
  }

  /**
   * Upon button-press, open the folder containing generated server packs in the users
   * file-explorer.
   *
   * @param actionEvent The event which triggers this method.
   * @author Griefed
   */
  private void openServerPacksDirectoryMenuItem(ActionEvent actionEvent) {
    LOG.debug("Clicked open server packs directory.");
    UTILITIES.FileUtils().openFolder(APPLICATIONPROPERTIES.getDirectoryServerPacks());
  }

  /**
   * Upon button-press, open the folder containing the server-icon.png and server.properties files
   * in the users file-explorer.
   *
   * @param actionEvent The event which triggers this method.
   * @author Griefed
   */
  private void openServerFilesDirectoryMenuItem(ActionEvent actionEvent) {
    LOG.debug("Clicked open server files directory.");
    UTILITIES.FileUtils().openFolder(APPLICATIONPROPERTIES.DIRECTORY_SERVER_FILES());
  }

  /**
   * Upon button-press, open an About-window containing information about ServerPackCreator.
   *
   * @param actionEvent The event which triggers this method.
   * @author Griefed
   */
  private void openAboutSPCMenuItem(ActionEvent actionEvent) {
    LOG.debug("Clicked open about window.");

    JScrollPane ABOUTWINDOWSCROLLPANE =
        new JScrollPane(
            ABOUT_WINDOW_TEXTPANE,
            JScrollPane.VERTICAL_SCROLLBAR_ALWAYS,
            JScrollPane.HORIZONTAL_SCROLLBAR_NEVER);
    ABOUTWINDOWSCROLLPANE.setMinimumSize(ABOUTDIMENSION);
    ABOUTWINDOWSCROLLPANE.setPreferredSize(ABOUTDIMENSION);
    ABOUTWINDOWSCROLLPANE.setMaximumSize(ABOUTDIMENSION);

    JOptionPane.showMessageDialog(
        FRAME_SERVERPACKCREATOR,
        ABOUTWINDOWSCROLLPANE,
        LOCALIZATIONMANAGER.getLocalizedString("createserverpack.gui.createserverpack.about.title"),
        JOptionPane.INFORMATION_MESSAGE,
        HELPICON);
  }

  /**
   * Upon button-press, open the ServerPackCreator repository GitHub page in the users
   * default-browser.
   *
   * @param actionEvent The event which triggers this method.
   * @author Griefed
   */
  private void openGitHubMenuItem(ActionEvent actionEvent) {
    LOG.debug("Clicked open GitHub repository link.");

    openLinkInBrowser(URI.create("https://github.com/Griefed/ServerPackCreator"));
  }

  /**
   * Upon button-press, open the GitHub Sponsors page in the users default-browser.
   *
   * @param actionEvent The event which triggers this method.
   * @author Griefed
   */
  private void openDonateMenuItem(ActionEvent actionEvent) {
    LOG.debug("Clicked open donations link.");

    openLinkInBrowser(URI.create("https://github.com/sponsors/Griefed"));
  }

  /**
   * Upon button-press, open the GitHub releases page in the users default-browser.
   *
   * @param actionEvent The event which triggers this method.
   * @author Griefed
   */
  private void openReleaseMenuItem(ActionEvent actionEvent) {
    LOG.debug("Clicked open releases link");

    openLinkInBrowser(URI.create("https://github.com/Griefed/ServerPackCreator/releases"));
  }

  /**
   * Checks the filesize of the given file whether it is smaller or bigger than 10 MB.
   *
   * @param fileToCheck The file or directory to check.
   * @return Boolean. True if the file is smaller, false if the file is bigger than 10 MB.
   * @author Griefed
   */
  private boolean hasteBinPreChecks(File fileToCheck) {
    long fileSize = FileUtils.sizeOf(fileToCheck);

    try {
      if (fileSize < 10000000
          && FileUtils.readFileToString(fileToCheck, StandardCharsets.UTF_8).length() < 400000) {
        LOG.debug("Smaller. " + fileSize + " byte.");
        return true;
      } else {
        LOG.debug("Bigger. " + fileSize + " byte.");
        return false;
      }
    } catch (IOException ex) {
      LOG.error("Couldn't read file: " + fileToCheck, ex);
    }

    return false;
  }

  /**
   * Create a HasteBin post from a given text file. The text file provided is read into a string and
   * then passed onto <a href="https://haste.zneix.eu">Haste zneix</a> which creates a HasteBin post
   * out of the passed String and returns the URL to the newly created post.<br>
   * Created with the help of <a href="https://github.com/kaimu-kun/hastebin.java">kaimu-kun's
   * hastebin.java (MIT License)</a> and edited to use HasteBin fork <a
   * href="https://github.com/zneix/haste-server">zneix/haste-server</a>. My fork of kaimu-kun's
   * hastebin.java is available at <a
   * href="https://github.com/Griefed/hastebin.java">Griefed/hastebin.java</a>.
   *
   * @param textFile The file which will be read into a String of which then to create a HasteBin
   *     post of.
   * @return String. Returns a String containing the URL to the newly created HasteBin post.
   * @author <a href="https://github.com/kaimu-kun">kaimu-kun/hastebin.java</a>
   * @author Griefed
   */
  private String createHasteBinFromFile(File textFile) {
    String text = null;
    String requestURL =
        APPLICATIONPROPERTIES.getProperty(
            "de.griefed.serverpackcreator.configuration.hastebinserver",
            "https://haste.zneix.eu/documents");

    String response = null;

    int postDataLength;

    URL url = null;

    HttpsURLConnection conn = null;

    byte[] postData;

    try {
      url = new URL(requestURL);
    } catch (IOException ex) {
      LOG.error("Error during acquisition of request URL.", ex);
    }

    try {
      text = FileUtils.readFileToString(textFile, "UTF-8");
    } catch (IOException ex) {
      LOG.error("Error reading text from file.", ex);
    }

    postData = Objects.requireNonNull(text).getBytes(StandardCharsets.UTF_8);
    postDataLength = postData.length;

    try {
      conn = (HttpsURLConnection) Objects.requireNonNull(url).openConnection();
    } catch (IOException ex) {
      LOG.error("Error during opening of connection to URL.", ex);
    }

    Objects.requireNonNull(conn).setDoOutput(true);
    conn.setInstanceFollowRedirects(false);

    try {
      conn.setRequestMethod("POST");
    } catch (ProtocolException ex) {
      LOG.error("Error during request of POST method.", ex);
    }

    conn.setRequestProperty("User-Agent", "HasteBin-Creator for ServerPackCreator");
    conn.setRequestProperty("Content-Length", Integer.toString(postDataLength));
    conn.setUseCaches(false);

    try (DataOutputStream dataOutputStream = new DataOutputStream(conn.getOutputStream())) {
      // dataOutputStream = new DataOutputStream(conn.getOutputStream());
      dataOutputStream.write(postData);

      try (BufferedReader bufferedReader =
          new BufferedReader(new InputStreamReader(conn.getInputStream()))) {

        response = bufferedReader.readLine();

      } catch (IOException ex) {
        LOG.error("Error encountered when acquiring HasteBin.", ex);
      }

    } catch (IOException ex) {
      LOG.error("Error encountered when acquiring HasteBin.", ex);
    }

    if (Objects.requireNonNull(response).contains("\"key\"")) {
      response =
          requestURL.replace("/documents", "/")
              + response.substring(response.indexOf(":") + 2, response.length() - 2);
    }

    if (response.contains(requestURL.replace("/documents", ""))) {
      return response;
    } else {
      return LOCALIZATIONMANAGER.getLocalizedString(
          "createserverpack.log.error.abouttab.hastebin.response");
    }
  }
}
