/**
 * Copyright (c) 2014 Denis Kuniss (http://www.grammarcraft.de).
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 */
package de.grammarcraft.xtend.flow.annotations;

import com.google.common.base.Objects;
import com.google.common.collect.Iterables;
import de.grammarcraft.xtend.flow.FunctionUnitWithOnlyOneInputPort;
import de.grammarcraft.xtend.flow.FunctionUnitWithOnlyOneOutputPort;
import de.grammarcraft.xtend.flow.IFunctionUnit;
import de.grammarcraft.xtend.flow.InputPort;
import de.grammarcraft.xtend.flow.OutputPort;
import de.grammarcraft.xtend.flow.annotations.FlowAnnotationSignature;
import de.grammarcraft.xtend.flow.data.None;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import org.eclipse.xtend.lib.macro.AbstractClassProcessor;
import org.eclipse.xtend.lib.macro.RegisterGlobalsContext;
import org.eclipse.xtend.lib.macro.TransformationContext;
import org.eclipse.xtend.lib.macro.declaration.AnnotationReference;
import org.eclipse.xtend.lib.macro.declaration.ClassDeclaration;
import org.eclipse.xtend.lib.macro.declaration.CompilationStrategy;
import org.eclipse.xtend.lib.macro.declaration.MutableClassDeclaration;
import org.eclipse.xtend.lib.macro.declaration.MutableConstructorDeclaration;
import org.eclipse.xtend.lib.macro.declaration.MutableFieldDeclaration;
import org.eclipse.xtend.lib.macro.declaration.MutableInterfaceDeclaration;
import org.eclipse.xtend.lib.macro.declaration.MutableMethodDeclaration;
import org.eclipse.xtend.lib.macro.declaration.Type;
import org.eclipse.xtend.lib.macro.declaration.TypeReference;
import org.eclipse.xtend.lib.macro.declaration.Visibility;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function0;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.InputOutput;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;

@SuppressWarnings("all")
public class FunctionUnitProcessor extends AbstractClassProcessor {
  private final static String defaultInputPortName = "start";
  
  @Override
  public void doRegisterGlobals(final ClassDeclaration annotatedClass, final RegisterGlobalsContext context) {
    final FlowAnnotationSignature flowAnnotation = new FlowAnnotationSignature(annotatedClass);
    boolean _and = false;
    Iterable<? extends AnnotationReference> _inputPortAnnotations = flowAnnotation.getInputPortAnnotations();
    Iterable<? extends AnnotationReference> _doubledAnnotations = FunctionUnitProcessor.doubledAnnotations(_inputPortAnnotations);
    boolean _isEmpty = IterableExtensions.isEmpty(_doubledAnnotations);
    if (!_isEmpty) {
      _and = false;
    } else {
      boolean _or = false;
      boolean _isFunctionUnit = flowAnnotation.isFunctionUnit();
      if (_isFunctionUnit) {
        _or = true;
      } else {
        boolean _isOperationUnit = flowAnnotation.isOperationUnit();
        _or = _isOperationUnit;
      }
      _and = _or;
    }
    if (_and) {
      Iterable<? extends AnnotationReference> _inputPortAnnotations_1 = flowAnnotation.getInputPortAnnotations();
      boolean _isEmpty_1 = IterableExtensions.isEmpty(_inputPortAnnotations_1);
      if (_isEmpty_1) {
        String _defaultInputPortInterfaceName = FunctionUnitProcessor.defaultInputPortInterfaceName(annotatedClass);
        context.registerInterface(_defaultInputPortInterfaceName);
      } else {
        Iterable<? extends AnnotationReference> _inputPortAnnotations_2 = flowAnnotation.getInputPortAnnotations();
        final Procedure1<AnnotationReference> _function = new Procedure1<AnnotationReference>() {
          @Override
          public void apply(final AnnotationReference inputPortAnnotation) {
            String _portName = FunctionUnitProcessor.portName(inputPortAnnotation);
            String _inputPortInterfaceName = FunctionUnitProcessor.getInputPortInterfaceName(annotatedClass, _portName);
            context.registerInterface(_inputPortInterfaceName);
          }
        };
        IterableExtensions.forEach(_inputPortAnnotations_2, _function);
      }
    }
  }
  
  private static String defaultInputPortInterfaceName(final ClassDeclaration annotatedClass) {
    return FunctionUnitProcessor.getInputPortInterfaceName(annotatedClass, FunctionUnitProcessor.defaultInputPortName);
  }
  
  private static String getInputPortInterfaceName(final ClassDeclaration annotatedClass, final String portName) {
    StringConcatenation _builder = new StringConcatenation();
    String _qualifiedName = annotatedClass.getQualifiedName();
    _builder.append(_qualifiedName, "");
    _builder.append("_InputPort_");
    _builder.append(portName, "");
    return _builder.toString();
  }
  
  private static String portName(final AnnotationReference annotationReference) {
    Object _value = annotationReference.getValue("name");
    return ((String) _value).trim();
  }
  
  private static TypeReference portType(final AnnotationReference annotationReference) {
    Object _value = annotationReference.getValue("type");
    return ((TypeReference) _value);
  }
  
  private static TypeReference[] portTypeParameters(final AnnotationReference annotationReference) {
    Object _value = annotationReference.getValue("typeArguments");
    return ((TypeReference[]) _value);
  }
  
  private static Iterable<? extends AnnotationReference> doubledAnnotations(final Iterable<? extends AnnotationReference> portAnnotations) {
    final Function1<AnnotationReference, Boolean> _function = new Function1<AnnotationReference, Boolean>() {
      @Override
      public Boolean apply(final AnnotationReference it) {
        return Boolean.valueOf(FunctionUnitProcessor.doubledAnnotation(it, portAnnotations));
      }
    };
    return IterableExtensions.filter(portAnnotations, _function);
  }
  
  private static boolean doubledAnnotation(final AnnotationReference portAnnotation, final Iterable<? extends AnnotationReference> portAnnotations) {
    final Function1<AnnotationReference, Boolean> _function = new Function1<AnnotationReference, Boolean>() {
      @Override
      public Boolean apply(final AnnotationReference it) {
        String _portName = FunctionUnitProcessor.portName(it);
        String _portName_1 = FunctionUnitProcessor.portName(portAnnotation);
        return Boolean.valueOf(Objects.equal(_portName, _portName_1));
      }
    };
    Iterable<? extends AnnotationReference> _filter = IterableExtensions.filter(portAnnotations, _function);
    int _size = IterableExtensions.size(_filter);
    return (_size > 1);
  }
  
  @Override
  public void doTransform(final MutableClassDeclaration annotatedClass, @Extension final TransformationContext context) {
    final FlowAnnotationSignature flowAnnotation = new FlowAnnotationSignature(annotatedClass);
    boolean hasContextErrors = false;
    boolean _checkForContextErrors = this.checkForContextErrors(context, annotatedClass, flowAnnotation);
    hasContextErrors = _checkForContextErrors;
    if (hasContextErrors) {
      context.addWarning(annotatedClass, "due to port annotation errors, no code can be generated");
      return;
    }
    this.checkForContextWarnings(context, annotatedClass, flowAnnotation);
    annotatedClass.setFinal(true);
    StringConcatenation _builder = new StringConcatenation();
    String _docComment = annotatedClass.getDocComment();
    _builder.append(_docComment, "");
    _builder.newLineIfNotEmpty();
    String _xifexpression = null;
    boolean _or = false;
    String _docComment_1 = annotatedClass.getDocComment();
    boolean _equals = Objects.equal(_docComment_1, null);
    if (_equals) {
      _or = true;
    } else {
      String _docComment_2 = annotatedClass.getDocComment();
      boolean _isEmpty = _docComment_2.isEmpty();
      _or = _isEmpty;
    }
    boolean _not = (!_or);
    if (_not) {
      _xifexpression = "<br><br>";
    }
    _builder.append(_xifexpression, "");
    _builder.newLineIfNotEmpty();
    _builder.append("Implements a function unit as defined by Flow Design paradigm.<br>");
    _builder.newLine();
    _builder.append("It consumes input messages over the input ports<br>");
    _builder.newLine();
    _builder.append("    ");
    Iterable<? extends AnnotationReference> _inputPortAnnotations = flowAnnotation.getInputPortAnnotations();
    final Function1<AnnotationReference, String> _function = new Function1<AnnotationReference, String>() {
      @Override
      public String apply(final AnnotationReference it) {
        StringConcatenation _builder = new StringConcatenation();
        _builder.append("\"");
        String _portName = FunctionUnitProcessor.portName(it);
        _builder.append(_portName, "");
        _builder.append("\" of type \"");
        TypeReference _portType = FunctionUnitProcessor.portType(it);
        _builder.append(_portType, "");
        _builder.append("\"");
        return _builder.toString();
      }
    };
    Iterable<String> _map = IterableExtensions.map(_inputPortAnnotations, _function);
    String _join = IterableExtensions.join(_map, "<br>\n");
    _builder.append(_join, "    ");
    _builder.append("<br>");
    _builder.newLineIfNotEmpty();
    _builder.append("And issues computation results over the output ports<br>");
    _builder.newLine();
    _builder.append("    ");
    Iterable<? extends AnnotationReference> _outputPortAnnotations = flowAnnotation.getOutputPortAnnotations();
    final Function1<AnnotationReference, String> _function_1 = new Function1<AnnotationReference, String>() {
      @Override
      public String apply(final AnnotationReference it) {
        StringConcatenation _builder = new StringConcatenation();
        _builder.append("\"");
        String _portName = FunctionUnitProcessor.portName(it);
        _builder.append(_portName, "");
        _builder.append("\" of type \"");
        TypeReference _portType = FunctionUnitProcessor.portType(it);
        _builder.append(_portType, "");
        _builder.append("\"");
        return _builder.toString();
      }
    };
    Iterable<String> _map_1 = IterableExtensions.map(_outputPortAnnotations, _function_1);
    String _join_1 = IterableExtensions.join(_map_1, "<br>\n");
    _builder.append(_join_1, "    ");
    _builder.append("<br>");
    _builder.newLineIfNotEmpty();
    annotatedClass.setDocComment(_builder.toString());
    FunctionUnitProcessor.addClassCommentToConstructors(annotatedClass, context);
    this.addInterfaces(annotatedClass, flowAnnotation, context);
    FunctionUnitProcessor.addNamingStuff(annotatedClass, context);
    this.addIntegrationErrorPort(annotatedClass, context);
    this.addInputPorts(annotatedClass, flowAnnotation, context);
    this.addOutputPorts(annotatedClass, flowAnnotation, context);
    this.addFlowOperators(annotatedClass, flowAnnotation, context);
  }
  
  /**
   * Adds the class documentation to all constructors after the user's given documentation.
   * This is done because in Xtend the types are typically not given when instantiating
   * a function unit. By promoting the class documentation to constructors they will
   * become visible by hovers in IDEs.
   */
  private static void addClassCommentToConstructors(final MutableClassDeclaration annotatedClass, @Extension final TransformationContext context) {
    Iterable<? extends MutableConstructorDeclaration> _declaredConstructors = annotatedClass.getDeclaredConstructors();
    final Procedure1<MutableConstructorDeclaration> _function = new Procedure1<MutableConstructorDeclaration>() {
      @Override
      public void apply(final MutableConstructorDeclaration it) {
        StringConcatenation _builder = new StringConcatenation();
        String _docComment = it.getDocComment();
        _builder.append(_docComment, "");
        _builder.newLineIfNotEmpty();
        String _xifexpression = null;
        boolean _or = false;
        String _docComment_1 = it.getDocComment();
        boolean _equals = Objects.equal(_docComment_1, null);
        if (_equals) {
          _or = true;
        } else {
          String _docComment_2 = it.getDocComment();
          boolean _isEmpty = _docComment_2.isEmpty();
          _or = _isEmpty;
        }
        boolean _not = (!_or);
        if (_not) {
          _xifexpression = "<br><br>";
        }
        _builder.append(_xifexpression, "");
        _builder.newLineIfNotEmpty();
        String _docComment_3 = annotatedClass.getDocComment();
        _builder.append(_docComment_3, "");
        _builder.newLineIfNotEmpty();
        it.setDocComment(_builder.toString());
      }
    };
    IterableExtensions.forEach(_declaredConstructors, _function);
  }
  
  /**
   * Adds "implements FunctionUnitWithOnlyOneInputPort<None>" if only no input port defined.
   * Adds "implements FunctionUnitWithOnlyOneInputPort<?>" if only one input port defined.
   * Adds "implements FunctionUnitWithOnlyOneOutputPort<?>" if only one output port defined
   */
  private void addInterfaces(final MutableClassDeclaration annotatedClass, final FlowAnnotationSignature flowAnnotation, @Extension final TransformationContext context) {
    final List<TypeReference> interfacesToBeAdded = new ArrayList<TypeReference>();
    TypeReference _newTypeReference = context.newTypeReference(IFunctionUnit.class);
    interfacesToBeAdded.add(_newTypeReference);
    Iterable<? extends AnnotationReference> _inputPortAnnotations = flowAnnotation.getInputPortAnnotations();
    boolean _isEmpty = IterableExtensions.isEmpty(_inputPortAnnotations);
    if (_isEmpty) {
      TypeReference _newTypeReference_1 = context.newTypeReference(None.class);
      TypeReference _newTypeReference_2 = context.newTypeReference(FunctionUnitWithOnlyOneInputPort.class, _newTypeReference_1);
      interfacesToBeAdded.add(_newTypeReference_2);
    } else {
      Iterable<? extends AnnotationReference> _inputPortAnnotations_1 = flowAnnotation.getInputPortAnnotations();
      int _size = IterableExtensions.size(_inputPortAnnotations_1);
      boolean _equals = (_size == 1);
      if (_equals) {
        Iterable<? extends AnnotationReference> _inputPortAnnotations_2 = flowAnnotation.getInputPortAnnotations();
        final AnnotationReference inputPortAnnotation = IterableExtensions.head(_inputPortAnnotations_2);
        TypeReference _portType = FunctionUnitProcessor.portType(inputPortAnnotation);
        Type _type = _portType.getType();
        TypeReference[] _portTypeParameters = FunctionUnitProcessor.portTypeParameters(inputPortAnnotation);
        TypeReference _newTypeReference_3 = context.newTypeReference(_type, _portTypeParameters);
        TypeReference _newTypeReference_4 = context.newTypeReference(FunctionUnitWithOnlyOneInputPort.class, _newTypeReference_3);
        interfacesToBeAdded.add(_newTypeReference_4);
      }
    }
    Iterable<? extends AnnotationReference> _outputPortAnnotations = flowAnnotation.getOutputPortAnnotations();
    int _size_1 = IterableExtensions.size(_outputPortAnnotations);
    boolean _equals_1 = (_size_1 == 1);
    if (_equals_1) {
      Iterable<? extends AnnotationReference> _outputPortAnnotations_1 = flowAnnotation.getOutputPortAnnotations();
      final AnnotationReference outputPortAnnotation = IterableExtensions.head(_outputPortAnnotations_1);
      TypeReference _portType_1 = FunctionUnitProcessor.portType(outputPortAnnotation);
      Type _type_1 = _portType_1.getType();
      TypeReference[] _portTypeParameters_1 = FunctionUnitProcessor.portTypeParameters(outputPortAnnotation);
      TypeReference _newTypeReference_5 = context.newTypeReference(_type_1, _portTypeParameters_1);
      TypeReference _newTypeReference_6 = context.newTypeReference(FunctionUnitWithOnlyOneOutputPort.class, _newTypeReference_5);
      interfacesToBeAdded.add(_newTypeReference_6);
    }
    annotatedClass.setImplementedInterfaces(interfacesToBeAdded);
  }
  
  private void checkForContextWarnings(@Extension final TransformationContext context, final MutableClassDeclaration annotatedClass, final FlowAnnotationSignature flowAnnotation) {
    Iterable<? extends AnnotationReference> _outputPortAnnotations = flowAnnotation.getOutputPortAnnotations();
    boolean _isEmpty = IterableExtensions.isEmpty(_outputPortAnnotations);
    if (_isEmpty) {
      context.addWarning(annotatedClass, "no output port defined");
    }
    boolean _and = false;
    boolean _or = false;
    boolean _isFunctionBoard = flowAnnotation.isFunctionBoard();
    if (_isFunctionBoard) {
      _or = true;
    } else {
      boolean _isIntegrationUnit = flowAnnotation.isIntegrationUnit();
      _or = _isIntegrationUnit;
    }
    if (!_or) {
      _and = false;
    } else {
      Iterable<? extends MutableMethodDeclaration> _declaredMethods = annotatedClass.getDeclaredMethods();
      final Function1<MutableMethodDeclaration, Boolean> _function = new Function1<MutableMethodDeclaration, Boolean>() {
        @Override
        public Boolean apply(final MutableMethodDeclaration it) {
          Visibility _visibility = it.getVisibility();
          return Boolean.valueOf((!Objects.equal(_visibility, Visibility.PRIVATE)));
        }
      };
      Iterable<? extends MutableMethodDeclaration> _filter = IterableExtensions.filter(_declaredMethods, _function);
      int _size = IterableExtensions.size(_filter);
      boolean _greaterThan = (_size > 0);
      _and = _greaterThan;
    }
    if (_and) {
      Iterable<? extends MutableMethodDeclaration> _declaredMethods_1 = annotatedClass.getDeclaredMethods();
      final Function1<MutableMethodDeclaration, Boolean> _function_1 = new Function1<MutableMethodDeclaration, Boolean>() {
        @Override
        public Boolean apply(final MutableMethodDeclaration it) {
          Visibility _visibility = it.getVisibility();
          return Boolean.valueOf((!Objects.equal(_visibility, Visibility.PRIVATE)));
        }
      };
      Iterable<? extends MutableMethodDeclaration> _filter_1 = IterableExtensions.filter(_declaredMethods_1, _function_1);
      final Procedure1<MutableMethodDeclaration> _function_2 = new Procedure1<MutableMethodDeclaration>() {
        @Override
        public void apply(final MutableMethodDeclaration it) {
          context.addWarning(it, "a FunctionBoard must not have other than private methods");
        }
      };
      IterableExtensions.forEach(_filter_1, _function_2);
    }
  }
  
  private boolean checkForContextErrors(@Extension final TransformationContext context, final MutableClassDeclaration annotatedClass, final FlowAnnotationSignature flowAnnotation) {
    boolean contextError = false;
    Object _inputPortAnnotationArgument = flowAnnotation.getInputPortAnnotationArgument();
    Class<?> _class = null;
    if (_inputPortAnnotationArgument!=null) {
      _class=_inputPortAnnotationArgument.getClass();
    }
    boolean _notEquals = (!Objects.equal(_class, AnnotationReference[].class));
    if (_notEquals) {
      context.addError(annotatedClass, "array of input port annotations expected");
      contextError = true;
    }
    Object _outputPortAnnotationArgument = flowAnnotation.getOutputPortAnnotationArgument();
    Class<?> _class_1 = null;
    if (_outputPortAnnotationArgument!=null) {
      _class_1=_outputPortAnnotationArgument.getClass();
    }
    boolean _notEquals_1 = (!Objects.equal(_class_1, AnnotationReference[].class));
    if (_notEquals_1) {
      context.addError(annotatedClass, "array of output port annotations expected");
      contextError = true;
    }
    if (contextError) {
      return true;
    }
    Iterable<? extends AnnotationReference> _inputPortAnnotations = flowAnnotation.getInputPortAnnotations();
    final Iterable<? extends AnnotationReference> doubledInputAnnotations = FunctionUnitProcessor.doubledAnnotations(_inputPortAnnotations);
    boolean _isEmpty = IterableExtensions.isEmpty(doubledInputAnnotations);
    boolean _not = (!_isEmpty);
    if (_not) {
      final Procedure1<AnnotationReference> _function = new Procedure1<AnnotationReference>() {
        @Override
        public void apply(final AnnotationReference it) {
          StringConcatenation _builder = new StringConcatenation();
          _builder.append("input port \"");
          String _portName = FunctionUnitProcessor.portName(it);
          _builder.append(_portName, "");
          _builder.append("\" is declared twice");
          context.addError(annotatedClass, _builder.toString());
        }
      };
      IterableExtensions.forEach(doubledInputAnnotations, _function);
      contextError = true;
    }
    Iterable<? extends AnnotationReference> _outputPortAnnotations = flowAnnotation.getOutputPortAnnotations();
    final Iterable<? extends AnnotationReference> doubledOutputAnnotations = FunctionUnitProcessor.doubledAnnotations(_outputPortAnnotations);
    boolean _isEmpty_1 = IterableExtensions.isEmpty(doubledOutputAnnotations);
    boolean _not_1 = (!_isEmpty_1);
    if (_not_1) {
      final Procedure1<AnnotationReference> _function_1 = new Procedure1<AnnotationReference>() {
        @Override
        public void apply(final AnnotationReference it) {
          StringConcatenation _builder = new StringConcatenation();
          _builder.append("output port \"");
          String _portName = FunctionUnitProcessor.portName(it);
          _builder.append(_portName, "");
          _builder.append("\" is declared twice");
          context.addError(annotatedClass, _builder.toString());
        }
      };
      IterableExtensions.forEach(doubledOutputAnnotations, _function_1);
      contextError = true;
    }
    boolean _and = false;
    AnnotationReference _unitModifier = flowAnnotation.getUnitModifier();
    boolean _equals = Objects.equal(_unitModifier, null);
    if (!_equals) {
      _and = false;
    } else {
      boolean _isFlowUnit = flowAnnotation.isFlowUnit();
      _and = _isFlowUnit;
    }
    if (_and) {
      context.addError(annotatedClass, "modifier @Operation or @Integration required");
    }
    return contextError;
  }
  
  /**
   * Adds function unit's naming field and overwriting of toString method, like
   * <pre>
   * private final static String _name = "MyFunctionUnit";
   * public final String toString() { return this._name; }
   * </pre>
   */
  private static void addNamingStuff(final MutableClassDeclaration annotatedClass, @Extension final TransformationContext context) {
    final Procedure1<MutableFieldDeclaration> _function = new Procedure1<MutableFieldDeclaration>() {
      @Override
      public void apply(final MutableFieldDeclaration it) {
        it.setVisibility(Visibility.PRIVATE);
        TypeReference _newTypeReference = context.newTypeReference(String.class);
        it.setType(_newTypeReference);
        final CompilationStrategy _function = new CompilationStrategy() {
          @Override
          public CharSequence compile(final CompilationStrategy.CompilationContext it) {
            StringConcatenation _builder = new StringConcatenation();
            _builder.append("\"");
            String _simpleName = annotatedClass.getSimpleName();
            _builder.append(_simpleName, "");
            _builder.append("\"");
            return _builder;
          }
        };
        it.setInitializer(_function);
      }
    };
    annotatedClass.addField("_name", _function);
    final Procedure1<MutableMethodDeclaration> _function_1 = new Procedure1<MutableMethodDeclaration>() {
      @Override
      public void apply(final MutableMethodDeclaration it) {
        it.setFinal(true);
        it.setVisibility(Visibility.PUBLIC);
        TypeReference _newTypeReference = context.newTypeReference(String.class);
        it.setReturnType(_newTypeReference);
        final CompilationStrategy _function = new CompilationStrategy() {
          @Override
          public CharSequence compile(final CompilationStrategy.CompilationContext it) {
            StringConcatenation _builder = new StringConcatenation();
            _builder.append("return this._name;");
            return _builder;
          }
        };
        it.setBody(_function);
      }
    };
    annotatedClass.addMethod("toString", _function_1);
    final Procedure1<MutableMethodDeclaration> _function_2 = new Procedure1<MutableMethodDeclaration>() {
      @Override
      public void apply(final MutableMethodDeclaration it) {
        it.setFinal(true);
        it.setVisibility(Visibility.PUBLIC);
        TypeReference _newTypeReference = context.newTypeReference(String.class);
        it.addParameter("newValue", _newTypeReference);
        final CompilationStrategy _function = new CompilationStrategy() {
          @Override
          public CharSequence compile(final CompilationStrategy.CompilationContext it) {
            StringConcatenation _builder = new StringConcatenation();
            _builder.append("this._name = newValue;");
            return _builder;
          }
        };
        it.setBody(_function);
      }
    };
    annotatedClass.addMethod("setName", _function_2);
  }
  
  /**
   * Adds methods and fields for implementing {@link IFunctionUnit}.
   */
  private MutableMethodDeclaration addIntegrationErrorPort(final MutableClassDeclaration annotatedClass, @Extension final TransformationContext context) {
    MutableMethodDeclaration _xblockexpression = null;
    {
      final String integrationErrorPortName = "integrationError";
      final Procedure1<MutableFieldDeclaration> _function = new Procedure1<MutableFieldDeclaration>() {
        @Override
        public void apply(final MutableFieldDeclaration it) {
          it.setFinal(true);
          it.setVisibility(Visibility.PRIVATE);
          TypeReference _newTypeReference = context.newTypeReference(Exception.class);
          TypeReference _newTypeReference_1 = context.newTypeReference(OutputPort.class, _newTypeReference);
          it.setType(_newTypeReference_1);
          final CompilationStrategy _function = new CompilationStrategy() {
            @Override
            public CharSequence compile(final CompilationStrategy.CompilationContext it) {
              StringConcatenation _builder = new StringConcatenation();
              _builder.append("new ");
              String _name = OutputPort.class.getName();
              _builder.append(_name, "");
              _builder.append("<");
              String _name_1 = Exception.class.getName();
              _builder.append(_name_1, "");
              _builder.append(">(\"integrationError\", ");
              _builder.newLineIfNotEmpty();
              _builder.append("    ");
              _builder.append("new org.eclipse.xtext.xbase.lib.Procedures.Procedure1<");
              String _name_2 = Exception.class.getName();
              _builder.append(_name_2, "    ");
              _builder.append(">() {");
              _builder.newLineIfNotEmpty();
              _builder.append("      ");
              _builder.append("@Override");
              _builder.newLine();
              _builder.append("      ");
              _builder.append("public void apply(final ");
              String _name_3 = Exception.class.getName();
              _builder.append(_name_3, "      ");
              _builder.append(" ex) {");
              _builder.newLineIfNotEmpty();
              _builder.append("        ");
              _builder.append("String _message = ex.getMessage();");
              _builder.newLine();
              _builder.append("        ");
              _builder.append("String _plus = (\"FATAL ERROR: \" + _message);");
              _builder.newLine();
              _builder.append("        ");
              String _name_4 = InputOutput.class.getName();
              _builder.append(_name_4, "        ");
              _builder.append(".<String>println(_plus);");
              _builder.newLineIfNotEmpty();
              _builder.append("      ");
              _builder.append("}");
              _builder.newLine();
              _builder.append("    ");
              _builder.append("}");
              _builder.newLine();
              _builder.append(")");
              _builder.newLine();
              return _builder;
            }
          };
          it.setInitializer(_function);
        }
      };
      annotatedClass.addField(integrationErrorPortName, _function);
      final Procedure1<MutableMethodDeclaration> _function_1 = new Procedure1<MutableMethodDeclaration>() {
        @Override
        public void apply(final MutableMethodDeclaration it) {
          it.setFinal(true);
          TypeReference _newTypeReference = context.newTypeReference(Exception.class);
          TypeReference _newTypeReference_1 = context.newTypeReference(OutputPort.class, _newTypeReference);
          it.setReturnType(_newTypeReference_1);
          final CompilationStrategy _function = new CompilationStrategy() {
            @Override
            public CharSequence compile(final CompilationStrategy.CompilationContext it) {
              StringConcatenation _builder = new StringConcatenation();
              _builder.append("return this.");
              _builder.append(integrationErrorPortName, "");
              _builder.append(";");
              return _builder;
            }
          };
          it.setBody(_function);
        }
      };
      annotatedClass.addMethod(integrationErrorPortName, _function_1);
      final Procedure1<MutableMethodDeclaration> _function_2 = new Procedure1<MutableMethodDeclaration>() {
        @Override
        public void apply(final MutableMethodDeclaration it) {
          it.setFinal(true);
          final String parameterName = "integrationException";
          TypeReference _newTypeReference = context.newTypeReference(Exception.class);
          it.addParameter(parameterName, _newTypeReference);
          final CompilationStrategy _function = new CompilationStrategy() {
            @Override
            public CharSequence compile(final CompilationStrategy.CompilationContext it) {
              StringConcatenation _builder = new StringConcatenation();
              _builder.append("this.");
              _builder.append(integrationErrorPortName, "");
              _builder.append(".operator_lessEqualsThan(");
              _builder.append(parameterName, "");
              _builder.append(");");
              return _builder;
            }
          };
          it.setBody(_function);
        }
      };
      _xblockexpression = annotatedClass.addMethod("forwardIntegrationError", _function_2);
    }
    return _xblockexpression;
  }
  
  private MutableMethodDeclaration addInputPorts(final MutableClassDeclaration annotatedClass, final FlowAnnotationSignature flowAnnotation, @Extension final TransformationContext context) {
    MutableMethodDeclaration _xifexpression = null;
    Iterable<? extends AnnotationReference> _inputPortAnnotations = flowAnnotation.getInputPortAnnotations();
    boolean _isEmpty = IterableExtensions.isEmpty(_inputPortAnnotations);
    if (_isEmpty) {
      MutableMethodDeclaration _xblockexpression = null;
      {
        final String portName = FunctionUnitProcessor.defaultInputPortName;
        final String inputPortInterfaceName = FunctionUnitProcessor.defaultInputPortInterfaceName(annotatedClass);
        final TypeReference msgType = context.newTypeReference(None.class);
        StringConcatenation _builder = new StringConcatenation();
        _builder.append("process$");
        _builder.append(portName, "");
        final String processInputMethodName = _builder.toString();
        boolean _or = false;
        boolean _isFunctionUnit = flowAnnotation.isFunctionUnit();
        if (_isFunctionUnit) {
          _or = true;
        } else {
          boolean _isOperationUnit = flowAnnotation.isOperationUnit();
          _or = _isOperationUnit;
        }
        final boolean isOperationUnit = _or;
        _xblockexpression = this.addInputPort(annotatedClass, portName, msgType, processInputMethodName, inputPortInterfaceName, isOperationUnit, context);
      }
      _xifexpression = _xblockexpression;
    } else {
      Iterable<? extends AnnotationReference> _inputPortAnnotations_1 = flowAnnotation.getInputPortAnnotations();
      final Procedure1<AnnotationReference> _function = new Procedure1<AnnotationReference>() {
        @Override
        public void apply(final AnnotationReference inputPortAnnotation) {
          final String portName = FunctionUnitProcessor.portName(inputPortAnnotation);
          final String inputPortInterfaceName = FunctionUnitProcessor.getInputPortInterfaceName(annotatedClass, portName);
          TypeReference _portType = FunctionUnitProcessor.portType(inputPortAnnotation);
          Type _type = _portType.getType();
          TypeReference[] _portTypeParameters = FunctionUnitProcessor.portTypeParameters(inputPortAnnotation);
          final TypeReference msgType = context.newTypeReference(_type, _portTypeParameters);
          StringConcatenation _builder = new StringConcatenation();
          _builder.append("process$");
          _builder.append(portName, "");
          final String processInputMethodName = _builder.toString();
          boolean _or = false;
          boolean _isFunctionUnit = flowAnnotation.isFunctionUnit();
          if (_isFunctionUnit) {
            _or = true;
          } else {
            boolean _isOperationUnit = flowAnnotation.isOperationUnit();
            _or = _isOperationUnit;
          }
          final boolean isOperationUnit = _or;
          FunctionUnitProcessor.this.addInputPort(annotatedClass, portName, msgType, processInputMethodName, inputPortInterfaceName, isOperationUnit, context);
        }
      };
      IterableExtensions.forEach(_inputPortAnnotations_1, _function);
    }
    return _xifexpression;
  }
  
  private MutableMethodDeclaration addInputPort(final MutableClassDeclaration annotatedClass, final String portName, final TypeReference msgType, final String processInputMethodName, final String inputPortInterfaceName, final boolean isOperationUnit, @Extension final TransformationContext context) {
    MutableMethodDeclaration _xblockexpression = null;
    {
      final Procedure1<MutableFieldDeclaration> _function = new Procedure1<MutableFieldDeclaration>() {
        @Override
        public void apply(final MutableFieldDeclaration it) {
          it.setFinal(true);
          it.setVisibility(Visibility.PRIVATE);
          TypeReference _newTypeReference = context.newTypeReference(InputPort.class, msgType);
          it.setType(_newTypeReference);
          final CompilationStrategy _function = new CompilationStrategy() {
            @Override
            public CharSequence compile(final CompilationStrategy.CompilationContext it) {
              StringConcatenation _builder = new StringConcatenation();
              _builder.append("new org.eclipse.xtext.xbase.lib.Functions.Function0<de.grammarcraft.xtend.flow.InputPort<");
              _builder.append(msgType, "");
              _builder.append(">>() {");
              _builder.newLineIfNotEmpty();
              _builder.append("    ");
              _builder.append("public de.grammarcraft.xtend.flow.InputPort<");
              _builder.append(msgType, "    ");
              _builder.append("> apply() {");
              _builder.newLineIfNotEmpty();
              _builder.append("      ");
              _builder.append("org.eclipse.xtend2.lib.StringConcatenation _builder = new org.eclipse.xtend2.lib.StringConcatenation();");
              _builder.newLine();
              _builder.append("      ");
              _builder.append("_builder.append(");
              String _simpleName = annotatedClass.getSimpleName();
              _builder.append(_simpleName, "      ");
              _builder.append(".this, \"\");");
              _builder.newLineIfNotEmpty();
              _builder.append("      ");
              _builder.append("_builder.append(\".");
              _builder.append(portName, "      ");
              _builder.append("\");");
              _builder.newLineIfNotEmpty();
              _builder.append("      ");
              _builder.append("final org.eclipse.xtext.xbase.lib.Procedures.Procedure1<Exception> _function_1 = new org.eclipse.xtext.xbase.lib.Procedures.Procedure1<Exception>() {");
              _builder.newLine();
              _builder.append("        ");
              _builder.append("public void apply(final Exception it) {");
              _builder.newLine();
              _builder.append("          ");
              String _simpleName_1 = annotatedClass.getSimpleName();
              _builder.append(_simpleName_1, "          ");
              _builder.append(".this.forwardIntegrationError(it);");
              _builder.newLineIfNotEmpty();
              _builder.append("        ");
              _builder.append("}");
              _builder.newLine();
              _builder.append("      ");
              _builder.append("};");
              _builder.newLine();
              {
                if (isOperationUnit) {
                  _builder.append("      ");
                  _builder.append("final org.eclipse.xtext.xbase.lib.Procedures.Procedure1<");
                  _builder.append(msgType, "      ");
                  _builder.append("> _function = new org.eclipse.xtext.xbase.lib.Procedures.Procedure1<");
                  _builder.append(msgType, "      ");
                  _builder.append(">() {");
                  _builder.newLineIfNotEmpty();
                  _builder.append("      ");
                  _builder.append("  ");
                  _builder.append("public void apply(final ");
                  _builder.append(msgType, "        ");
                  _builder.append(" msg) {");
                  _builder.newLineIfNotEmpty();
                  _builder.append("      ");
                  _builder.append("    ");
                  String _simpleName_2 = annotatedClass.getSimpleName();
                  _builder.append(_simpleName_2, "          ");
                  _builder.append(".this.");
                  _builder.append(processInputMethodName, "          ");
                  _builder.append("(msg);");
                  _builder.newLineIfNotEmpty();
                  _builder.append("      ");
                  _builder.append("  ");
                  _builder.append("}");
                  _builder.newLine();
                  _builder.append("      ");
                  _builder.append("};");
                  _builder.newLine();
                  _builder.append("      ");
                  _builder.append("de.grammarcraft.xtend.flow.InputPort<");
                  _builder.append(msgType, "      ");
                  _builder.append("> _inputPort = new de.grammarcraft.xtend.flow.InputPort<");
                  _builder.append(msgType, "      ");
                  _builder.append(">(_builder.toString(), _function, _function_1);");
                  _builder.newLineIfNotEmpty();
                } else {
                  _builder.append("      ");
                  _builder.append("de.grammarcraft.xtend.flow.InputPort<");
                  _builder.append(msgType, "      ");
                  _builder.append("> _inputPort = new de.grammarcraft.xtend.flow.InputPort<");
                  _builder.append(msgType, "      ");
                  _builder.append(">(_builder.toString(), _function_1);");
                  _builder.newLineIfNotEmpty();
                }
              }
              _builder.append("      ");
              _builder.append("return _inputPort;");
              _builder.newLine();
              _builder.append("    ");
              _builder.append("}");
              _builder.newLine();
              _builder.append("}.apply();");
              _builder.newLine();
              return _builder;
            }
          };
          it.setInitializer(_function);
        }
      };
      annotatedClass.addField(portName, _function);
      final Procedure1<MutableMethodDeclaration> _function_1 = new Procedure1<MutableMethodDeclaration>() {
        @Override
        public void apply(final MutableMethodDeclaration it) {
          it.setFinal(true);
          TypeReference _newTypeReference = context.newTypeReference(InputPort.class, msgType);
          it.setReturnType(_newTypeReference);
          StringConcatenation _builder = new StringConcatenation();
          _builder.append("Input port \'");
          _builder.append(portName, "");
          _builder.append("\' of function unit \'");
          String _simpleName = annotatedClass.getSimpleName();
          _builder.append(_simpleName, "");
          _builder.append("\', receives messages of ");
          _builder.newLineIfNotEmpty();
          _builder.append("type \'");
          _builder.append(msgType, "");
          _builder.append("\' for further processing by this function unit.");
          _builder.newLineIfNotEmpty();
          it.setDocComment(_builder.toString());
          final CompilationStrategy _function = new CompilationStrategy() {
            @Override
            public CharSequence compile(final CompilationStrategy.CompilationContext it) {
              StringConcatenation _builder = new StringConcatenation();
              _builder.append("return this.");
              _builder.append(portName, "");
              _builder.append(";");
              _builder.newLineIfNotEmpty();
              return _builder;
            }
          };
          it.setBody(_function);
        }
      };
      annotatedClass.addMethod(portName, _function_1);
      MutableMethodDeclaration _xifexpression = null;
      if (isOperationUnit) {
        MutableMethodDeclaration _xblockexpression_1 = null;
        {
          final MutableInterfaceDeclaration interfaceType = context.findInterface(inputPortInterfaceName);
          Iterable<? extends TypeReference> _implementedInterfaces = annotatedClass.getImplementedInterfaces();
          TypeReference _newTypeReference = context.newTypeReference(interfaceType);
          Iterable<TypeReference> _plus = Iterables.<TypeReference>concat(_implementedInterfaces, Collections.<TypeReference>unmodifiableList(CollectionLiterals.<TypeReference>newArrayList(_newTypeReference)));
          annotatedClass.setImplementedInterfaces(_plus);
          final Procedure1<MutableMethodDeclaration> _function_2 = new Procedure1<MutableMethodDeclaration>() {
            @Override
            public void apply(final MutableMethodDeclaration it) {
              StringConcatenation _builder = new StringConcatenation();
              _builder.append("Implement this method to process input arriving via input port \"");
              _builder.append(portName, "");
              _builder.append("\".");
              _builder.newLineIfNotEmpty();
              _builder.append("Message coming in has type \"");
              _builder.append(msgType, "");
              _builder.append("\".");
              _builder.newLineIfNotEmpty();
              it.setDocComment(_builder.toString());
              it.addParameter("msg", msgType);
            }
          };
          _xblockexpression_1 = interfaceType.addMethod(processInputMethodName, _function_2);
        }
        _xifexpression = _xblockexpression_1;
      }
      _xblockexpression = _xifexpression;
    }
    return _xblockexpression;
  }
  
  private void addOutputPorts(final MutableClassDeclaration annotatedClass, final FlowAnnotationSignature flowAnnotation, @Extension final TransformationContext context) {
    Iterable<? extends AnnotationReference> _outputPortAnnotations = flowAnnotation.getOutputPortAnnotations();
    final Procedure1<AnnotationReference> _function = new Procedure1<AnnotationReference>() {
      @Override
      public void apply(final AnnotationReference outputPortAnnotation) {
        final String portName = FunctionUnitProcessor.portName(outputPortAnnotation);
        TypeReference _portType = FunctionUnitProcessor.portType(outputPortAnnotation);
        Type _type = _portType.getType();
        TypeReference[] _portTypeParameters = FunctionUnitProcessor.portTypeParameters(outputPortAnnotation);
        final TypeReference msgType = context.newTypeReference(_type, _portTypeParameters);
        final Procedure1<MutableFieldDeclaration> _function = new Procedure1<MutableFieldDeclaration>() {
          @Override
          public void apply(final MutableFieldDeclaration it) {
            it.setFinal(true);
            it.setVisibility(Visibility.PRIVATE);
            TypeReference _newTypeReference = context.newTypeReference(OutputPort.class, msgType);
            it.setType(_newTypeReference);
            final CompilationStrategy _function = new CompilationStrategy() {
              @Override
              public CharSequence compile(final CompilationStrategy.CompilationContext it) {
                StringConcatenation _builder = new StringConcatenation();
                _builder.append("new org.eclipse.xtext.xbase.lib.Functions.Function0<de.grammarcraft.xtend.flow.OutputPort<");
                _builder.append(msgType, "");
                _builder.append(">>() {");
                _builder.newLineIfNotEmpty();
                _builder.append("    ");
                _builder.append("public de.grammarcraft.xtend.flow.OutputPort<");
                _builder.append(msgType, "    ");
                _builder.append("> apply() {");
                _builder.newLineIfNotEmpty();
                _builder.append("        ");
                _builder.append("org.eclipse.xtend2.lib.StringConcatenation _builder = new org.eclipse.xtend2.lib.StringConcatenation();");
                _builder.newLine();
                _builder.append("        ");
                _builder.append("_builder.append(");
                String _simpleName = annotatedClass.getSimpleName();
                _builder.append(_simpleName, "        ");
                _builder.append(".this, \"\");");
                _builder.newLineIfNotEmpty();
                _builder.append("        ");
                _builder.append("_builder.append(\".");
                _builder.append(portName, "        ");
                _builder.append("\");");
                _builder.newLineIfNotEmpty();
                _builder.append("        ");
                _builder.append("final org.eclipse.xtext.xbase.lib.Procedures.Procedure1<Exception> _function = new org.eclipse.xtext.xbase.lib.Procedures.Procedure1<Exception>() {");
                _builder.newLine();
                _builder.append("          ");
                _builder.append("public void apply(final Exception it) {");
                _builder.newLine();
                _builder.append("            ");
                String _simpleName_1 = annotatedClass.getSimpleName();
                _builder.append(_simpleName_1, "            ");
                _builder.append(".this.forwardIntegrationError(it);");
                _builder.newLineIfNotEmpty();
                _builder.append("          ");
                _builder.append("}");
                _builder.newLine();
                _builder.append("        ");
                _builder.append("};");
                _builder.newLine();
                _builder.append("        ");
                _builder.append("de.grammarcraft.xtend.flow.OutputPort<");
                _builder.append(msgType, "        ");
                _builder.append("> _outputPort = new de.grammarcraft.xtend.flow.OutputPort<");
                _builder.append(msgType, "        ");
                _builder.append(">(_builder.toString(), _function);");
                _builder.newLineIfNotEmpty();
                _builder.append("        ");
                _builder.append("return _outputPort;");
                _builder.newLine();
                _builder.append("    ");
                _builder.append("}");
                _builder.newLine();
                _builder.append("}.apply();");
                _builder.newLine();
                return _builder;
              }
            };
            it.setInitializer(_function);
          }
        };
        annotatedClass.addField(portName, _function);
        final Procedure1<MutableMethodDeclaration> _function_1 = new Procedure1<MutableMethodDeclaration>() {
          @Override
          public void apply(final MutableMethodDeclaration it) {
            it.setFinal(true);
            TypeReference _newTypeReference = context.newTypeReference(OutputPort.class, msgType);
            it.setReturnType(_newTypeReference);
            StringConcatenation _builder = new StringConcatenation();
            _builder.append("Output port \'");
            _builder.append(portName, "");
            _builder.append("\' of function unit \'");
            String _simpleName = annotatedClass.getSimpleName();
            _builder.append(_simpleName, "");
            _builder.append("\', issues messages of ");
            _builder.newLineIfNotEmpty();
            _builder.append("type \'");
            _builder.append(msgType, "");
            _builder.append("\' as computation result of this function unit.");
            _builder.newLineIfNotEmpty();
            it.setDocComment(_builder.toString());
            final CompilationStrategy _function = new CompilationStrategy() {
              @Override
              public CharSequence compile(final CompilationStrategy.CompilationContext it) {
                StringConcatenation _builder = new StringConcatenation();
                _builder.append("return this.");
                _builder.append(portName, "");
                _builder.append(";");
                _builder.newLineIfNotEmpty();
                return _builder;
              }
            };
            it.setBody(_function);
          }
        };
        annotatedClass.addMethod(portName, _function_1);
      }
    };
    IterableExtensions.forEach(_outputPortAnnotations, _function);
  }
  
  private MutableMethodDeclaration addFlowOperators(final MutableClassDeclaration annotatedClass, final FlowAnnotationSignature flowAnnotation, @Extension final TransformationContext context) {
    MutableMethodDeclaration _xblockexpression = null;
    {
      Iterable<? extends AnnotationReference> _inputPortAnnotations = flowAnnotation.getInputPortAnnotations();
      boolean _isEmpty = IterableExtensions.isEmpty(_inputPortAnnotations);
      if (_isEmpty) {
        final String portName = FunctionUnitProcessor.defaultInputPortName;
        final TypeReference msgType = context.newTypeReference(None.class);
        this.addInputPortFlowOperators(annotatedClass, portName, msgType, context);
      } else {
        Iterable<? extends AnnotationReference> _inputPortAnnotations_1 = flowAnnotation.getInputPortAnnotations();
        int _size = IterableExtensions.size(_inputPortAnnotations_1);
        boolean _equals = (_size == 1);
        if (_equals) {
          Iterable<? extends AnnotationReference> _inputPortAnnotations_2 = flowAnnotation.getInputPortAnnotations();
          AnnotationReference _head = IterableExtensions.head(_inputPortAnnotations_2);
          final String portName_1 = FunctionUnitProcessor.portName(_head);
          Iterable<? extends AnnotationReference> _inputPortAnnotations_3 = flowAnnotation.getInputPortAnnotations();
          AnnotationReference _head_1 = IterableExtensions.head(_inputPortAnnotations_3);
          TypeReference _portType = FunctionUnitProcessor.portType(_head_1);
          Type _type = _portType.getType();
          Iterable<? extends AnnotationReference> _inputPortAnnotations_4 = flowAnnotation.getInputPortAnnotations();
          AnnotationReference _head_2 = IterableExtensions.head(_inputPortAnnotations_4);
          TypeReference[] _portTypeParameters = FunctionUnitProcessor.portTypeParameters(_head_2);
          final TypeReference msgType_1 = context.newTypeReference(_type, _portTypeParameters);
          this.addInputPortFlowOperators(annotatedClass, portName_1, msgType_1, context);
        }
      }
      MutableMethodDeclaration _xifexpression = null;
      Iterable<? extends AnnotationReference> _outputPortAnnotations = flowAnnotation.getOutputPortAnnotations();
      int _size_1 = IterableExtensions.size(_outputPortAnnotations);
      boolean _equals_1 = (_size_1 == 1);
      if (_equals_1) {
        MutableMethodDeclaration _xblockexpression_1 = null;
        {
          Iterable<? extends AnnotationReference> _outputPortAnnotations_1 = flowAnnotation.getOutputPortAnnotations();
          AnnotationReference _head_3 = IterableExtensions.head(_outputPortAnnotations_1);
          final String portName_2 = FunctionUnitProcessor.portName(_head_3);
          Iterable<? extends AnnotationReference> _outputPortAnnotations_2 = flowAnnotation.getOutputPortAnnotations();
          AnnotationReference _head_4 = IterableExtensions.head(_outputPortAnnotations_2);
          TypeReference _portType_1 = FunctionUnitProcessor.portType(_head_4);
          Type _type_1 = _portType_1.getType();
          Iterable<? extends AnnotationReference> _outputPortAnnotations_3 = flowAnnotation.getOutputPortAnnotations();
          AnnotationReference _head_5 = IterableExtensions.head(_outputPortAnnotations_3);
          TypeReference[] _portTypeParameters_1 = FunctionUnitProcessor.portTypeParameters(_head_5);
          final TypeReference msgType_2 = context.newTypeReference(_type_1, _portTypeParameters_1);
          _xblockexpression_1 = this.addOutputPortFlowOperators(annotatedClass, portName_2, msgType_2, context);
        }
        _xifexpression = _xblockexpression_1;
      }
      _xblockexpression = _xifexpression;
    }
    return _xblockexpression;
  }
  
  private MutableMethodDeclaration addInputPortFlowOperators(final MutableClassDeclaration annotatedClass, final String portName, final TypeReference msgType, @Extension final TransformationContext context) {
    MutableMethodDeclaration _xblockexpression = null;
    {
      final Procedure1<MutableMethodDeclaration> _function = new Procedure1<MutableMethodDeclaration>() {
        @Override
        public void apply(final MutableMethodDeclaration it) {
          it.setFinal(true);
          TypeReference _newTypeReference = context.newTypeReference(InputPort.class, msgType);
          it.setReturnType(_newTypeReference);
          final CompilationStrategy _function = new CompilationStrategy() {
            @Override
            public CharSequence compile(final CompilationStrategy.CompilationContext it) {
              StringConcatenation _builder = new StringConcatenation();
              _builder.append("return this.");
              _builder.append(portName, "");
              _builder.append(";");
              _builder.newLineIfNotEmpty();
              return _builder;
            }
          };
          it.setBody(_function);
        }
      };
      annotatedClass.addMethod("theOneAndOnlyInputPort", _function);
      final Procedure1<MutableMethodDeclaration> _function_1 = new Procedure1<MutableMethodDeclaration>() {
        @Override
        public void apply(final MutableMethodDeclaration it) {
          it.setFinal(true);
          StringConcatenation _builder = new StringConcatenation();
          _builder.append("Flow DSL operator \"&lt;=\" for forwarding a message of type \'");
          String _name = msgType.getName();
          _builder.append(_name, "");
          _builder.append("\' value to ");
          _builder.newLineIfNotEmpty();
          _builder.append("the one and only input port \'");
          _builder.append(portName, "");
          _builder.append("\' for being processed.<br>");
          _builder.newLineIfNotEmpty();
          _builder.append("example:<pre> ");
          _builder.newLine();
          _builder.append("  ");
          _builder.append("input <= \"some string\"");
          _builder.newLine();
          _builder.append("</pre>");
          _builder.newLine();
          it.setDocComment(_builder.toString());
          final String parameterVarName = "msg";
          it.addParameter(parameterVarName, msgType);
          final CompilationStrategy _function = new CompilationStrategy() {
            @Override
            public CharSequence compile(final CompilationStrategy.CompilationContext it) {
              StringConcatenation _builder = new StringConcatenation();
              _builder.append("this.");
              _builder.append(portName, "");
              _builder.append(".operator_lessEqualsThan(");
              _builder.append(parameterVarName, "");
              _builder.append(");");
              _builder.newLineIfNotEmpty();
              return _builder;
            }
          };
          it.setBody(_function);
        }
      };
      annotatedClass.addMethod("operator_lessEqualsThan", _function_1);
      final Procedure1<MutableMethodDeclaration> _function_2 = new Procedure1<MutableMethodDeclaration>() {
        @Override
        public void apply(final MutableMethodDeclaration it) {
          it.setFinal(true);
          StringConcatenation _builder = new StringConcatenation();
          _builder.append("Flow DSL operator \"&lt;=\" for forwarding a message value of type \'");
          String _name = msgType.getName();
          _builder.append(_name, "");
          _builder.append("\' to ");
          _builder.newLineIfNotEmpty();
          _builder.append("the one and only input port \'");
          _builder.append(portName, "");
          _builder.append("\' for being processed.<br>");
          _builder.newLineIfNotEmpty();
          _builder.append("For computing the message to be forwarded the right side closure is applied.");
          _builder.newLine();
          _builder.append("example:<pre> ");
          _builder.newLine();
          _builder.append("  ");
          _builder.append("input &lt;= [ if (state&gt;0) \"some string\" else \"some other string\"");
          _builder.newLine();
          _builder.append("</pre>");
          _builder.newLine();
          it.setDocComment(_builder.toString());
          final String msgClosureVarName = "msgClosure";
          TypeReference _newWildcardTypeReference = context.newWildcardTypeReference(msgType);
          TypeReference _newTypeReference = context.newTypeReference(Function0.class, _newWildcardTypeReference);
          it.addParameter(msgClosureVarName, _newTypeReference);
          final CompilationStrategy _function = new CompilationStrategy() {
            @Override
            public CharSequence compile(final CompilationStrategy.CompilationContext it) {
              StringConcatenation _builder = new StringConcatenation();
              _builder.append(msgType, "");
              _builder.append(" _apply = ");
              _builder.append(msgClosureVarName, "");
              _builder.append(".apply();");
              _builder.newLineIfNotEmpty();
              _builder.append("this.");
              _builder.append(portName, "");
              _builder.append(".operator_lessEqualsThan(_apply);");
              _builder.newLineIfNotEmpty();
              return _builder;
            }
          };
          it.setBody(_function);
        }
      };
      _xblockexpression = annotatedClass.addMethod("operator_lessEqualsThan", _function_2);
    }
    return _xblockexpression;
  }
  
  private MutableMethodDeclaration addOutputPortFlowOperators(final MutableClassDeclaration annotatedClass, final String portName, final TypeReference msgType, @Extension final TransformationContext context) {
    MutableMethodDeclaration _xblockexpression = null;
    {
      final Procedure1<MutableMethodDeclaration> _function = new Procedure1<MutableMethodDeclaration>() {
        @Override
        public void apply(final MutableMethodDeclaration it) {
          it.setFinal(true);
          StringConcatenation _builder = new StringConcatenation();
          _builder.append("Flow DSL operator \"-&gt;\" for letting all message issued by the one and only output port \'");
          _builder.append(portName, "");
          _builder.append("\' being ");
          _builder.newLineIfNotEmpty();
          _builder.append("processed by the right side closure.");
          _builder.newLine();
          _builder.append("Typically this is used to process the message for a side effect like printing on standard out. ");
          _builder.newLine();
          _builder.append("example:<pre>");
          _builder.newLine();
          _builder.append("  ");
          _builder.append("fu -&gt; [msg|println(\"message received: \" + msg\")]");
          _builder.newLine();
          _builder.append("</pre>");
          _builder.newLine();
          it.setDocComment(_builder.toString());
          final String msgProessingClosureVarName = "msgProcessingClosure";
          TypeReference _newWildcardTypeReferenceWithLowerBound = context.newWildcardTypeReferenceWithLowerBound(msgType);
          final TypeReference operationType = context.newTypeReference(Procedure1.class, _newWildcardTypeReferenceWithLowerBound);
          it.addParameter(msgProessingClosureVarName, operationType);
          final CompilationStrategy _function = new CompilationStrategy() {
            @Override
            public CharSequence compile(final CompilationStrategy.CompilationContext it) {
              StringConcatenation _builder = new StringConcatenation();
              _builder.append("this.");
              _builder.append(portName, "");
              _builder.append(".operator_mappedTo(");
              _builder.append(msgProessingClosureVarName, "");
              _builder.append(");");
              return _builder;
            }
          };
          it.setBody(_function);
        }
      };
      annotatedClass.addMethod("operator_mappedTo", _function);
      final Procedure1<MutableMethodDeclaration> _function_1 = new Procedure1<MutableMethodDeclaration>() {
        @Override
        public void apply(final MutableMethodDeclaration it) {
          it.setFinal(true);
          StringConcatenation _builder = new StringConcatenation();
          _builder.append("Flow DSL operator \"-&gt;\" for connecting two function units. ");
          _builder.newLine();
          _builder.append("Connects the left one function unit\'s one and only one output port \'");
          _builder.append(portName, "");
          _builder.append("\' with ");
          _builder.newLineIfNotEmpty();
          _builder.append("the right side funtion unit which has one and only one input port.<br>");
          _builder.newLine();
          _builder.append("example:<pre>");
          _builder.newLine();
          _builder.append("  ");
          _builder.append("fu -&gt; fu\'");
          _builder.newLine();
          _builder.append("</pre>");
          _builder.newLine();
          it.setDocComment(_builder.toString());
          final String rightSideFunctionUnitVarName = "rightSideFunctionUnit";
          TypeReference _newTypeReference = context.newTypeReference(FunctionUnitWithOnlyOneInputPort.class, msgType);
          it.addParameter(rightSideFunctionUnitVarName, _newTypeReference);
          final CompilationStrategy _function = new CompilationStrategy() {
            @Override
            public CharSequence compile(final CompilationStrategy.CompilationContext it) {
              StringConcatenation _builder = new StringConcatenation();
              _builder.append("de.grammarcraft.xtend.flow.InputPort<");
              _builder.append(msgType, "");
              _builder.append("> _theOneAndOnlyInputPort = ");
              _builder.append(rightSideFunctionUnitVarName, "");
              _builder.append(".theOneAndOnlyInputPort();");
              _builder.newLineIfNotEmpty();
              _builder.append("org.eclipse.xtext.xbase.lib.Procedures.Procedure1<? super ");
              _builder.append(msgType, "");
              _builder.append("> _inputProcessingOperation = _theOneAndOnlyInputPort.inputProcessingOperation();");
              _builder.newLineIfNotEmpty();
              _builder.append("this.");
              _builder.append(portName, "");
              _builder.append(".operator_mappedTo(_inputProcessingOperation);");
              _builder.newLineIfNotEmpty();
              return _builder;
            }
          };
          it.setBody(_function);
        }
      };
      annotatedClass.addMethod("operator_mappedTo", _function_1);
      final Procedure1<MutableMethodDeclaration> _function_2 = new Procedure1<MutableMethodDeclaration>() {
        @Override
        public void apply(final MutableMethodDeclaration it) {
          it.setFinal(true);
          StringConcatenation _builder = new StringConcatenation();
          _builder.append("Flow DSL operator \"-&gt;\" for connecting two function units. ");
          _builder.newLine();
          _builder.append("Connects the left one function unit\'s one and only one output port \'");
          _builder.append(portName, "");
          _builder.append("\' with ");
          _builder.newLineIfNotEmpty();
          _builder.append("the right side function unit\'s named input port.<br>");
          _builder.newLine();
          _builder.append("example:<pre>");
          _builder.newLine();
          _builder.append("  ");
          _builder.append("fu -&gt; fu\'.input");
          _builder.newLine();
          _builder.append("</pre>");
          _builder.newLine();
          it.setDocComment(_builder.toString());
          final String rightSideFunctionUnitInputPortVarName = "rightSideFunctionUnitInputPort";
          TypeReference _newTypeReference = context.newTypeReference(InputPort.class, msgType);
          it.addParameter(rightSideFunctionUnitInputPortVarName, _newTypeReference);
          final CompilationStrategy _function = new CompilationStrategy() {
            @Override
            public CharSequence compile(final CompilationStrategy.CompilationContext it) {
              StringConcatenation _builder = new StringConcatenation();
              _builder.append("this.");
              _builder.append(portName, "");
              _builder.append(".operator_mappedTo(");
              _builder.append(rightSideFunctionUnitInputPortVarName, "");
              _builder.append(");");
              _builder.newLineIfNotEmpty();
              return _builder;
            }
          };
          it.setBody(_function);
        }
      };
      annotatedClass.addMethod("operator_mappedTo", _function_2);
      final Procedure1<MutableMethodDeclaration> _function_3 = new Procedure1<MutableMethodDeclaration>() {
        @Override
        public void apply(final MutableMethodDeclaration it) {
          it.setFinal(true);
          StringConcatenation _builder = new StringConcatenation();
          _builder.append("Flow DSL operator \"-&gt;\" for connecting two function units. ");
          _builder.newLine();
          _builder.append("Connects the left side function unit\'s one and only one output port \'");
          _builder.append(portName, "");
          _builder.append("\' with ");
          _builder.newLineIfNotEmpty();
          _builder.append("the right side function unit\'s named output port.<br>");
          _builder.newLine();
          _builder.append("This is normally only used in integrating function units for connecting an integrated function unit with");
          _builder.newLine();
          _builder.append("an output port of the integrating function unit.");
          _builder.newLine();
          _builder.append("example:<pre>");
          _builder.newLine();
          _builder.append("  ");
          _builder.append("fu -&gt; output");
          _builder.newLine();
          _builder.append("</pre>");
          _builder.newLine();
          it.setDocComment(_builder.toString());
          final String rightSideFunctionUnitOutputPortVarName = "rightSideFunctionUnitOutputPort";
          TypeReference _newTypeReference = context.newTypeReference(OutputPort.class, msgType);
          it.addParameter(rightSideFunctionUnitOutputPortVarName, _newTypeReference);
          final CompilationStrategy _function = new CompilationStrategy() {
            @Override
            public CharSequence compile(final CompilationStrategy.CompilationContext it) {
              StringConcatenation _builder = new StringConcatenation();
              _builder.append("this.");
              _builder.append(portName, "");
              _builder.append(".operator_mappedTo(");
              _builder.append(rightSideFunctionUnitOutputPortVarName, "");
              _builder.append(");");
              _builder.newLineIfNotEmpty();
              return _builder;
            }
          };
          it.setBody(_function);
        }
      };
      _xblockexpression = annotatedClass.addMethod("operator_mappedTo", _function_3);
    }
    return _xblockexpression;
  }
}
