/**
 * Copyright (c) 2014 Denis Kuniss (http://www.grammarcraft.de).
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 */
package de.grammarcraft.xtend.flow;

import de.grammarcraft.xtend.flow.FunctionUnitWithOnlyOneInputPort;
import de.grammarcraft.xtend.flow.InputPort;
import java.util.ArrayList;
import java.util.List;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.xbase.lib.Functions.Function0;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;

@SuppressWarnings("all")
public class OutputPort<MessageType extends Object> {
  private final String name;
  
  private final Procedure1<? super Exception> integrationErrorOperation;
  
  /**
   * Creates a named output port with the given port name.
   * @param name the name of the port
   * @param integrationErrorOperation the closure to be executed if no foreign input port at all has been bound to this output port
   */
  public OutputPort(final String name, final Procedure1<? super Exception> integrationErrorOperation) {
    this.name = name;
    this.integrationErrorOperation = integrationErrorOperation;
  }
  
  @Override
  public String toString() {
    return this.name;
  }
  
  private final List<Procedure1<? super MessageType>> outputOperations = new ArrayList<Procedure1<? super MessageType>>();
  
  private void forward(final MessageType msg) {
    boolean _isEmpty = this.outputOperations.isEmpty();
    boolean _not = (!_isEmpty);
    if (_not) {
      final Procedure1<Procedure1<? super MessageType>> _function = new Procedure1<Procedure1<? super MessageType>>() {
        @Override
        public void apply(final Procedure1<? super MessageType> operation) {
          operation.apply(msg);
        }
      };
      IterableExtensions.<Procedure1<? super MessageType>>forEach(this.outputOperations, _function);
    } else {
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("no binding defined for \'");
      _builder.append(this, "");
      _builder.append("\' - message \'");
      _builder.append(msg, "");
      _builder.append("\' could not be delivered.");
      RuntimeException _runtimeException = new RuntimeException(_builder.toString());
      this.integrationErrorOperation.apply(_runtimeException);
    }
  }
  
  /**
   * Wiring operation of the flow DSL.<br>
   * Connects function unit own output port <i>fu.output</i> to an
   * anonymous closure (which may have side effects).<br>
   * E.g.: fu.output -> [closure]
   */
  public void operator_mappedTo(final Procedure1<? super MessageType> operation) {
    this.outputOperations.add(operation);
  }
  
  /**
   * Wiring operation of the flow DSL.<br>
   * Connects function unit own output port <i>fu.output</i> to an
   * foreign function unit with one and only one input port.<br>
   * E.g.: fu.output -> fu'
   */
  public void operator_mappedTo(final FunctionUnitWithOnlyOneInputPort<MessageType> foreignFu) {
    InputPort<MessageType> _theOneAndOnlyInputPort = foreignFu.theOneAndOnlyInputPort();
    Procedure1<? super MessageType> _inputProcessingOperation = _theOneAndOnlyInputPort.inputProcessingOperation();
    this.outputOperations.add(_inputProcessingOperation);
  }
  
  /**
   * Wiring operation of the flow DSL.<br>
   * Connects function unit own output port <i>fu.output</i> to a
   * named input port of a foreign function unit.<br>
   * E.g.: fu.output -> fu'.input
   */
  public void operator_mappedTo(final InputPort<MessageType> foreignInputPort) {
    Procedure1<? super MessageType> _inputProcessingOperation = foreignInputPort.inputProcessingOperation();
    this.outputOperations.add(_inputProcessingOperation);
  }
  
  /**
   * Wiring operation of the flow DSL.<br>
   * Connects function unit own output port <i>fu.output</i> to an
   * anonymous closure (which may have side effects).<br>
   * E.g.: fu.output -> fu'
   */
  public void operator_mappedTo(final OutputPort<MessageType> ownOutputPort) {
    final Procedure1<MessageType> _function = new Procedure1<MessageType>() {
      @Override
      public void apply(final MessageType msg) {
        ownOutputPort.forward(msg);
      }
    };
    this.outputOperations.add(_function);
  }
  
  /**
   * Forward operation of the flow DSL.<br>
   * Forwards a message value to an function unit own output port <i>fu.output</i>.<br>
   * E.g.: .output <= output value<br>
   * This is typically used inside the implementation of function unit to forward
   * results of the function unit's computation to outside over outut ports.
   */
  public void operator_lessEqualsThan(final MessageType msg) {
    this.forward(msg);
  }
  
  /**
   * Forward operation of the flow DSL.<br>
   * Forwards a message value to an function unit own output port <i>fu.output</i>.<br>
   * E.g.: .output <= [closure computing output value]<br>
   * This is typically used inside the implementation of function unit to forward
   * results of the function unit's computation to outside over outut ports.
   */
  public void operator_lessEqualsThan(final Function0<? extends MessageType> msgClosure) {
    MessageType _apply = msgClosure.apply();
    this.forward(_apply);
  }
}
