/**
 * Copyright (c) 2015 Denis Kuniss (http://www.grammarcraft.de).
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 */
package de.grammarcraft.xtend.flow;

import com.google.common.base.Objects;
import de.grammarcraft.xtend.flow.FunctionUnitWithOnlyOneInputPort;
import java.util.ArrayList;
import java.util.List;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.xbase.lib.Functions.Function0;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;

@SuppressWarnings("all")
public class InputPort<MessageType extends Object> {
  private final String name;
  
  private final Procedure1<? super MessageType> processInputOperation;
  
  private final List<Procedure1<? super MessageType>> inputOperations = new ArrayList<Procedure1<? super MessageType>>();
  
  private final Procedure1<? super Exception> integrationErrorOperation;
  
  /**
   * Creates a named input port with the given port name.
   * @param name the name of the port
   * @param processInputOperation the closure to be applied to an input message received over this port
   * @param integrationErrorOperation the closure to be executed if the input processing closure given before is null
   */
  public InputPort(final String name, final Procedure1<? super MessageType> processInputOperation, final Procedure1<? super Exception> integrationErrorOperation) {
    this.name = name;
    this.processInputOperation = processInputOperation;
    this.integrationErrorOperation = integrationErrorOperation;
  }
  
  /**
   * Creates a named input port with the given port name without predefined input processing closure.<br>
   * This is intended to be used inside integration function units where the input processing closure is defined at the constructor by
   * an dedicated wiring operation.
   * @param name the name of the port
   * @param integrationErrorOperation the closure to be executed if the input processing closure is not defined
   */
  public InputPort(final String name, final Procedure1<? super Exception> integrationErrorOperation) {
    this.name = name;
    final Procedure1<MessageType> _function = new Procedure1<MessageType>() {
      @Override
      public void apply(final MessageType it) {
        InputPort.this.forward(it);
      }
    };
    this.processInputOperation = _function;
    this.integrationErrorOperation = integrationErrorOperation;
  }
  
  private void forward(final MessageType msg) {
    boolean _isEmpty = this.inputOperations.isEmpty();
    boolean _not = (!_isEmpty);
    if (_not) {
      final Procedure1<Procedure1<? super MessageType>> _function = new Procedure1<Procedure1<? super MessageType>>() {
        @Override
        public void apply(final Procedure1<? super MessageType> operation) {
          operation.apply(msg);
        }
      };
      IterableExtensions.<Procedure1<? super MessageType>>forEach(this.inputOperations, _function);
    } else {
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("no binding defined for \'");
      _builder.append(this, "");
      _builder.append("\' - message \'");
      _builder.append(msg, "");
      _builder.append("\' could not be delivered.");
      RuntimeException _runtimeException = new RuntimeException(_builder.toString());
      this.integrationErrorOperation.apply(_runtimeException);
    }
  }
  
  @Override
  public String toString() {
    return this.name;
  }
  
  private void processInput(final MessageType msg) {
    boolean _equals = Objects.equal(this.processInputOperation, null);
    if (_equals) {
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("no binding defined for \'");
      _builder.append(this, "");
      _builder.append("\' - message \'");
      _builder.append(msg, "");
      _builder.append("\' could not be processed.");
      RuntimeException _runtimeException = new RuntimeException(_builder.toString());
      this.integrationErrorOperation.apply(_runtimeException);
    } else {
      this.processInputOperation.apply(msg);
    }
  }
  
  /**
   * Forward operation of the flow DSL.<br>
   * Forwards a message value to the function unit's input port <i>fu.input</i>.<br>
   * E.g.: fu.input <= input value<br>
   * This is typically used to provide an initial value to the function unit's flow this
   * input port belongs to.
   */
  public void operator_lessEqualsThan(final MessageType msg) {
    this.processInput(msg);
  }
  
  /**
   * Forward operation of the flow DSL.<br>
   * Forwards a message value computed by the passed closure to the function unit's input port <i>fu.input</i>.<br>
   * E.g.: fu.input <= input value<br>
   * This is typically used to provide an initial value to the function unit's flow this
   * input port belongs to.
   */
  public void operator_lessEqualsThan(final Function0<? extends MessageType> msgClosure) {
    MessageType _apply = msgClosure.apply();
    this.processInput(_apply);
  }
  
  public Procedure1<? super MessageType> inputProcessingOperation() {
    return this.processInputOperation;
  }
  
  /**
   * Wiring operation of the flow DSL.<br>
   * Connects function unit own input port <i>input</i> to a
   * named input port of an integrated function unit.<br>
   * E.g.: .input -> fu'.input<br>
   * This is typically used to forward input messages inside an integration function unit to
   * the an input port of an function unit which is integrated.
   */
  public void operator_mappedTo(final InputPort<MessageType> integratedInputPort) {
    Procedure1<? super MessageType> _inputProcessingOperation = integratedInputPort.inputProcessingOperation();
    this.inputOperations.add(_inputProcessingOperation);
  }
  
  /**
   * Wiring operation of the flow DSL.<br>
   * Connects function unit own input port <i>input</i> to the one and only one
   * input port of an integrated function unit.<br>
   * E.g.: .input -> fu'<br>
   * This is typically used to forward input messages inside an integration function unit to
   * the an input port of an function unit which is integrated.
   */
  public void operator_mappedTo(final FunctionUnitWithOnlyOneInputPort<MessageType> integratedFunctionUnit) {
    InputPort<MessageType> _theOneAndOnlyInputPort = integratedFunctionUnit.theOneAndOnlyInputPort();
    Procedure1<? super MessageType> _inputProcessingOperation = _theOneAndOnlyInputPort.inputProcessingOperation();
    this.inputOperations.add(_inputProcessingOperation);
  }
}
