package de.gofabian.jmigrate.jooq;

import de.gofabian.jmigrate.MigrationException;
import org.jooq.DSLContext;
import org.jooq.exception.DataAccessException;
import org.jooq.impl.DSL;
import org.jooq.impl.SQLDataType;

import java.util.List;

import static org.jooq.impl.DSL.*;

/**
 * The {@link LockRepository} manages the lock table via JOOQ.
 */
class LockRepository {

    private DSLContext context;
    private String tableName;

    LockRepository(DSLContext context, String tableName) {
        this.context = context;
        this.tableName = tableName;
    }

    /**
     * Verify that lock flag is not set in database.
     *
     * @throws MigrationException lock flag is set, or data in lock table is invalid
     */
    void verifyNoLockExists() {
        List<LockEntry> lockEntries;
        try {
            lockEntries = context.selectFrom(table(name(tableName)))
                    .fetchInto(LockEntry.class);
        } catch (DataAccessException e) {
            // table does not exist -> no lock
            return;
        }

        switch (lockEntries.size()) {
            case 0:
                // no entry -> no lock
                break;

            case 1:
                if (lockEntries.get(0).isLocked()) {
                    throw new MigrationException("locked flag in table " + tableName + " is set. " +
                            "Solve this error manually and clear the flag to continue.");
                }
                break;

            default:
                throw new MigrationException("Expected at most 1 row in table " + tableName + " but found " +
                        lockEntries.size());
        }
    }

    /**
     * Set persistent lock flag in database.
     */
    void lock() {
        context.createTableIfNotExists(table(name(tableName)))
                .column(name("locked"), SQLDataType.BOOLEAN.nullable(false))
                .column(name("updated_at"), SQLDataType.LOCALDATETIME.nullable(false))
                .execute();

        int updatedRows = context.update(table(name(tableName)))
                .set(field(name("locked")), true)
                .set(field(name("updated_at")), DSL.currentLocalDateTime())
                .execute();

        if (updatedRows == 0) {
            context.insertInto(table(name(tableName)))
                    .columns(
                            field(name("locked")),
                            field(name("updated_at"))
                    )
                    .values(
                            true,
                            DSL.currentLocalDateTime()
                    )
                    .execute();
        }
    }

    /**
     * Clear persistent lock flag in database.
     */
    void unlock() {
        context.update(table(name(tableName)))
                .set(field(name("locked")), false)
                .set(field(name("updated_at")), DSL.currentLocalDateTime())
                .execute();
    }

}
