package de.gofabian.jmigrate.jooq;

import de.gofabian.jmigrate.HistoryEntry;
import org.jooq.DSLContext;
import org.jooq.InsertValuesStep6;
import org.jooq.Record;
import org.jooq.impl.SQLDataType;

import java.util.List;
import java.util.stream.Collectors;

import static org.jooq.impl.DSL.*;

/**
 * The {@link HistoryRepository} manages the history table via JOOQ.
 */
class HistoryRepository {

    private DSLContext context;
    private String tableName;

    HistoryRepository(DSLContext context, String tableName) {
        this.context = context;
        this.tableName = tableName;
    }

    void createTableIfNotExists() {
        context.createTableIfNotExists(name(tableName))
                .column(name("order"), SQLDataType.INTEGER.identity(true))
                .column(name("migration_name"), SQLDataType.VARCHAR(255).nullable(false))
                .column(name("applied_at"), SQLDataType.LOCALDATETIME.nullable(false))
                .column(name("author"), SQLDataType.VARCHAR(255).nullable(false))
                .column(name("description"), SQLDataType.VARCHAR(255).nullable(false))
                .column(name("jmigrate_version"), SQLDataType.VARCHAR(20).nullable(false))
                .constraints(
                        constraint(name("pk_" + tableName + "_order")).primaryKey(name("order")),
                        constraint(name("uk_" + tableName + "_migration_name")).unique(name("migration_name"))
                )
                .execute();
    }

    List<HistoryEntry> findHistory() {
        return context.selectFrom(table(name(tableName)))
                .orderBy(field(name("order")).asc())
                .fetchInto(HistoryEntry.class);
    }

    void appendHistory(List<HistoryEntry> moreHistory) {
        InsertValuesStep6<Record, Object, Object, Object, Object, Object, Object> step =
                context.insertInto(table(name(tableName)))
                        .columns(
                                field(name("order")),
                                field(name("migration_name")),
                                field(name("applied_at")),
                                field(name("author")),
                                field(name("description")),
                                field(name("jmigrate_version"))
                        );
        for (HistoryEntry entry : moreHistory) {
            step = step.values(
                    entry.getOrder(),
                    entry.getMigrationName(),
                    entry.getAppliedAt(),
                    entry.getAuthor(),
                    entry.getDescription(),
                    entry.getJmigrateVersion()
            );
        }
        step.execute();
    }

    void deleteHistory(List<HistoryEntry> lessHistory) {
        List<Integer> orders = lessHistory.stream()
                .map(HistoryEntry::getOrder)
                .collect(Collectors.toList());

        context.deleteFrom(table(name(tableName)))
                .where(field(name("order"), SQLDataType.BIGINT).in(orders))
                .execute();
    }
}
