/*
 * Decompiled with CFR 0.152.
 */
package de.gerdiproject.harvest.etls.utils;

import com.google.gson.Gson;
import de.gerdiproject.harvest.config.Configuration;
import de.gerdiproject.harvest.config.parameters.BooleanParameter;
import de.gerdiproject.harvest.etls.AbstractETL;
import de.gerdiproject.harvest.etls.ETLPreconditionException;
import de.gerdiproject.harvest.etls.constants.ETLConstants;
import de.gerdiproject.harvest.etls.enums.ETLHealth;
import de.gerdiproject.harvest.etls.enums.ETLState;
import de.gerdiproject.harvest.etls.events.GetETLManagerEvent;
import de.gerdiproject.harvest.etls.events.GetRepositoryNameEvent;
import de.gerdiproject.harvest.etls.events.HarvestFinishedEvent;
import de.gerdiproject.harvest.etls.events.HarvestStartedEvent;
import de.gerdiproject.harvest.etls.json.ETLInfosJson;
import de.gerdiproject.harvest.etls.json.ETLJson;
import de.gerdiproject.harvest.etls.json.ETLManagerJson;
import de.gerdiproject.harvest.etls.utils.EtlUtils;
import de.gerdiproject.harvest.etls.utils.TimestampedEntry;
import de.gerdiproject.harvest.etls.utils.TimestampedList;
import de.gerdiproject.harvest.event.EventSystem;
import de.gerdiproject.harvest.rest.AbstractRestObject;
import de.gerdiproject.harvest.scheduler.Scheduler;
import de.gerdiproject.harvest.scheduler.events.GetSchedulerEvent;
import de.gerdiproject.harvest.utils.data.DiskIO;
import de.gerdiproject.harvest.utils.file.ICachedObject;
import java.io.File;
import java.nio.charset.StandardCharsets;
import java.util.Date;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import javax.ws.rs.core.MultivaluedMap;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ETLManager
extends AbstractRestObject<ETLManager, ETLManagerJson>
implements ICachedObject {
    private static final Logger LOGGER = LoggerFactory.getLogger(ETLManager.class);
    private final DiskIO diskIo;
    private final File cacheFile;
    private final List<AbstractETL<?, ?>> etls;
    private final BooleanParameter concurrentParam;
    private final BooleanParameter forceHarvestParameter;
    private final TimestampedList<ETLState> combinedStateHistory = new TimestampedList<ETLState>(ETLState.INITIALIZING, 10);
    private String lastHarvestHash;

    public ETLManager(String moduleName) {
        super(moduleName, GetETLManagerEvent.class);
        this.etls = new LinkedList();
        this.concurrentParam = Configuration.registerParameter(ETLConstants.CONCURRENT_PARAM);
        this.forceHarvestParameter = Configuration.registerParameter(ETLConstants.FORCED_PARAM);
        this.cacheFile = new File(String.format("cache/%s/state.json", moduleName));
        this.diskIo = new DiskIO(new Gson(), StandardCharsets.UTF_8);
        this.setStatus(ETLState.IDLE);
    }

    @Override
    public void loadFromDisk() {
        ETLInfosJson loadedState = this.diskIo.getObject(this.cacheFile, ETLInfosJson.class);
        if (loadedState != null) {
            ETLJson overallInfo = loadedState.getOverallInfo();
            Map<String, ETLJson> etlInfos = loadedState.getEtlInfos();
            if (overallInfo == null || etlInfos == null) {
                LOGGER.warn(ETLConstants.ETL_MANAGER_LOAD_ERROR);
            } else {
                this.lastHarvestHash = overallInfo.getVersionHash();
                this.combinedStateHistory.addAllSorted(overallInfo.getStateHistory());
                for (AbstractETL<?, ?> etl : this.etls) {
                    String etlName = etl.getName();
                    ETLJson etlInfo = etlInfos.get(etlName);
                    if (etlInfo == null) {
                        LOGGER.warn(String.format("Could not load %s state from cache!", etlName));
                        continue;
                    }
                    etl.loadFromJson(etlInfo);
                }
                LOGGER.debug(String.format(ETLConstants.ETL_MANAGER_LOADED, this.cacheFile));
            }
        }
    }

    @Override
    public void saveToDisk() {
        this.diskIo.writeObjectToFile(this.cacheFile, (Object)this.getETLsAsJson());
    }

    @Override
    protected String getPrettyPlainText() {
        StringBuilder sb = new StringBuilder();
        int totalCurrCount = 0;
        int totalMaxCount = 0;
        for (AbstractETL<?, ?> etl : this.etls) {
            sb.append(etl.toString());
            if (!etl.isEnabled()) continue;
            totalCurrCount += etl.getHarvestedCount();
            int maxCount = etl.getMaxNumberOfDocuments();
            if (maxCount == -1 || totalMaxCount == -1) {
                totalMaxCount = -1;
                continue;
            }
            totalMaxCount += maxCount;
        }
        ETLState state = this.getState();
        StringBuilder stateStringBuilder = new StringBuilder();
        stateStringBuilder.append(state.toString().toLowerCase(Locale.ENGLISH));
        if (state == ETLState.HARVESTING) {
            if (totalMaxCount == -1) {
                stateStringBuilder.append(String.format(" (%d / ???)", totalCurrCount));
            } else {
                stateStringBuilder.append(String.format(" % 3d%% (%d / %d)", Math.round(100.0f * (float)totalCurrCount / (float)totalMaxCount), totalCurrCount, totalMaxCount));
            }
        }
        sb.append(String.format("%s : %s [Health: %s]%n", new Object[]{"---\nOVERALL", stateStringBuilder.toString(), EtlUtils.getCombinedHealth(this.etls)}));
        if (state == ETLState.HARVESTING) {
            long remainingMilliSeconds = EtlUtils.estimateRemainingHarvestTime(this.combinedStateHistory.getLatestTimestamp(), state, totalCurrCount, totalMaxCount);
            sb.append(EtlUtils.formatHarvestTime(remainingMilliSeconds));
        }
        return sb.toString();
    }

    @Override
    public ETLManagerJson getAsJson(MultivaluedMap<String, String> query) {
        String repositoryName = (String)EventSystem.sendSynchronousEvent(new GetRepositoryNameEvent());
        int harvestedCount = this.getHarvestedCount();
        int maxDocumentCount = this.getMaxNumberOfDocuments();
        long remainingHarvestTime = this.estimateRemainingHarvestTime();
        long lastHarvestTimestamp = this.getLatestHarvestTimestamp();
        Date nextHarvestDate = ((Scheduler)EventSystem.sendSynchronousEvent(new GetSchedulerEvent())).getNextHarvestDate();
        boolean hasEnabledETLs = false;
        for (AbstractETL<?, ?> etl : this.etls) {
            if (!etl.isEnabled()) continue;
            hasEnabledETLs = true;
            break;
        }
        return new ETLManagerJson(repositoryName, this.getState(), EtlUtils.getCombinedHealth(this.etls), harvestedCount, maxDocumentCount == -1 ? null : Integer.valueOf(maxDocumentCount), remainingHarvestTime == -1L ? null : Long.valueOf(remainingHarvestTime), lastHarvestTimestamp == -1L ? null : new Date(lastHarvestTimestamp).toString(), nextHarvestDate == null ? null : nextHarvestDate.toString(), hasEnabledETLs);
    }

    public ETLInfosJson getETLsAsJson() {
        return new ETLInfosJson(new ETLJson(this.getClass().getSimpleName(), this.combinedStateHistory, new TimestampedList<ETLHealth>(EtlUtils.getCombinedHealth(this.etls), 1), this.getHarvestedCount(), this.getMaxNumberOfDocuments(), EtlUtils.getCombinedHashes(this.etls)), this.etls);
    }

    public ETLJson getETLAsJson(MultivaluedMap<String, String> query) throws IllegalArgumentException {
        List nameList;
        List list = nameList = query == null ? null : (List)query.get((Object)"name");
        if (nameList == null || nameList.isEmpty()) {
            throw new IllegalArgumentException("Missing query parameter 'name'!");
        }
        String etlName = (String)nameList.get(0);
        Optional<AbstractETL> etl = this.etls.stream().filter(e -> e.getName().equalsIgnoreCase(etlName)).findFirst();
        if (!etl.isPresent()) {
            throw new IllegalArgumentException(String.format("Unknown ETL name '%s'!", etlName));
        }
        return etl.get().getAsJson();
    }

    public void register(AbstractETL<?, ?> addedEtl) {
        if (this.etls.contains(addedEtl)) {
            LOGGER.info(String.format("Did not register %s, because it was already registered!", addedEtl.getClass().getSimpleName()));
        } else {
            String etlNameOrig;
            String etlName = etlNameOrig = addedEtl.getName().replaceAll("[^a-zA-Z0-9]", "");
            int duplicateCount = 1;
            while (true) {
                String etlNameTemp = etlName;
                if (this.etls.stream().noneMatch(e -> e.getName().equals(etlNameTemp))) break;
                etlName = etlNameOrig + ++duplicateCount;
            }
            addedEtl.setName(etlName);
            this.etls.add(addedEtl);
        }
    }

    public boolean hasOutdatedETLs() {
        ETLState currentStatus = this.getState();
        if (currentStatus == ETLState.IDLE || currentStatus == ETLState.DONE) {
            EtlUtils.processETLs(this.etls, harvester -> harvester.update());
        }
        int maxDocs = this.getMaxNumberOfDocuments();
        int currentDocs = this.getHarvestedCount();
        if (maxDocs == -1 && currentDocs == 0 || currentDocs < maxDocs) {
            return true;
        }
        String currentHash = EtlUtils.getCombinedHashes(this.etls);
        return currentHash == null || !currentHash.equals(this.lastHarvestHash);
    }

    @Override
    public void addEventListeners() {
        super.addEventListeners();
        for (AbstractETL<?, ?> etl : this.etls) {
            etl.addEventListeners();
        }
    }

    @Override
    public void removeEventListeners() {
        super.removeEventListeners();
        for (AbstractETL<?, ?> etl : this.etls) {
            etl.removeEventListeners();
        }
    }

    public void harvest() throws IllegalStateException {
        if (this.getState() != ETLState.IDLE) {
            throw new IllegalStateException("Cannot start harvest: Please wait for the current harvest to finish, or abort it!");
        }
        if (!((Boolean)this.forceHarvestParameter.getValue()).booleanValue() && !this.hasOutdatedETLs()) {
            throw new ETLPreconditionException("Did not start harvest, because no changes were detected!");
        }
        ((CompletableFuture)CompletableFuture.runAsync(() -> {
            boolean isPrepared = this.prepareETLsForHarvest();
            if (!isPrepared) {
                throw new ETLPreconditionException("Cannot start harvest: No ETL could be prepared!");
            }
            this.harvestETLs();
        }).thenAccept(v -> {
            this.lastHarvestHash = EtlUtils.getCombinedHashes(this.etls);
            this.saveToDisk();
            ETLState status = this.getState();
            if (status == ETLState.ABORTING) {
                LOGGER.info("All ETLs have been aborted!");
            } else if (status == ETLState.HARVESTING) {
                LOGGER.info("Harvest finished!");
            }
            EventSystem.sendEvent(new HarvestFinishedEvent(true, this.lastHarvestHash));
            this.setStatus(ETLState.IDLE);
        })).exceptionally(reason -> {
            if (reason.getCause() instanceof ETLPreconditionException) {
                LOGGER.error("Cannot start harvest: No ETL could be prepared!");
            } else {
                LOGGER.error("Harvesting interrupted by unexpected error!", reason);
            }
            EtlUtils.processETLs(this.etls, harvester -> harvester.cancelHarvest());
            this.saveToDisk();
            LOGGER.info("Harvest failed!");
            EventSystem.sendEvent(new HarvestFinishedEvent(false, EtlUtils.getCombinedHashes(this.etls)));
            this.setStatus(ETLState.IDLE);
            return null;
        });
    }

    public int getMaxNumberOfDocuments() {
        List<Integer> sizes = EtlUtils.processETLsAsList(this.etls, harvester -> harvester.getMaxNumberOfDocuments());
        int total = 0;
        for (int size : sizes) {
            if (size == -1) {
                return -1;
            }
            total += size;
        }
        return total;
    }

    public void abortHarvest() {
        ETLState currentStatus = this.getState();
        if (currentStatus != ETLState.QUEUED && currentStatus != ETLState.HARVESTING) {
            throw new IllegalStateException(String.format("Cannot abort a harvest when it is '%s'!", this.combinedStateHistory.toString()));
        }
        this.setStatus(ETLState.ABORTING);
        EtlUtils.processETLs(this.etls, harvester -> {
            if (harvester.getState() == ETLState.HARVESTING || harvester.getState() == ETLState.QUEUED) {
                harvester.abortHarvest();
            }
        });
    }

    public int getHarvestedCount() {
        return EtlUtils.sumUpETLValues(this.etls, harvester -> harvester.getHarvestedCount());
    }

    public ETLHealth getHealth() {
        return EtlUtils.getCombinedHealth(this.etls);
    }

    public ETLState getState() {
        return this.combinedStateHistory.getLatestValue();
    }

    public long estimateRemainingHarvestTime() {
        return EtlUtils.estimateRemainingHarvestTime(this.combinedStateHistory.getLatestTimestamp(), this.combinedStateHistory.getLatestValue(), this.getHarvestedCount(), this.getMaxNumberOfDocuments());
    }

    private long getLatestHarvestTimestamp() {
        Iterator reverseIter = this.combinedStateHistory.descendingIterator();
        while (reverseIter.hasNext()) {
            TimestampedEntry item = (TimestampedEntry)reverseIter.next();
            if (item.getValue() != ETLState.HARVESTING) continue;
            return item.getTimestamp();
        }
        return -1L;
    }

    private boolean prepareETLsForHarvest() {
        LOGGER.info("Preparing ETLs for harvest.");
        this.setStatus(ETLState.QUEUED);
        int preparedCount = EtlUtils.sumUpETLValues(this.etls, etl -> {
            try {
                if (this.getState() != ETLState.ABORTING) {
                    etl.prepareHarvest();
                    return 1;
                }
            }
            catch (ETLPreconditionException e) {
                LOGGER.info(String.format("Unable to prepare %s for harvest!", etl.getName()), (Throwable)e);
            }
            return 0;
        });
        if (preparedCount == 0 || this.getState() == ETLState.ABORTING) {
            this.setStatus(ETLState.IDLE);
            return false;
        }
        return true;
    }

    private void harvestETLs() {
        LOGGER.info("Starting ETLs.");
        this.setStatus(ETLState.HARVESTING);
        EventSystem.sendEvent(new HarvestStartedEvent(EtlUtils.getCombinedHashes(this.etls), this.getMaxNumberOfDocuments()));
        if (((Boolean)this.concurrentParam.getValue()).booleanValue()) {
            List<CompletableFuture> asyncHarvests = EtlUtils.processETLsAsList(this.etls, etl -> CompletableFuture.runAsync(() -> {
                if (this.getState() != ETLState.ABORTING && etl.getState() == ETLState.QUEUED) {
                    etl.harvest();
                }
            }));
            try {
                CompletableFuture[] asyncHarvestArray = asyncHarvests.toArray(new CompletableFuture[asyncHarvests.size()]);
                CompletableFuture.allOf(asyncHarvestArray).get();
            }
            catch (InterruptedException | ExecutionException e) {
                LOGGER.error("Error iterating through ETL components!", (Throwable)e);
            }
        } else {
            EtlUtils.processETLs(this.etls, etl -> {
                if (this.getState() != ETLState.ABORTING && etl.getState() == ETLState.QUEUED) {
                    etl.harvest();
                }
            });
        }
    }

    private void setStatus(ETLState status) {
        this.combinedStateHistory.addValue(status);
    }
}

