/*
 * Copyright (c) 2020 gematik GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *    http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package de.gematik.ti.healthcard.control.nfdm;

import java.lang.reflect.Method;
import java.nio.charset.Charset;
import java.util.Arrays;
import java.util.Optional;
import java.util.stream.Stream;

import org.hamcrest.core.Is;
import org.junit.Assert;
import org.junit.Test;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import de.gematik.ti.utils.codec.Hex;

/**
 * UnitTest of some methode in {@link NfdDpeReader}
  */
public class NfdDpeReaderTest {
    private static final Logger LOG = LoggerFactory.getLogger(NfdDpeReaderTest.class);
    private final String nfdWestEuropeCharset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
    private final String nfdUtfEightCharset = "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";

    @Test
    public void testGetCharSettingFirst() {
        try {
            final Method getCharSettingFirst = NfdDpeReader.class.getDeclaredMethod("getCharSettingFirst", byte[].class);
            getCharSettingFirst.setAccessible(true);
            final byte[] compressedData = Hex.decode(nfdWestEuropeCharset);
            final Charset charset = (Charset) getCharSettingFirst.invoke(NfdDpeReader.class, compressedData);
            LOG.debug("charset: " + charset);
            Assert.assertThat(charset.displayName(), Is.is("ISO-8859-15"));
        } catch (final Exception e) {
            Assert.fail(e.toString());
        }

        try {
            final Method getCharSettingFirst = NfdDpeReader.class.getDeclaredMethod("getCharSettingFirst", byte[].class);
            getCharSettingFirst.setAccessible(true);
            final byte[] compressedData = Hex.decode(nfdUtfEightCharset);
            final Charset charset = (Charset) getCharSettingFirst.invoke(NfdDpeReader.class, compressedData);
            LOG.debug("charset: " + charset);
            Assert.assertThat(charset.displayName(), Is.is("UTF-8"));
        } catch (final Exception e) {
            Assert.fail(e.toString());
        }
    }

    @Test
    public void testInternalProcessingOnGetCharSettingFirst() {
        String encodingDef = "encoding=";
        String xml = "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"no\"?>";
        String chardSetString = internalOnGetCharSettingFirst(encodingDef, xml);
        LOG.debug("chardSetString: " + chardSetString);
        Assert.assertThat(chardSetString.toLowerCase(), Is.is("utf-8"));
        //

        xml = "<?xml version='1.0' encoding='ISO-8850-1'?>";
        chardSetString = internalOnGetCharSettingFirst(encodingDef, xml);
        LOG.debug("chardSetString: " + chardSetString);
        Assert.assertThat(chardSetString.toLowerCase(), Is.is("iso-8850-1"));
        //
        encodingDef = "encoding=";
        xml = "<?XML VERSION='1.0' ENCODING='ISO-8850-1'?>";
        chardSetString = internalOnGetCharSettingFirst(encodingDef, xml);
        LOG.debug("chardSetString: " + chardSetString);
        Assert.assertThat(chardSetString.toLowerCase(), Is.is("iso-8850-1"));
    }

    private String internalOnGetCharSettingFirst(final String encodingDef, final String xml) {
        final int defLength = encodingDef.length();
        final String[] sa = new String[] { "" };
        final String[] split = xml.split(" ");
        Arrays.stream(split).filter(s -> s.trim().toLowerCase().startsWith(encodingDef)).flatMap(str -> {
            Stream<Character> characterStream = str.substring(defLength).chars().mapToObj(c -> (char) c);
            Optional<Character> first = characterStream.findFirst();
            if (!first.isPresent()) {
                return Stream.of();
            }
            Character separator = first.get();
            String temp = str.substring(defLength + 1);
            int nextPosition = temp.indexOf(separator);
            characterStream = temp.substring(0, nextPosition).chars().mapToObj(c -> (char) c);
            return characterStream;
        }).forEach(a -> sa[0] = sa[0] + a);

        return sa[0];
    }
}
