/*
 * Copyright (c) 2022 gematik GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the License);
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an 'AS IS' BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package de.gematik.test.tiger.lib.rbel;

import static org.assertj.core.api.Assertions.assertThat;
import static org.assertj.core.api.Assertions.assertThatThrownBy;
import de.gematik.rbellogger.data.RbelElement;
import de.gematik.rbellogger.data.facet.RbelHttpRequestFacet;
import org.junit.Test;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.CsvSource;

public class RbelMessageValidatorTest {

    @Test
    public void testPathEqualsWithRelativePath_OK() {
        assertThat(new RbelMessageValidator().doesPathOfMessageMatch(
            buildRequestWithPath("/foo/bar?sch=mar"), "/foo/bar"))
            .isTrue();
    }

    @Test
    public void testPathEqualsWithUrl_OK() {
        assertThat(new RbelMessageValidator().doesPathOfMessageMatch(
            buildRequestWithPath("http://bl.ub/foo/bar?sch=mar"), "/foo/bar"))
            .isTrue();
    }

    @Test
    public void testPathMatchingWithUrlLeading_OK() {
        assertThat(new RbelMessageValidator().doesPathOfMessageMatch(
            buildRequestWithPath("http://bl.ub/foo/bar?sch=mar"), "\\/.*\\/bar"))
            .isTrue();
    }

    @Test
    public void testPathMatchingWithUrlTrailing_OK() {
        assertThat(new RbelMessageValidator().doesPathOfMessageMatch(
            buildRequestWithPath("http://bl.ub/foo/bar?sch=mar"), "\\/foo\\/.*"))
            .isTrue();
    }

    @Test
    public void testPathMatchingWithUrlInMid_OK() {
        assertThat(new RbelMessageValidator().doesPathOfMessageMatch(
            buildRequestWithPath("http://bl.ub/foo/bar/test?sch=mar"), "\\/foo\\/.*/test"))
            .isTrue();
    }

    @Test
    public void testPathMatchingWithInvalidRegex_NOK() {
        assertThat(new RbelMessageValidator().doesPathOfMessageMatch(
            buildRequestWithPath("http://bl.ub/foo/bar/test?sch=mar"), "/foo/.*/[test]"))
            .isFalse();
    }

    private RbelElement buildRequestWithPath(final String path) {
        final RbelElement rbelElement = new RbelElement(null, null);
        rbelElement.addFacet(RbelHttpRequestFacet.builder()
            .path(new RbelElement(path.getBytes(), null))
            .build());
        return rbelElement;
    }

    @Test
    public void testSourceTestInvalid_NOK() {
        assertThatThrownBy(() -> new RbelMessageValidator().compareXMLStructure(
            "<root><header></header><body><!-- test comment-- --></body>",
            "<root><header></header><body></body></root>"))
            .isInstanceOf(org.xmlunit.XMLUnitException.class);
    }

    @Test
    public void testSourceOracleInvalid_NOK() {
        assertThatThrownBy(() -> new RbelMessageValidator().compareXMLStructure(
            "<root><header></header><body><!-- test comment --></body></root>",
            "<root><header></header><body></body>"))
            .isInstanceOf(org.xmlunit.XMLUnitException.class);
    }

    @Test
    public void testSourceNoComment_OK() {
        new RbelMessageValidator().compareXMLStructure(
            "<root><header></header><body><!-- test comment --></body></root>",
            "<root><header></header><body></body></root>", "nocomment");
    }

    @Test
    public void testSourceNoCommetTxtTrim_OK() {
        new RbelMessageValidator().compareXMLStructure(
            "<root><header></header><body>test     <!-- test comment --></body></root>",
            "<root><header></header><body>test</body></root>", "nocomment,txttrim");
    }

    @Test
    public void testSourceNoCommetTxtTrim2_OK() {
        new RbelMessageValidator().compareXMLStructure(
            "<root><header></header><body>    test     <!-- test comment --></body></root>",
            "<root><header></header><body>test</body></root>", "nocomment,txttrim");
    }

    @Test
    public void testSourceNoCommetTxtTrim3_OK() {
        new RbelMessageValidator().compareXMLStructure(
            "<root><header></header><body>    test xxx    <!-- test comment --></body></root>",
            "<root><header></header><body>test xxx</body></root>", "nocomment,txttrim");
    }

    @Test
    public void testSourceNoCommetTxtTrim4_OK() {
        assertThatThrownBy(() -> new RbelMessageValidator().compareXMLStructure(
            "<root><header></header><body>    test   xxx    <!-- test comment --></body></root>",
            "<root><header></header><body>test xxx</body></root>", "nocomment,txttrim"))
            .isInstanceOf(AssertionError.class);
    }

    @Test
    public void testSourceNoCommetTxtNormalize_OK() {
        new RbelMessageValidator().compareXMLStructure(
            "<root><header></header><body>  test    xxxx   </body>  <!-- test comment --></root>",
            "<root><header></header><body>test xxxx </body></root>", "nocomment,txtnormalize");
    }

    @Test
    public void testSourceAttrOrder_OK() {
        new RbelMessageValidator().compareXMLStructure(
            "<root><header></header><body attr1='1'   attr2='2'></body></root>",
            "<root><header></header><body attr2='2' attr1='1'></body></root>");
    }

    @ParameterizedTest
    @CsvSource({"http://server, ''", "http://server/, /", "http://server, /", "http://server/, ''"})
    public void testEmptyPathMatching(final String url, final String path) {
        assertThat(new RbelMessageValidator().doesPathOfMessageMatch(
            buildRequestWithPath(url), path))
            .isTrue();
    }

    @ParameterizedTest
    @CsvSource({"http://server/blu/, /", "http://server/, /bla", "http://server/bla, ''"})
    public void testPathOfMessageMatch_NOK(final String url, final String path) {
        assertThat(new RbelMessageValidator().doesPathOfMessageMatch(
            buildRequestWithPath(url), path))
            .isFalse();
    }
}
