/*
 * Copyright (c) 2022 gematik GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the License);
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an 'AS IS' BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package de.gematik.test.tiger.lib;

import static org.awaitility.Awaitility.await;
import de.gematik.rbellogger.RbelOptions;
import de.gematik.rbellogger.util.RbelAnsiColors;
import de.gematik.test.tiger.common.Ansi;
import de.gematik.test.tiger.common.banner.Banner;
import de.gematik.test.tiger.common.config.TigerGlobalConfiguration;
import de.gematik.test.tiger.common.data.config.tigerProxy.TigerProxyConfiguration;
import de.gematik.test.tiger.lib.exception.TigerStartupException;
import de.gematik.test.tiger.lib.monitor.MonitorUI;
import de.gematik.test.tiger.lib.parser.model.gherkin.Step;
import de.gematik.test.tiger.lib.serenityRest.SerenityRestUtils;
import de.gematik.test.tiger.testenvmgr.TigerTestEnvMgr;
import de.gematik.test.tiger.testenvmgr.TigerTestEnvMgrApplication;
import de.gematik.test.tiger.testenvmgr.env.TigerStatusUpdate;
import java.awt.HeadlessException;
import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.nio.charset.StandardCharsets;
import java.time.Duration;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import lombok.Getter;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.io.IOUtils;
import org.springframework.boot.WebApplicationType;
import org.springframework.boot.builder.SpringApplicationBuilder;
import org.springframework.context.ConfigurableApplicationContext;

/**
 * The TigerDirector is the public interface of the high level features of the Tiger test framework.
 * <ul>
 *     <li>read and apply Tiger test framework configuration from tiger.yaml</li>
 *     <li>start monitoring UI, Tiger test environment manager and local Tiger Proxy</li>
 *     <li>Sync test cases with Polarion</li>
 *     <li>Sync test reports with Aurora and Polarion</li>
 *     <li>Create requirement coverage report based on @Afo annotations and requirements downloaded from Polarion</li>
 * </ul>
 * It also provides access to the Tiger test environment manager, the local Tiger Proxy and the Monitor UI interface.
 */
@SuppressWarnings("unused")
@Slf4j
public class TigerDirector {

    private static TigerTestEnvMgr tigerTestEnvMgr;
    @SuppressWarnings("OptionalUsedAsFieldOrParameterType")
    private static Optional<MonitorUI> optionalMonitorUI = Optional.empty();
    private static final Pattern SHOW_STEPS = Pattern.compile(".*TGR (zeige|show) ([\\w|ü|ß]*) (Banner|banner|text|Text) \"(.*)\"");
    private static boolean initialized = false;

    @Getter
    private static TigerLibConfig libConfig;
    private static ConfigurableApplicationContext envMgrApplicationContext;

    public static void start() {
        if (initialized) {
            log.info("Tiger Director already started, skipping");
            return;
        }

        showTigerBanner();

        readConfiguration();
        applyTestLibConfig();
        startMonitorUi();
        startTestEnvMgr();
        setDefaultProxyToLocalTigerProxy();

        initialized = true;
        log.info("\n" + Banner.toBannerStr("DIRECTOR STARTUP OK", RbelAnsiColors.GREEN_BOLD.toString()));
    }

    private static synchronized void readConfiguration() {
        libConfig = TigerGlobalConfiguration.instantiateConfigurationBean(TigerLibConfig.class, "TIGER_LIB");
    }

    private static void showTigerBanner() {
        if (!TigerGlobalConfiguration.readBoolean("TIGER_NOLOGO", false)) {
            try {
                log.info("\n" + IOUtils.toString(
                    Objects.requireNonNull(TigerDirector.class.getResourceAsStream("/tiger2-logo.ansi")),
                    StandardCharsets.UTF_8));
            } catch (IOException e) {
                throw new TigerStartupException("Unable to read tiger logo!");
            }
        }
    }

    private static void applyTestLibConfig() {
        if (libConfig.isRbelPathDebugging()) {
            RbelOptions.activateRbelPathDebugging();
        } else {
            RbelOptions.deactivateRbelPathDebugging();
        }
        if (libConfig.isRbelAnsiColors()) {
            RbelOptions.activateAnsiColors();
        } else {
            RbelOptions.deactivateAnsiColors();
        }
    }

    private static synchronized void startMonitorUi() {
        if (libConfig.activateMonitorUI) {
            try {
                optionalMonitorUI = MonitorUI.getMonitor();
            } catch (HeadlessException hex) {
                log.error("Unable to start Monitor UI on a headless server!", hex);
            }
        }
    }

    private static synchronized void startTestEnvMgr() {
        log.info("\n" + Banner.toBannerStr("STARTING TESTENV MGR...", RbelAnsiColors.BLUE_BOLD.toString()));
        envMgrApplicationContext = new SpringApplicationBuilder()
            .properties(Map.of("server.port",
                TigerGlobalConfiguration.readIntegerOptional("free.port.255").orElse(0)))
            .sources(TigerTestEnvMgrApplication.class)
            .web(WebApplicationType.SERVLET)
            .initializers()
            .run();

        tigerTestEnvMgr = envMgrApplicationContext.getBean(TigerTestEnvMgr.class);
    }

    private static synchronized void setDefaultProxyToLocalTigerProxy() {
        TigerProxyConfiguration tpCfg = tigerTestEnvMgr.getConfiguration().getTigerProxy();
        // set proxy to local tiger proxy for test suites
        if (tigerTestEnvMgr.isLocalTigerProxyActive()) {
            if (System.getProperty("http.proxyHost") != null || System.getProperty("https.proxyHost") != null) {
                log.info(Ansi.colorize("SKIPPING TIGER PROXY settings as System Property is set already...",
                    RbelAnsiColors.RED_BOLD));
            } else {
                log.info(Ansi.colorize("SETTING TIGER PROXY...", RbelAnsiColors.BLUE_BOLD));
                System.setProperty("http.proxyHost", "localhost");
                System.setProperty("http.proxyPort", "" + tigerTestEnvMgr.getLocalTigerProxy().getPort());
                System.setProperty("http.nonProxyHosts", "localhost|127.0.0.1");
                System.setProperty("https.proxyHost", "localhost");
                System.setProperty("https.proxyPort", "" + tigerTestEnvMgr.getLocalTigerProxy().getPort());
                SerenityRestUtils.setupSerenityRest();
            }
        } else {
            log.info(Ansi.colorize("SKIPPING TIGER PROXY settings...", RbelAnsiColors.RED_BOLD));
        }

        // TODO TGR-295 DO NOT DELETE!
        // set proxy to local tigerproxy for erezept idp client
        // Unirest.config().proxy("localhost", TigerDirector.getTigerTestEnvMgr().getLocalDockerProxy().getPort());
    }

    public static synchronized boolean isInitialized() {
        return initialized;
    }

    public static TigerTestEnvMgr getTigerTestEnvMgr() {
        assertThatTigerIsInitialized();
        return tigerTestEnvMgr;
    }

    public static void synchronizeTestCasesWithPolarion() {
        assertThatTigerIsInitialized();

        if (TigerGlobalConfiguration.readBoolean("TIGER_SYNC_TESTCASES", false)) {
            try {
                Method polarionToolBoxMain = Class.forName("de.gematik.polarion.toolbox.ToolBox")
                    .getDeclaredMethod("main", String[].class);
                String[] args = new String[]{"-m", "tcimp", "-dryrun"};
                // TODO TGR-251 - read from tiger-testlib.yaml or env vars values for -h -u -p -prj -aq -fd -f -bdd

                log.info("Syncing test cases with Polarion...");
                polarionToolBoxMain.invoke(null, (Object[]) args);
                log.info("Test cases synched with Polarion...");
            } catch (NoSuchMethodException | ClassNotFoundException e) {
                throw new TigerLibraryException("Unable to access Polarion Toolbox! "
                    + "Be sure to have it included in mvn dependencies.", e);
                // TODO TGR-258 add the mvn dependency lines to log output
            } catch (InvocationTargetException | IllegalAccessException e) {
                throw new TigerLibraryException("Unable to call Polarion Toolbox's main method!", e);
            }
        }
    }

    public static void createAfoRepoort() {
        assertThatTigerIsInitialized();
        // TODO TGR-259 (see architecture decision about pluggable (TGR-253)) create Aforeport and embedd it into serenity report
    }

    public static String getProxySettings() {
        assertThatTigerIsInitialized();
        if (tigerTestEnvMgr.getLocalTigerProxy() == null || !tigerTestEnvMgr.getConfiguration().isLocalProxyActive()) {
            return null;
        } else {
            return tigerTestEnvMgr.getLocalTigerProxy().getBaseUrl();
        }
    }

    public static void waitForQuit() {
        optionalMonitorUI.ifPresentOrElse(
            (monitor) -> monitor.waitForQuit("Tiger Testsuite"),
            () -> TigerTestEnvMgr.waitForQuit("Tiger Testsuite"));
        envMgrApplicationContext.close();
        await()
            .pollInterval(Duration.ofMillis(100))
            .atMost(Duration.ofDays(1))
            .until(() -> !envMgrApplicationContext.isRunning());
    }

    public static void updateStepInMonitor(Step step) {
        optionalMonitorUI.ifPresent((monitor) -> monitor.updateStep(step));

        var stepText = String.join("\n", step.getLines());

        Matcher m = SHOW_STEPS.matcher(stepText);
        if (m.find()) {
            tigerTestEnvMgr.receiveTestEnvUpdate(
                TigerStatusUpdate.builder()
                    .statusMessage(m.group(4))
                    .build());
        }
    }

    private static void assertThatTigerIsInitialized() {
        if (!initialized) {
            throw new TigerStartupException("Tiger test environment has not been initialized. "
                + "Did you call TigerDirector.beforeTestRun before starting test run?");
        }
    }
    public static boolean isSerenityAvailable() {
        return TigerDirector.isSerenityAvailable(false);
    }

    public static boolean isSerenityAvailable(boolean quiet) {
        try {
            Class.forName("net.serenitybdd.core.Serenity");
            return true;
        } catch (ClassNotFoundException e) {
            if (!quiet) {
                log.warn("Trying to use Serenity functionality, but Serenity BDD packages are not declared as runtime dependency.", e);
            }
            return false;
        }
    }

    public static void testUninitialize() {
        initialized = false;
        tigerTestEnvMgr = null;

        System.clearProperty("TIGER_TESTENV_CFGFILE");
        System.clearProperty("http.proxyHost");
        System.clearProperty("https.proxyHost");
        System.clearProperty("http.proxyPort");
        System.clearProperty("https.proxyPort");

        TigerGlobalConfiguration.reset();
    }
}
