/*
 * Copyright (c) 2023 gematik GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the License);
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an 'AS IS' BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package de.gematik.test.tiger.common.config;

import java.util.function.Function;
import java.util.function.Supplier;

/**
 * Holds local configuration. You can add values to it, but the configuration will only be active within the `execute`
 * statement. This class is not intended for threading!
 */
public class TigerScopedExecutor {

    private BasicTigerConfigurationSource scopedValueSource
        = new BasicTigerConfigurationSource(SourceType.SCOPE_LOCAL_CONTEXT);

    public TigerScopedExecutor withValue(String key, String value) {
        scopedValueSource.putValue(new TigerConfigurationKey(
                TigerGlobalConfiguration.resolvePlaceholders(key)),
            value);
        return this;
    }

    public void execute(Runnable runnable) {
        TigerGlobalConfiguration.addConfigurationSource(scopedValueSource);
        try {
            runnable.run();
        } finally {
            TigerGlobalConfiguration.removeConfigurationSource(scopedValueSource);
        }
    }

    public <T> T retrieve(Supplier<T> supplier) {
        TigerGlobalConfiguration.addConfigurationSource(scopedValueSource);
        try {
            return supplier.get();
        } finally {
            TigerGlobalConfiguration.removeConfigurationSource(scopedValueSource);
        }
    }
}
