/*
 * Copyright (c) 2021 gematik GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the License);
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an 'AS IS' BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package de.gematik.test.tiger.common.config;

import static org.assertj.core.api.Assertions.assertThat;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.module.SimpleModule;
import com.fasterxml.jackson.dataformat.yaml.YAMLFactory;
import java.io.File;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.nio.file.Path;
import java.util.Map;

import de.gematik.rbellogger.util.RbelPkiIdentity;
import de.gematik.test.tiger.common.pki.TigerPkiIdentityLoader;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.io.IOUtils;
import org.json.JSONArray;
import org.json.JSONObject;

/**
 * This helper class helps managing test suite configuration based on yaml config files.
 * <p>
 * First step is to use {@link #yamlToJson(String)} to create a JSON representation of the yaml config file. Now you can
 * optionally apply a template by calling {@link #applyTemplate(JSONArray, String, JSONArray, String)}. Then you can
 * overwrite yaml config values with env vars or system properties by calling {@link
 * #overwriteWithSysPropsAndEnvVars(String, String, JSONObject)}. Finally you can convert to your data structure config
 * class by calling {@link #jsonStringToConfig(String, Class)}.
 * <p>
 * For simple test configurations without templating you can use the instance method {@link
 * #yamlReadOverwriteToConfig(String, String, Class)}. This method also performs the overwriting of yaml config values
 * with env vars and system properties. Due to Java Generics restrictions you will need to instantiate an instance to
 * use this method.
 *
 * <p>The format of the environment variables looks exemplaric like:
 * <ul>
 * <li>TIGER_TESTENV_TIGERPROXY_PROXYLOGLEVEL</li>
 * <li>TIGER_TESTENV_TIGERPROXY_PORT</li>
 * <li>TIGER_TESTENV_SERVERS_0_TEMPLATE</li>
 * <li>TIGER_TESTENV_SERVERS_0_STARTUPTIMEOUTSEC</li>
 * <li>...</li>
 * </ul></p>
 * <p>For <b>Envrionmental variables:</b><br/>TIGER is the product name passed in as parameter to {@link #overwriteWithSysPropsAndEnvVars(String, String, JSONObject)}
 * and then separated by "_" the hierarchy walking down all properties / path nodes being uppercase.
 * Entries in Lists are indexed by integer value.
 * <p>For <b>System properties:</b><br/>
 * <ul>
 *     <li>tiger.testenv.tigerProxy.proxyLogLevel</li>
 *     <li>tiger.testenv.tigerProxy.port</li>
 *     <li>tiger.testenv.tigerProxy.servers.0.template</li>
 *     <li>tiger.testenv.tigerProxy.servers.0.startupTimeoutSec</li>
 * </ul>
 * <p>
 * To use tokens such as ${TESTENV.xxxx} in the yaml file and replace it with appropriate values, first convert
 * the JSON Object to string and use the {@link de.gematik.test.tiger.common.TokenSubstituteHelper#substitute(String, String, Map)}
 * method to replace all tokens. Afterwards convert it back to JSONObject.
 *
 * @param <T>
 */
@Slf4j
public class TigerConfigurationHelper<T> {

    private static final ObjectMapper yamlMapper = new ObjectMapper(new YAMLFactory());
    private static final ObjectMapper objMapper = new ObjectMapper();

    /**
     * Old method, see {@link #yamlReadOverwriteToConfig(String, String, Class)}.
     *
     * @param yamlPath absolute path to yaml config file
     * @param product  name/id of product
     * @param cfgClazz class reference for the Configuration object to be created from config yaml file.
     * @return Configuration object
     * @deprecated
     */
    public T yamlToConfig(String yamlPath, String product, Class<T> cfgClazz) {
        return yamlReadOverwriteToConfig(yamlPath, product, cfgClazz);
    }

    /**
     * reads given yaml file to JSON object, applies env var / system property overwrite and returns the configuration
     * class's instance.
     *
     * @param yamlPath path to yaml config file
     * @param product  name/id of product
     * @param cfgClazz class reference for the Configuration object to be created from config yaml file.
     * @return Configuration object
     */
    @SneakyThrows
    public T yamlReadOverwriteToConfig(String yamlPath, String product, Class<T> cfgClazz) {
        JSONObject json = yamlToJson(yamlPath);
        overwriteWithSysPropsAndEnvVars(product.toUpperCase(), product, json);
        return objMapper.readValue(json.toString(), cfgClazz);
    }

    /**
     * reads a given yaml file to JSON object.
     *
     * @param yamlPath path to yaml config file
     * @return json object
     */
    @SneakyThrows
    public static JSONObject yamlToJson(String yamlPath) {
        Object yamlCfg = yamlMapper.
            readValue(IOUtils.toString(Path.of(yamlPath).toUri(), StandardCharsets.UTF_8), Object.class);
        return new JSONObject(objMapper.writeValueAsString(yamlCfg));
    }

    /**
     * converts a given yaml content string to JSON object.
     * @param yamlStr
     * @return
     */
    @SneakyThrows
    public static JSONObject yamlStringToJson(String yamlStr) {
        Object yamlCfg = yamlMapper.readValue(yamlStr, Object.class);
        return new JSONObject(objMapper.writeValueAsString(yamlCfg));
    }

    @SneakyThrows
    public static JSONObject yamlConfigReadOverwriteToJson(String yamlBaseFilename, String product) {
        final File cfgFile = Path.of("config", product, yamlBaseFilename).toFile();
        final String yamlStr;
        if (cfgFile.canRead()) {
            yamlStr = IOUtils.toString(cfgFile.toURI(), StandardCharsets.UTF_8);
        } else {
            InputStream is = TigerConfigurationHelper.class.getResourceAsStream(
                "/config/" + product + "/" + yamlBaseFilename);
            assertThat(is)
                .withFailMessage("Configuration file '" + cfgFile.getAbsolutePath()
                    + "' neither found in file system nor in classpath!")
                .isNotNull();
            try (is) {
                yamlStr = IOUtils.toString(is, StandardCharsets.UTF_8);
            }
        }
        final JSONObject json = yamlStringToJson(yamlStr);
        overwriteWithSysPropsAndEnvVars(product.toUpperCase(), product, json);
        return json;
    }

    @SneakyThrows
    public T jsonToConfig(String jsonFile, Class<T> cfgClazz) {
        return objMapper.readValue(IOUtils.toString(Path.of(jsonFile).toUri(), StandardCharsets.UTF_8), cfgClazz);
    }

    @SneakyThrows
    public T jsonStringToConfig(String jsonStr, Class<T> cfgClazz) {
        return objMapper.readValue(jsonStr, cfgClazz);
    }

    @SneakyThrows
    public static String toJson(Object cfg) {
        return objMapper.writerWithDefaultPrettyPrinter().writeValueAsString(cfg);
    }

    @SneakyThrows
    public static String toYaml(Object cfg) {
        return yamlMapper.writerWithDefaultPrettyPrinter().writeValueAsString(cfg);
    }


    public static void overwriteWithSysPropsAndEnvVars(String rootEnv, String rootProps, JSONObject json) {
        json.keys().forEachRemaining(key -> {
            Object obj = json.get(key);
            if (obj instanceof JSONObject) {
                overwriteWithSysPropsAndEnvVars(rootEnv + "_" + key.toUpperCase(), rootProps + "." + key,
                    (JSONObject) obj);
            } else if (obj instanceof JSONArray) {
                overwriteWithSysPropsAndEnvVars(rootEnv + "_" + key.toUpperCase(), rootProps + "." + key,
                    (JSONArray) obj);
            } else {
                log.info("checking for env " + rootEnv + "_" + key.toUpperCase() + ":" + obj);
                String value = System
                    .getProperty(rootProps + "." + key, System.getenv(rootEnv + "_" + key.toUpperCase()));
                if (value != null) {
                    log.info("modifying " + rootEnv + "_" + key.toUpperCase() + ":" + obj + " with " + value);
                    json.put(key, value);
                }
            }
        });
    }

    @SneakyThrows
    public static void applyTemplate(JSONArray cfgArray, String templateKey, JSONArray templates,
        String templateIdKey) {
        for (var i = 0; i < cfgArray.length(); i++) {
            var json = cfgArray.getJSONObject(i);
            if (json.has(templateKey)) {
                var templateId = json.getString(templateKey);
                boolean foundTemplate = false;
                for (var j = 0; j < templates.length(); j++) {
                    var jsonTemplate = templates.getJSONObject(j);
                    if (jsonTemplate.getString(templateIdKey).equals(templateId)) {
                        jsonTemplate.keySet().stream()
                            .filter(key -> jsonTemplate.get(key) != null)
                            .filter(key -> jsonTemplate.get(key) instanceof JSONArray)
                            .filter(key -> !jsonTemplate.getJSONArray(key).isEmpty())
                            .filter(key -> !json.has(key) || json.get(key) == null || json.getJSONArray(key).isEmpty())
                            .forEach(key -> json.put(key, new JSONArray(jsonTemplate.getJSONArray(key))));
                        jsonTemplate.keySet().stream()
                            .filter(key -> jsonTemplate.get(key) != null)
                            .filter(key -> !(jsonTemplate.get(key) instanceof JSONArray))
                            .filter(key -> !json.has(key) || json.get(key) == null)
                            .forEach(key -> json.put(key, jsonTemplate.get(key)));
                        foundTemplate = true;
                    }
                }
                if (!foundTemplate) {
                    throw new TigerConfigurationException("Unable to locate template '" + templateId + "'");
                }
            }
        }
    }

    private static void overwriteWithSysPropsAndEnvVars(String rootEnv, String rootProps, JSONArray jsonArray) {
        for (var i = 0; i < jsonArray.length(); i++) {
            Object obj = jsonArray.get(i);
            if (obj instanceof JSONObject) {
                overwriteWithSysPropsAndEnvVars(rootEnv + "_" + i, rootProps + "." + i, (JSONObject) obj);
            } else if (obj instanceof JSONArray) {
                overwriteWithSysPropsAndEnvVars(rootEnv + "_" + i, rootProps + "." + i, (JSONArray) obj);
            } else {
                log.info("checking for env " + rootEnv + "_" + i + ":" + obj);
                String value = System.getProperty(rootProps + "." + i, System.getenv(rootEnv + "_" + i));
                if (value != null) {
                    log.info("modifying " + rootEnv + "_" + i + ":" + obj + " with " + value);
                    jsonArray.put(i, value);
                }
            }
        }
    }
}
