/*
 * Copyright (c) 2022 gematik GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the License);
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an 'AS IS' BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package de.gematik.test.tiger.aforeporter;

import de.gematik.idp.tests.aforeport.AfoReporter;
import java.io.File;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import org.apache.commons.io.FileUtils;
import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;


@Mojo(name = "aforeport")
public class AfoReporterMavenPlugin extends AbstractMojo {

    @Parameter(property = "aforeport.serenityRootFolder", defaultValue = "target/site/serenity")
    String serenityRootFolder;

    @Parameter(property = "aforeport.bddRootFolder", defaultValue = "src/test/resources/features")
    String bddRootFolder;

    @Parameter(property = "aforeport.afoFile", defaultValue = "requirements.json")
    String afoFile;

    @Parameter(property = "aforeport.reportHtmlFile", defaultValue = "target/site/serenity/aforeport.html")
    String reportHtmlFile;

    public void execute() throws MojoExecutionException {
        try {
            AfoReporter.main(new String[]{"-bdd",
                "-rr", serenityRootFolder, "-tr", bddRootFolder, "-f", afoFile, "-o", reportHtmlFile});
        } catch (Exception e) {
            throw new MojoExecutionException("Failed to create afo report!", e);
        }
        try {
            File indexFile = new File(serenityRootFolder + "/index.html");
            String serenityIndex = FileUtils
                .readFileToString(indexFile, StandardCharsets.UTF_8);
            if (!serenityIndex.contains("Afo Überdeckung")) {
                getLog().info("Adding Afo Tab to Serenity Report...");
                final int ul = serenityIndex.indexOf("<ul class=\"nav nav-tabs\" role=\"tablist\">");
                final int ulend = serenityIndex.indexOf("</ul>", ul);
                serenityIndex =
                    serenityIndex.substring(0, ulend)
                        + "\n        "
                        + "<li><a href='aforeport.html'> <img style='height:16px;vertical-align: top;filter: sepia(100%) saturate(200%) brightness(80%) hue-rotate(180deg);' src=\"data:image/png;base64,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\" alt=\"Logo\">"
                        + " Afo Überdeckung </a></li>"
                        + "\n    " + serenityIndex.substring(ulend);
                FileUtils.writeStringToFile(indexFile, serenityIndex, StandardCharsets.UTF_8);
            }
        } catch (final IOException e) {
            getLog().error("Adding Afo tab to Serenity failed", e);
            // Don't throw anything as adding the afo report should not break the build
        }
    }
}
