/*
 * Copyright 2025 gematik GmbH
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package de.gematik.test.erezept.fhir.r4.erp;

import ca.uhn.fhir.model.api.annotation.ResourceDef;
import de.gematik.bbriccs.fhir.coding.WithSystem;
import de.gematik.bbriccs.fhir.coding.exceptions.MissingFieldException;
import de.gematik.test.erezept.fhir.parser.profiles.definitions.ErpWorkflowStructDef;
import de.gematik.test.erezept.fhir.parser.profiles.definitions.KbvItaErpStructDef;
import de.gematik.test.erezept.fhir.r4.kbv.KbvErpBundle;
import java.util.Optional;
import lombok.extern.slf4j.Slf4j;
import org.hl7.fhir.r4.model.Bundle;
import org.hl7.fhir.r4.model.ResourceType;

@Slf4j
@ResourceDef(name = "Bundle")
@SuppressWarnings({"java:S110"})
public class ErxPrescriptionBundle extends Bundle {

  public ErxTask getTask() {
    return this.getEntry().stream()
        .map(BundleEntryComponent::getResource)
        .filter(resource -> resource.getResourceType().equals(ResourceType.Task))
        .map(ErxTask::fromTask)
        .findFirst()
        .orElseThrow(
            () ->
                new MissingFieldException(ErxPrescriptionBundle.class, ErpWorkflowStructDef.TASK));
  }

  public Optional<KbvErpBundle> getKbvBundle() {
    return this.getEntry().stream()
        .map(BundleEntryComponent::getResource)
        .filter(resource -> resource.getResourceType().equals(ResourceType.Bundle))
        .filter(KbvItaErpStructDef.BUNDLE::matches)
        .map(KbvErpBundle.class::cast)
        .findFirst();
  }

  /**
   * This receipt is only available if the Task is in status complete and this bundle was retrieved
   * by a pharmacy with a secret (A_19226)
   *
   * @return an optional ErxReceipt
   */
  public Optional<ErxReceipt> getReceipt() {
    return this.getEntry().stream()
        .map(BundleEntryComponent::getResource)
        .filter(resource -> resource.getResourceType().equals(ResourceType.Bundle))
        .filter(
            resource ->
                WithSystem.anyOf(
                        ErpWorkflowStructDef.RECEIPT, ErpWorkflowStructDef.GEM_ERP_PR_BUNDLE)
                    .matches(resource))
        .map(ErxReceipt::fromBundle)
        .findFirst();
  }
}
