/*
 * Decompiled with CFR 0.152.
 */
package de.gematik.rbellogger.captures;

import com.sun.jna.Platform;
import de.gematik.rbellogger.captures.RbelCapturer;
import de.gematik.rbellogger.converter.RbelConverter;
import de.gematik.rbellogger.data.RbelElement;
import java.io.ByteArrayOutputStream;
import java.io.EOFException;
import java.io.File;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.TimeoutException;
import java.util.stream.Stream;
import org.apache.commons.lang3.math.NumberUtils;
import org.pcap4j.core.BpfProgram;
import org.pcap4j.core.NotOpenException;
import org.pcap4j.core.PacketListener;
import org.pcap4j.core.PcapDumper;
import org.pcap4j.core.PcapHandle;
import org.pcap4j.core.PcapNativeException;
import org.pcap4j.core.PcapNetworkInterface;
import org.pcap4j.core.PcapStat;
import org.pcap4j.core.Pcaps;
import org.pcap4j.packet.Packet;
import org.pcap4j.packet.TcpPacket;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class PCapCapture
extends RbelCapturer {
    private static final Logger log = LoggerFactory.getLogger(PCapCapture.class);
    private String deviceName;
    private String pcapFile;
    private String filter;
    private boolean printMessageToSystemOut;
    private Thread captureThread;
    private PcapHandle handle;
    private PcapDumper dumper;
    private List<TcpPacket> unhandledTcpRequests = new ArrayList<TcpPacket>();
    private List<TcpPacket> unhandledTcpResponses = new ArrayList<TcpPacket>();
    private int tcpServerPort = -1;

    public PCapCapture(RbelConverter rbelConverter, String deviceName, String pcapFile, String filter, boolean printMessageToSystemOut) {
        super(rbelConverter);
        this.deviceName = deviceName;
        this.pcapFile = pcapFile;
        this.filter = filter;
        this.printMessageToSystemOut = printMessageToSystemOut;
    }

    private static void setWindowsNpcapPath() {
        if (System.getProperty("os.name").startsWith("Windows")) {
            Object prop = System.getProperty("jna.library.path");
            prop = prop == null || ((String)prop).isEmpty() ? "C:/Windows/System32/Npcap" : (String)prop + ";C:/Windows/System32/Npcap";
            System.setProperty("jna.library.path", (String)prop);
        }
    }

    @Override
    public RbelCapturer initialize() {
        PCapCapture.setWindowsNpcapPath();
        if (this.pcapFile != null) {
            this.getOfflinePcapHandle();
        } else if (this.deviceName != null) {
            this.getOnlineHandle();
        } else {
            throw new IllegalArgumentException("Either device or pcap file must be specified");
        }
        if (!this.handle.isOpen()) {
            throw new RuntimeException("Source not open for reading!");
        }
        if (this.filter == null) {
            this.filter = "host 127.0.0.1 and tcp port 8080";
        }
        log.info("Applying filter '" + this.filter + "'");
        RBelPacketListener packetListener = new RBelPacketListener(this.handle, this.dumper);
        if (this.pcapFile != null) {
            try {
                while (true) {
                    packetListener.gotPacket(this.handle.getNextPacketEx());
                    log.trace("Read-In loop. Currently there are {} request and {} response TCP-Packets in their respective buffers.", (Object)this.unhandledTcpRequests.size(), (Object)this.unhandledTcpResponses.size());
                }
            }
            catch (EOFException e) {
                log.info("Reached EOF");
            }
            catch (TimeoutException | NotOpenException | PcapNativeException e) {
                throw new RuntimeException(e);
            }
            log.info("After loop");
        } else {
            this.captureThread = new Thread(() -> {
                try {
                    this.handle.setFilter(this.filter, BpfProgram.BpfCompileMode.OPTIMIZE);
                    int maxPackets = -1;
                    this.handle.loop(-1, (PacketListener)packetListener);
                }
                catch (InterruptedException e) {
                    log.info("Packet capturing interrupted...");
                }
                catch (NotOpenException | PcapNativeException e) {
                    throw new RuntimeException(e);
                }
            });
            this.captureThread.start();
        }
        return this;
    }

    private void getOnlineHandle() {
        try {
            this.getLivePcapHandle();
            this.dumper = this.handle.dumpOpen("out.pcap");
        }
        catch (NotOpenException | PcapNativeException e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    public void close() {
        if (this.pcapFile != null) {
            this.initialize();
            return;
        }
        try {
            this.captureThread.join();
        }
        catch (InterruptedException interruptedException) {
            // empty catch block
        }
        try {
            this.handle.breakLoop();
        }
        catch (NotOpenException notOpenException) {
            // empty catch block
        }
        this.tryToPrintStats();
        this.handle.close();
        if (this.dumper != null) {
            this.dumper.close();
        }
    }

    private void tryToPrintStats() {
        if (this.deviceName != null && this.printMessageToSystemOut) {
            try {
                PcapStat stats = this.handle.getStats();
                log.info("Packets received: " + stats.getNumPacketsReceived());
                log.info("Packets dropped: " + stats.getNumPacketsDropped());
                log.info("Packets dropped by interface: " + stats.getNumPacketsDroppedByIf());
                if (Platform.isWindows()) {
                    log.info("Packets captured: " + stats.getNumPacketsCaptured());
                }
            }
            catch (Exception exception) {
                // empty catch block
            }
        }
    }

    private void getLivePcapHandle() {
        log.info("Capturing traffic live from device " + this.deviceName);
        PcapNetworkInterface device = Pcaps.getDevByName((String)this.deviceName);
        int snapshotLength = 65536;
        int readTimeout = 50;
        this.handle = device.openLive(65536, PcapNetworkInterface.PromiscuousMode.PROMISCUOUS, 50);
    }

    private void getOfflinePcapHandle() {
        log.info("Reading traffic from pcap file " + new File(this.pcapFile).getAbsolutePath());
        try {
            this.handle = Pcaps.openOffline((String)this.pcapFile, (PcapHandle.TimestampPrecision)PcapHandle.TimestampPrecision.NANO);
        }
        catch (PcapNativeException e) {
            try {
                this.handle = Pcaps.openOffline((String)this.pcapFile);
            }
            catch (PcapNativeException e1) {
                throw new RuntimeException(e1);
            }
        }
    }

    private byte[] getCurrentBuffer(List<TcpPacket> requestList) {
        ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
        requestList.stream().sorted(Comparator.comparing(tcpPacket -> tcpPacket.getHeader().getSequenceNumber())).map(Packet::getPayload).map(Packet::getRawData).forEach(data -> {
            try {
                outputStream.write((byte[])data);
            }
            catch (IOException iOException) {
                // empty catch block
            }
        });
        return outputStream.toByteArray();
    }

    public static PCapCaptureBuilder builder() {
        return new PCapCaptureBuilder();
    }

    public static class PCapCaptureBuilder {
        private RbelConverter rbelConverter;
        private String deviceName;
        private String pcapFile;
        private String filter;
        private boolean printMessageToSystemOut;

        PCapCaptureBuilder() {
        }

        public PCapCaptureBuilder rbelConverter(RbelConverter rbelConverter) {
            this.rbelConverter = rbelConverter;
            return this;
        }

        public PCapCaptureBuilder deviceName(String deviceName) {
            this.deviceName = deviceName;
            return this;
        }

        public PCapCaptureBuilder pcapFile(String pcapFile) {
            this.pcapFile = pcapFile;
            return this;
        }

        public PCapCaptureBuilder filter(String filter) {
            this.filter = filter;
            return this;
        }

        public PCapCaptureBuilder printMessageToSystemOut(boolean printMessageToSystemOut) {
            this.printMessageToSystemOut = printMessageToSystemOut;
            return this;
        }

        public PCapCapture build() {
            return new PCapCapture(this.rbelConverter, this.deviceName, this.pcapFile, this.filter, this.printMessageToSystemOut);
        }

        public String toString() {
            return "PCapCapture.PCapCaptureBuilder(rbelConverter=" + this.rbelConverter + ", deviceName=" + this.deviceName + ", pcapFile=" + this.pcapFile + ", filter=" + this.filter + ", printMessageToSystemOut=" + this.printMessageToSystemOut + ")";
        }
    }

    class RBelPacketListener
    implements PacketListener {
        private static final String CONTENT_LENGTH_HEADER_START = "Content-Length: ";
        private final PcapHandle handle;
        private final PcapDumper dumper;

        public void gotPacket(Packet packet) {
            Optional<TcpPacket> tcpPacket = this.extractTcpPacket(packet);
            if (PCapCapture.this.tcpServerPort == -1 && tcpPacket.get().getHeader().getSyn() && !tcpPacket.get().getHeader().getAck()) {
                PCapCapture.this.tcpServerPort = tcpPacket.get().getHeader().getDstPort().valueAsInt();
            }
            if (tcpPacket.isEmpty() || tcpPacket.get().getPayload() == null) {
                return;
            }
            if (tcpPacket.get().getHeader().getDstPort().valueAsInt() == PCapCapture.this.tcpServerPort) {
                this.addToBufferAndExtractCompletedMessages(tcpPacket, PCapCapture.this.unhandledTcpRequests);
            } else {
                this.addToBufferAndExtractCompletedMessages(tcpPacket, PCapCapture.this.unhandledTcpResponses);
            }
            try {
                if (this.dumper != null) {
                    this.dumper.dump(packet, this.handle.getTimestamp());
                }
            }
            catch (NotOpenException e) {
                e.printStackTrace();
            }
        }

        private void addToBufferAndExtractCompletedMessages(Optional<TcpPacket> tcpPacket, List<TcpPacket> buffer) {
            buffer.add(tcpPacket.get());
            Optional<String> nextMessage = this.extractCompleteHttpMessage(PCapCapture.this.getCurrentBuffer(buffer));
            if (nextMessage.isPresent()) {
                this.processSimpleHttpPackets(nextMessage.get());
                buffer.clear();
            }
        }

        private Optional<String> extractCompleteHttpMessage(byte[] currentBuffer) {
            String dumpString = new String(currentBuffer, StandardCharsets.US_ASCII);
            if (!this.isHttp(dumpString)) {
                log.trace("No HTTP-message recognized, skipping");
                return Optional.empty();
            }
            String[] messageParts = dumpString.split("\r\n\r\n");
            String[] headerFields = messageParts[0].split("\r\n");
            Optional<Integer> messageLength = Stream.of(headerFields).filter(field -> field.startsWith(CONTENT_LENGTH_HEADER_START)).map(field -> field.substring(CONTENT_LENGTH_HEADER_START.length())).filter(NumberUtils::isParsable).map(Integer::parseInt).findAny();
            if (messageLength.isPresent()) {
                if (messageParts.length < 2) {
                    if (messageLength.isEmpty() || messageLength.get() == 0) {
                        return Optional.of(dumpString);
                    }
                    log.trace("Header found, body segmented away. \n'{}'", (Object)dumpString);
                    return Optional.empty();
                }
                if (messageParts[1].length() == messageLength.get().intValue()) {
                    return Optional.of(dumpString);
                }
                if (messageParts[1].length() > messageLength.get()) {
                    throw new RuntimeException("Overshot while parsing message (collected more bytes then the message has)");
                }
                log.trace("Message not yet complete. Wanted {} bytes, but found only {}", (Object)messageLength.get(), (Object)messageParts[1].length());
                return Optional.empty();
            }
            boolean chunked = Arrays.asList(headerFields).contains("Transfer-Encoding: chunked");
            if (!chunked) {
                log.trace("Returning (hopefully) body-less message");
                return Optional.of(dumpString);
            }
            if (!dumpString.endsWith("0\r\n\r\n")) {
                log.trace("Chunked message, incomplete");
                return Optional.empty();
            }
            log.trace("Returning chunked message");
            return Optional.ofNullable(dumpString);
        }

        private Optional<TcpPacket> extractTcpPacket(Packet packet) {
            Packet ptr;
            for (ptr = packet; ptr != null && !(ptr instanceof TcpPacket); ptr = ptr.getPayload()) {
            }
            if (ptr instanceof TcpPacket) {
                return Optional.ofNullable((TcpPacket)ptr);
            }
            return Optional.empty();
        }

        private void processSimpleHttpPackets(String content) {
            if (content.contains("GET /EXIT_RBELLOGGER")) {
                this.handle.breakLoop();
                return;
            }
            RbelElement convertMessage = PCapCapture.this.getRbelConverter().convertMessage(content);
            if (PCapCapture.this.printMessageToSystemOut && convertMessage != null && !content.isEmpty()) {
                if (convertMessage.getContent() != null) {
                    log.info("RBEL: " + convertMessage.getContent());
                } else {
                    log.info("RBEL: <null> message encountered!");
                }
            }
        }

        private boolean isHttp(String content) {
            return this.isHttpRequest(content) || this.isHttpResponse(content);
        }

        private boolean isHttpResponse(String content) {
            return content.startsWith("HTTP/");
        }

        private boolean isHttpRequest(String content) {
            return this.isGetOrDeleteRequest(content) || this.isPostOrPutRequest(content);
        }

        private boolean isGetOrDeleteRequest(String content) {
            return content.startsWith("GET ") || content.startsWith("DELETE ");
        }

        private boolean isPostOrPutRequest(String content) {
            return content.startsWith("POST ") || content.startsWith("PUT ");
        }

        public RBelPacketListener(PcapHandle handle, PcapDumper dumper) {
            this.handle = handle;
            this.dumper = dumper;
        }
    }
}

