/*
 * Copyright 2024 gematik GmbH
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package de.gematik.bbriccs.fhir.fuzzing.impl.mutators.types;

import de.gematik.bbriccs.fhir.fuzzing.FhirTypeMutatorProvider;
import de.gematik.bbriccs.fhir.fuzzing.FuzzingMutator;
import de.gematik.bbriccs.fhir.fuzzing.Randomness;
import java.util.LinkedList;
import java.util.List;
import lombok.Getter;
import lombok.val;
import org.hl7.fhir.r4.model.Population;

@Getter
public class PopulationMutatorProvider implements FhirTypeMutatorProvider<Population> {

  private final List<FuzzingMutator<Population>> mutators;

  public PopulationMutatorProvider() {
    this.mutators = createMutators();
  }

  private static List<FuzzingMutator<Population>> createMutators() {
    val mutators = new LinkedList<FuzzingMutator<Population>>();
    mutators.add((ctx, population) -> ctx.fuzzIdElement(Population.class, population));
    mutators.add(
        (ctx, population) ->
            ctx.fuzzChildTypes(
                Population.class, ensureNotNull(ctx.randomness(), population).getExtension()));

    mutators.add(
        (ctx, population) ->
            ctx.fuzzChildTypes(
                Population.class,
                ensureNotNull(ctx.randomness(), population).getModifierExtension()));

    mutators.add(
        (ctx, population) ->
            ctx.fuzzChild(Population.class, ensureNotNull(ctx.randomness(), population).getAge()));

    mutators.add(
        (ctx, population) ->
            ctx.fuzzChild(
                Population.class, ensureNotNull(ctx.randomness(), population).getGender()));

    mutators.add(
        (ctx, population) ->
            ctx.fuzzChild(Population.class, ensureNotNull(ctx.randomness(), population).getRace()));

    mutators.add(
        (ctx, population) ->
            ctx.fuzzChild(
                Population.class,
                ensureNotNull(ctx.randomness(), population).getPhysiologicalCondition()));

    return mutators;
  }

  private static Population ensureNotNull(Randomness randomness, Population population) {
    if (population == null) {
      population = randomness.fhir().createType(Population.class);
    }

    return population;
  }
}
