/*
 * Copyright (C) 2023 Fraunhofer Institut IOSB, Fraunhoferstr. 1, D 76131
 * Karlsruhe, Germany.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
package de.fraunhofer.iosb.ilt.frostclient.model.property;

import de.fraunhofer.iosb.ilt.frostclient.model.EntityType;
import de.fraunhofer.iosb.ilt.frostclient.model.property.type.TypeComplex;
import de.fraunhofer.iosb.ilt.frostclient.model.property.type.TypeEntity;
import de.fraunhofer.iosb.ilt.frostclient.model.property.type.TypeEntitySet;
import java.util.Objects;

/**
 * @param <P> The entityType of the value of the property.
 */
public abstract class NavigationPropertyAbstract<P> extends PropertyAbstract<P> implements NavigationProperty<P> {

    /**
     * The entityType of entity that this navigation property points to.
     */
    private EntityType entityType;

    /**
     * Flag indication the path is to an EntitySet.
     */
    private final boolean entitySet;

    private NavigationPropertyAbstract<?> inverse;

    protected NavigationPropertyAbstract(String propertyName, boolean isSet) {
        super(propertyName, TypeComplex.STA_OBJECT, false);
        this.entitySet = isSet;
    }

    public void setEntityType(EntityType entityType) {
        this.entityType = entityType;
        if (entitySet) {
            setType(new TypeEntitySet(entityType));
        } else {
            setType(new TypeEntity(entityType));
        }
    }

    @Override
    public NavigationPropertyAbstract getInverse() {
        return inverse;
    }

    /**
     * Set the inverse to the given NP. Does not set the inverse of the given NP
     * to this.
     *
     * @param inverse The inverse to set.
     */
    public final void setInverse(NavigationPropertyAbstract inverse) {
        this.inverse = inverse;
    }

    /**
     * Set the inverse of this NP to to given NP, and set the inverse of the
     * given NP to this NP.
     *
     * @param inverse The inverse to set.
     */
    public final void setInverses(NavigationPropertyAbstract inverse) {
        this.inverse = inverse;
        inverse.setInverse(this);
    }

    @Override
    public EntityType getEntityType() {
        return entityType;
    }

    @Override
    public boolean isEntitySet() {
        return entitySet;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }
        final NavigationPropertyAbstract<?> other = (NavigationPropertyAbstract<?>) obj;
        return Objects.equals(getName(), other.getName());
    }

    @Override
    public int hashCode() {
        return Objects.hashCode(getName());
    }

}
