/*-
 * =================================LICENSE_START=================================
 * IND2UCE
 * %%
 * Copyright (C) 2016 Fraunhofer IESE (www.iese.fraunhofer.de)
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =================================LICENSE_END=================================
 */

package de.fraunhofer.iese.ind2uce.registry;

import de.fraunhofer.iese.ind2uce.api.component.description.MethodInterfaceDescription;
import de.fraunhofer.iese.ind2uce.api.component.identifier.ComponentId;
import de.fraunhofer.iese.ind2uce.api.component.identifier.EnforcementScopeId;
import de.fraunhofer.iese.ind2uce.connectors.ConnectorFactory;
import de.fraunhofer.iese.ind2uce.connectors.OAuthCredentials;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.lang.reflect.Method;
import java.net.URI;
import java.util.Map;

/**
 * The Class ComponentRegistry is able to register a PIP or PXP Object at
 * PMP.
 * Component Registry uses the {@link InterfaceDescriptionDiscovery} to find all
 * Component related Methods, that can be registered as
 * {@link MethodInterfaceDescription} at PMP. Therefor the discovery lists all
 * methods that is annotated with an Annotation of Type
 * {@link ActionDescription} and creates an instance of
 * {@link MethodInterfaceDescription} with the information provided by the
 * MethodSignature, the annotation and the {@link ActionParameterDescription}
 * annotations of each method parameter.
 * For Example The Method
 *
 * <pre>
 *
 *
 * {@literal @}ActionDescription(description = "Retreives the authority (role) of an user")
 *  public String getAuthority(@ActionParameterDescription(name = "username", description = "The user the authority should be evaluated for.", mandatory = true) String username) {
 *  }
 * </pre>
 *
 * Results in an InterfaceDescription with:
 * <ul>
 * <li>name = getAuthority</li>
 * <li>description = Retreives the authority (role) of an user</li>
 * <li>parameter =
 * <ul>
 * <li>InputParameterDescription: name = username, description = The user the
 * authority should be evaluated for, mandatory = true</li>
 * </ul>
 * </li>
 * </ul>
 * The ComponentRegistry should be used like this:
 *
 * <pre class="code">
 * MyPipService serviceThatContainsMethodsWithActionDescription = ..;
 * ComponentRegistry registry = new ComponentRegistry(ComponentType.PIP,pmp);
 * boolean isRegistered = registry.componentId(new
 * ComponentId("urn:component:test:pip:1234"))
 * .addService(serviceThatContainsMethodsWithActionDescription)
 * .url("http://localhost:8008") .register(); </pre>
 */
public class ComponentRegistry {

  /**
   * The Constant LOG.
   */
  private static final Logger LOG = LoggerFactory.getLogger(ComponentRegistry.class);

  /**
   * The registry builder.
   */
  private final RegistryBuilder registryBuilder;

  /**
   * Discovery that creates MethodInterfaceDescriptions.
   */
  private final InterfaceDescriptionDiscovery discovery = new InterfaceDescriptionDiscovery();

  /**
   * The type.
   */
  private final ComponentType type;

  /**
   * The enforcement scope id.
   */
  private EnforcementScopeId enforcementScopeId;

  /**
   * Instantiates a new component registry.
   *
   * @param type the type
   * @param pmpUrl the pmp url
   * @param credentials OAuthCredentials used to connect
   */
  public ComponentRegistry(ComponentType type, URI pmpUrl, OAuthCredentials credentials) {
    this.type = type;
    this.registryBuilder = new RegistryBuilder(type, ConnectorFactory.getPmpClient(pmpUrl, credentials));
  }
  
  /**
   * Adds the service.
   *
   * @param componentService the component service
   * @return the map
   */
  public Map<MethodInterfaceDescription, Method> addService(Object componentService) {
    final Map<MethodInterfaceDescription, Method> interfaceDescriptions = this.createInterfaceDescriptions(componentService);
    this.registryBuilder.addAll(interfaceDescriptions.keySet());
    return interfaceDescriptions;
  }

  /**
   * Component id.
   *
   * @param componentId the component id
   * @return the component registry
   */
  public ComponentRegistry componentId(ComponentId componentId) {
    this.registryBuilder.componentId(componentId);
    return this;
  }

  /**
   * Component id.
   *
   * @param componentId the component id
   * @return the component registry
   */
  public ComponentRegistry componentId(String componentId) {
    this.registryBuilder.componentId(componentId);
    return this;
  }

  /**
   * Creates the interface descriptions.
   *
   * @param service the service
   * @return the map
   */
  private Map<MethodInterfaceDescription, Method> createInterfaceDescriptions(Object service) {
    return this.discovery.discover(service.getClass(), this.type, this.enforcementScopeId);

  }

  /**
   * Enforcement scope id.
   *
   * @param enforcementScopeId the enforcement scope id
   * @return the component registry
   */
  public ComponentRegistry enforcementScopeId(EnforcementScopeId enforcementScopeId) {
    this.enforcementScopeId = enforcementScopeId;
    return this;
  }

  /**
   * Register.
   *
   * @return true, if successful
   */
  public boolean register() {
    try {
      LOG.trace("Registering at PMP " + this.type);
      return this.registryBuilder.register();
    } catch (final Exception e) {
      LOG.warn("Component could not be registered. " + this.type, e);
      return false;
    }
  }

  /*
   * (non-Javadoc)
   * @see java.lang.Object#toString()
   */
  @Override
  public String toString() {
    return "ComponentRegistry{" + "type=" + this.type + '}';
  }

  /**
   * Url.
   *
   * @param url the url
   * @return the component registry
   */
  public ComponentRegistry url(String url) {
    this.registryBuilder.addUri(url);
    return this;
  }
}
