/*-
 * =================================LICENSE_START=================================
 * IND2UCE
 * %%
 * Copyright (C) 2016 Fraunhofer IESE (www.iese.fraunhofer.de)
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =================================LICENSE_END=================================
 */

package de.fraunhofer.iese.ind2uce.pep.modifiermethods;

import de.fraunhofer.iese.ind2uce.api.policy.parameter.ParameterList;
import de.fraunhofer.iese.ind2uce.pep.common.CommonUtil;
import de.fraunhofer.iese.ind2uce.pep.common.ModifierMethod;
import de.fraunhofer.iese.ind2uce.registry.ActionDescription;
import de.fraunhofer.iese.ind2uce.registry.ActionParameterDescription;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import com.google.gson.JsonPrimitive;
import com.jayway.jsonpath.Criteria;
import com.jayway.jsonpath.DocumentContext;
import com.jayway.jsonpath.Filter;
import com.jayway.jsonpath.MapFunction;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Random;
import java.util.Set;

/***
 * This a default Modifier method with PEP sdk which replaces a string with
 * another one where event parameter is either a Json Object or string
 * (primitive type)
 */
public class PatternMatcherModifierMethod implements ModifierMethod {
  /**
   * the Logger
   */
  private static final Logger LOG = LoggerFactory.getLogger(PatternMatcherModifierMethod.class);

  private static final String DEBTOR = "debtor";

  private static final String CREDITOR = "creditor";

  String name = "patternMatcher";

  private String parameterName;

  private final Set<String> idsToModify = new HashSet<>();

  @Override
  public DocumentContext doModification(DocumentContext documentContext, String expression, ParameterList modifierMethodParameterList) {
    final boolean resetList = (Boolean)modifierMethodParameterList.getParameterValueForName("resetList");

    if (resetList) {
      this.idsToModify.clear();
    }

    this.parameterName = (String)modifierMethodParameterList.getParameterValueForName("parameterName");

    @SuppressWarnings("unchecked")
    final Map<String, Integer> tuples = (Map<String, Integer>)modifierMethodParameterList.getParameterValueForName("tuples");

    final String method = (String)modifierMethodParameterList.getParameterValueForName("method");
    final String amountExpression = modifierMethodParameterList.getParameterValueForName("amountExpression").toString();
    final int percentage = (int)Math.round((Double)modifierMethodParameterList.getParameterValueForName("percentage"));

    return this.patternMatcher(tuples, amountExpression, percentage, documentContext, expression, method, resetList);
  }

  @Override
  public Object doModification(Object currentObject, ParameterList modifierMethodParameterList) {
    return null;

  }

  private final Gson gson = new GsonBuilder().setPrettyPrinting().create();

  /**
   * Domodify.
   *
   * @param tuples the tuples
   * @param amountExpression the json amount expression
   * @param percentage the percentage for the anagram
   * @param documentContext the document context
   * @param expression the expression to blur
   * @param method the method to apply
   * @return the document context
   */
  @ActionDescription(description = "Appends a prefix and/or suffix to an event parameter value", pepSupportedType = String.class)
  public DocumentContext patternMatcher(@ActionParameterDescription(name = "tuples", description = "tuples", mandatory = false) Object tuples,
      @ActionParameterDescription(name = "amountExpression", description = "amountExpression", mandatory = false) String amountExpression,
      @ActionParameterDescription(name = "percentage", description = "percentage", mandatory = false) Number percentage, DocumentContext documentContext, String expression,
      @ActionParameterDescription(name = "method", description = "method", mandatory = false) String method,
      @ActionParameterDescription(name = "resetList", description = "resetList", mandatory = false) Boolean resetList) {
    LOG.info("doModify PatternMatcherr with expression {} ", expression);

    if (resetList) {
      this.idsToModify.clear();
    }

    String pathFilter;
    String pathReader;
    try {
      documentContext.read("$['" + this.parameterName + "'][*]");
      pathFilter = "$['" + this.parameterName + "'][?]";
      pathReader = "$['" + this.parameterName + "'][*]";

    } catch (final Exception e) {
      LOG.info("Problem trying to read parameterName, try with *", e);
      documentContext.read("$[*]");
      pathFilter = "$[?]";
      pathReader = "$[*]";
    }
    final List<MapFunction> blurMethods = new ArrayList<>();

    final String methodName = method.toString().split("-")[0];
    String caracter = null;
    try {
      caracter = method.toString().split("-")[1];
    } catch (final Exception e) {
      LOG.debug("Problem trying to get 2nd element, try with DEFAULT", e);
      caracter = "DEFAULT";
    }
    if ((int)percentage > 0) {
      caracter += percentage + "%";
    }
    final String carac = caracter;

    if (((int)percentage) > 0) {
      blurMethods.add((o, configuration) -> this.anagram(o, (int)percentage).getAsString());
    } else {

      if (methodName.equalsIgnoreCase("replace") && caracter != null) {
        blurMethods.add((o, configuration) -> this.replace(carac).getAsString());
      }

    }
    if (methodName.equalsIgnoreCase("suffix") && caracter != null) {
      blurMethods.add((o, configuration) -> this.append(o, carac).getAsString());
    }
    JsonArray copy;
    Filter excludeFilter;

    JsonArray document;
    final Map<String, Double> tuplesMap = (Map<String, Double>)tuples;
    Double maxAmount;

    for (final Map.Entry<String, Double> mapEntry : tuplesMap.entrySet()) {
      final String mapKey = mapEntry.getKey();
      Filter expensiveFilter;
      if (mapEntry.getValue() != -1d) {
        maxAmount = Math.abs(mapEntry.getValue());

        expensiveFilter = Filter.filter(Criteria.where(amountExpression).gt(0).and(amountExpression).gt(maxAmount).and(DEBTOR).contains(mapKey))
            .or(Criteria.where(amountExpression).gt(0).and(amountExpression).gt(maxAmount).and(CREDITOR).contains(mapKey))
            .or(Criteria.where(amountExpression).lt(0).and(amountExpression).lt(maxAmount * -1).and(DEBTOR).contains(mapKey))
            .or(Criteria.where(amountExpression).lt(0).and(amountExpression).lt(maxAmount * -1).and(CREDITOR).contains(mapKey)).or(Criteria.where("id").in(this.idsToModify));
      } else {
        expensiveFilter = Filter.filter(Criteria.where(amountExpression).gt(0).and(DEBTOR).contains(mapKey)).or(Criteria.where(amountExpression).gt(0).and(CREDITOR).contains(mapKey))
            .or(Criteria.where(amountExpression).lt(0).and(DEBTOR).contains(mapKey)).or(Criteria.where(amountExpression).lt(0).and(CREDITOR).contains(mapKey))
            .or(Criteria.where("id").in(this.idsToModify));
      }
      final Object filtred = documentContext.read(pathFilter, expensiveFilter);
      final JsonArray filteredArray = (JsonArray)filtred;

      final List<String> ids = new ArrayList<>();
      for (final JsonElement jsonElement : filteredArray) {
        ids.add(jsonElement.getAsJsonObject().get("id").getAsString());
      }
      this.idsToModify.addAll(ids);
      excludeFilter = Filter.filter(Criteria.where("id").nin(ids));

      copy = documentContext.read(pathFilter, excludeFilter);
      documentContext.delete(pathReader);

      try {
        documentContext.put("$", this.parameterName, filteredArray);
      } catch (final Exception e) {
        LOG.info("Problem putting parameterName and filteredArray to documentContext, retry without parameterName", e);
        documentContext.add("$", filteredArray);
      }

      if (methodName.equalsIgnoreCase("delete") && caracter != null) {
        document = copy;
        documentContext.delete(pathReader);
      } else {
        for (final MapFunction blur : blurMethods) {
          documentContext = documentContext.map(expression, blur, expensiveFilter);
        }

        document = documentContext.read(pathReader);
        document.addAll(copy);
        documentContext.delete(pathReader);
      }
      try {
        documentContext.put("$", this.parameterName, document);
      } catch (final Exception e) {
        LOG.info("Problem putting parameterName and document to documentContext, retry without parameterName", e);
        documentContext.add("$", document);
      }

    }
    return documentContext;

  }

  private JsonElement append(Object input, Object suffix) {

    final String sInput = CommonUtil.getFromStringOrJsonPrimitive(input);

    final StringBuilder builder = new StringBuilder();

    builder.append(sInput);
    if (suffix != null && sInput.indexOf(suffix.toString()) < 0) {
      builder.append(suffix);
    }

    return this.gson.fromJson(this.gson.toJson(builder.toString()), JsonPrimitive.class);

  }

  private JsonElement replace(String caracter) {
    String json = caracter;
    try {
      if (!json.startsWith("{") && !json.startsWith("[") && !json.startsWith("\"")) {
        json = "" + json + "";
      }
    } catch (

    final NumberFormatException e) {
      if (!json.startsWith("{") && !json.startsWith("[") && !json.startsWith("\"")) {
        json = "'" + json + "'";
      }
    }

    return this.gson.fromJson(this.gson.toJson(json), JsonPrimitive.class);
  }

  private JsonElement anagram(Object input, int percentage) {

    final String sInput = CommonUtil.getFromStringOrJsonPrimitive(input);

    final String resultString = this.scramble(sInput, percentage);

    return this.gson.fromJson(this.gson.toJson(resultString), JsonPrimitive.class);
  }

  /**
   * @param inputString string to scramble
   * @param count count
   * @return Scrambled string
   */
  protected String scramble(String inputString, int count) {
    if (inputString == null || inputString.length() <= 2) {
      return inputString;
    }

    boolean canBeScrambled = false;
    for (int i = 1; i < inputString.length(); i++) {
      final char c = inputString.charAt(i);
      if (c != inputString.charAt(0)) {
        canBeScrambled = true;
        break;
      }
    }

    if (!canBeScrambled) {
      return inputString;
    }

    final Random random = new Random();
    final char[] a = inputString.toCharArray();

    final int limit = Math.round((a.length * (Math.min(count, 100) / 100f)));

    for (int i = 0; i < Math.min(a.length - 1, limit); i++) {
      final int j = random.nextInt(Math.min(a.length - 1, limit));
      final char temp = a[i];
      a[i] = a[j];
      a[j] = temp;
    }
    String result = new String(a);
    if (result.equals(inputString)) {
      result = this.scramble(inputString, count);
    }
    LOG.trace("Built anagram {}% of {} --> {}", count, inputString, result);
    return result;
  }

  @Override
  public String getDisplayName() {
    return this.name;
  }

  @Override
  public boolean nameIsValid() {
    final Method[] methods = this.getClass().getMethods();
    for (final Method m : methods) {
      if (m.getName().equalsIgnoreCase(this.getDisplayName())) {
        return true;
      }
    }
    return false;
  }
}
