/*-
 * =================================LICENSE_START=================================
 * IND2UCE
 * %%
 * Copyright (C) 2016 Fraunhofer IESE (www.iese.fraunhofer.de)
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =================================LICENSE_END=================================
 */

package de.fraunhofer.iese.ind2uce.pep.modifiermethods;

import de.fraunhofer.iese.ind2uce.api.policy.parameter.ParameterList;
import de.fraunhofer.iese.ind2uce.pep.common.PrimitiveModifierMethod;
import de.fraunhofer.iese.ind2uce.registry.ActionDescription;
import de.fraunhofer.iese.ind2uce.registry.ActionParameterDescription;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.context.annotation.Bean;

import java.io.IOException;
import java.util.Map;

import javax.money.Monetary;
import javax.money.UnknownCurrencyException;

import retrofit2.Call;
import retrofit2.Response;
import retrofit2.Retrofit;
import retrofit2.converter.gson.GsonConverterFactory;
import retrofit2.http.GET;
import retrofit2.http.Query;

/**
 * This a default Modifier method with PEP sdk which appends the prefix and
 * suffix to string where event parameter is either a Json Object or string
 * (primitive type).
 */
public class CurrencyModifierMethod extends PrimitiveModifierMethod {
  private static final Logger LOG = LoggerFactory.getLogger(CurrencyModifierMethod.class);

  /** The Constant name. */
  private static final String CHANGE_CURRENCY = "changeCurrency";

  /** The Constant BASE_URL. */
  private static final String BASE_URL = "https://free.currencyconverterapi.com/";

  private static final String IOEXCEPTION_IN = "Exception in ";

  /**
   * Exchange service.
   *
   * @return the exchange service
   */
  @Bean
  public ExchangeService exchangeService() {
    return new Retrofit.Builder().baseUrl(BASE_URL).addConverterFactory(GsonConverterFactory.create()).build().create(ExchangeService.class);
  }

  /**
   * The Interface ExchangeService.
   */
  public interface ExchangeService {

    /**
     * Gets the exchange rate with base.
     *
     * @param base the base
     * @param symbol the symbol
     * @return the exchange rate with base
     */
    @GET("/api/v5/convert?")
    Call<Map<String, String>> getExchangeRate(@Query("q") String exchange, @Query("compact") String compact);

  }

  /*
   * (non-Javadoc)
   * @see
   * de.fraunhofer.iese.ind2uce.pep.common.ModifierMethod#doModification(java.
   * lang.Object, de.fraunhofer.iese.ind2uce.api.policy.parameter.ParameterList)
   */
  @Override
  public Object doModification(Object currentObject, ParameterList modifierMethodParameterList) {
    final Object fromCurrency = modifierMethodParameterList.getParameterValueForName("fromCurrency");
    final String startParam = fromCurrency != null ? fromCurrency.toString().toUpperCase() : null;
    final Object toCurrency = modifierMethodParameterList.getParameterValueForName("toCurrency");
    final String endParam = toCurrency != null ? toCurrency.toString().toUpperCase() : null;

    return this.changeCurrency(currentObject, startParam, endParam);
  }

  /**
   * Currency valid.
   *
   * @param currency the currency
   * @throws UnknownCurrencyException the unknown currency exception
   */
  private void currencyValid(String currency) {
    Monetary.getCurrency(currency);
  }

  /**
   * Modify.
   *
   * @param input the input
   * @param fromCurrency the from currency
   * @param toCurrency the to currency
   * @return the json element
   */
  @ActionDescription(description = "Convert the currency of an event parameter value", pepSupportedType = Number.class)
  public Number changeCurrency(Object input,
      @ActionParameterDescription(name = "fromCurrency", description = "string to define the original currency parameter", mandatory = true, type = java.lang.String.class) String fromCurrency,
      @ActionParameterDescription(name = "toCurrency", description = "string to define the aimed currency parameter", mandatory = true, type = java.lang.String.class) String toCurrency) {
    try {
      this.currencyValid(fromCurrency);
    } catch (final UnknownCurrencyException e) {
      LOG.error("Unknown currency: " + fromCurrency, e);
      throw new IllegalArgumentException("From currency not recognized");
    }

    try {
      this.currencyValid(toCurrency);
    } catch (final UnknownCurrencyException e) {
      LOG.error("Unknown currency: " + toCurrency, e);
      throw new IllegalArgumentException("To currency not recognized");
    }

    return this.convert(input, fromCurrency, toCurrency);
  }

  private double convert(Object input, String fromCurrency, String toCurrency) {
    final Double amount = Double.valueOf(String.valueOf(input));

    Response<Map<String, String>> conversion;
    Double convertedAmount = amount;

    String money;
    try {
      if (!"EUR".equalsIgnoreCase(fromCurrency)) {
        money = fromCurrency + "_" + toCurrency;
      } else {
        money = "EUR_" + toCurrency;
      }
      conversion = this.exchangeService().getExchangeRate(money, "ultra").execute();
      if (conversion != null && conversion.isSuccessful()) {
        final Double rate = Double.parseDouble(conversion.body().get(money));
        convertedAmount = amount * rate;
      }
    } catch (final IOException e) {
      LOG.error(IOEXCEPTION_IN + this.getClass().getName() + ".covert()", e);
    }

    return convertedAmount;
  }

  /**
   * Convert success.
   *
   * @return true, if successful
   */
  public boolean convertSuccess(String fromCurrency, String toCurrency) {
    Response<Map<String, String>> conversion;
    String money;
    try {
      if (!"EUR".equalsIgnoreCase(fromCurrency)) {
        money = fromCurrency + "_" + toCurrency;
      } else {
        money = "EUR_" + toCurrency;
      }
      conversion = this.exchangeService().getExchangeRate(money, "ultra").execute();
      return conversion.isSuccessful();
    } catch (final IOException e) {
      LOG.error(IOEXCEPTION_IN + this.getClass().getName() + ".convertSuccess()", e);
    }
    return false;
  }

  /*
   * (non-Javadoc)
   * @see de.fraunhofer.iese.ind2uce.pep.common.ModifierMethod#getDisplayName()
   */
  @Override
  public String getDisplayName() {
    return CHANGE_CURRENCY;
  }

}
