/*-
 * =================================LICENSE_START=================================
 * IND2UCE
 * %%
 * Copyright (C) 2016 Fraunhofer IESE (www.iese.fraunhofer.de)
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =================================LICENSE_END=================================
 */

package de.fraunhofer.iese.ind2uce.pep;

import de.fraunhofer.iese.ind2uce.api.component.Component;
import de.fraunhofer.iese.ind2uce.api.component.exception.EvaluationUndecidableException;
import de.fraunhofer.iese.ind2uce.api.component.exception.InhibitException;
import de.fraunhofer.iese.ind2uce.api.component.identifier.ComponentId;
import de.fraunhofer.iese.ind2uce.api.component.interfaces.IPolicyDecisionPoint;
import de.fraunhofer.iese.ind2uce.api.component.interfaces.IPolicyEnforcementPoint;
import de.fraunhofer.iese.ind2uce.api.component.interfaces.IPolicyManagementPoint;
import de.fraunhofer.iese.ind2uce.api.policy.AuthorizationDecision;
import de.fraunhofer.iese.ind2uce.api.policy.Event;
import de.fraunhofer.iese.ind2uce.connectors.ConnectorFactory;
import de.fraunhofer.iese.ind2uce.connectors.OAuthCredentials;
import de.fraunhofer.iese.ind2uce.pep.common.DecisionEnforcer;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.net.URI;
import java.util.List;

/**
 * This class should be used with Spring auto registration
 */
public class DefaultPolicyEnforcementPoint implements IPolicyEnforcementPoint {

  /**
   *
   */
  private static final long serialVersionUID = -6859810815155938002L;

  private static final Logger LOG = LoggerFactory.getLogger(DefaultPolicyEnforcementPoint.class);

  private final transient DecisionEnforcer decisionEnforcer;

  protected final OAuthCredentials oauthClientCredentials;

  protected URI pmpUrl;

  private IPolicyDecisionPoint remotePDP;

  /**
   * Constructor
   *
   * @param pmpUrl pmp url
   * @param decisionEnforcer concrete implementation of decisionEnforcer
   * @param oauthClientCredentials OAuth Credentials
   */
  public DefaultPolicyEnforcementPoint(URI pmpUrl, DecisionEnforcer decisionEnforcer, OAuthCredentials oauthClientCredentials) {
    this.pmpUrl = pmpUrl;
    this.decisionEnforcer = decisionEnforcer;
    this.oauthClientCredentials = oauthClientCredentials;
  }

  /**
   * Make sure the PDP is available
   *
   * @return true if the pdp is available.
   * @throws IOException If connection could not be established.
   */
  public boolean assurePDP() throws IOException {
    return this.remotePDP != null || this.initPpdConnection() != null;
  }

  /**
   * Publish the event and enforce the {@link AuthorizationDecision} to
   * {@link Event}
   *
   * @param event The event to publish.
   * @throws EvaluationUndecidableException if PDP can't decide.
   * @throws InhibitException if event is not allowed
   * @throws IOException if connection to PDP is not established
   */
  @Override
  public void enforce(Event event) throws EvaluationUndecidableException, InhibitException, IOException {
    LOG.info("Received event for enforcement !!!!: {}", event);
    if (this.assurePDP()) {
      LOG.debug("Requesting decision");
      final AuthorizationDecision authorizationDecision = this.remotePDP.decisionRequest(event);
      LOG.debug("Received decision: {}", authorizationDecision);
      this.enforceDecision(event, authorizationDecision);
    }
  }

  /**
   * Impose the {@link AuthorizationDecision} authorizationDecision on
   * {@link Event} event.
   *
   * @param event on which authorizationDecision to be impose
   * @param authorizationDecision the decision to enforce on event
   * @throws InhibitException if event is not allowed
   */
  @Override
  public void enforceDecision(Event event, AuthorizationDecision authorizationDecision) throws InhibitException {
    if (authorizationDecision.isEventAllowed()) {
      LOG.info("Event will be allowed");
      this.decisionEnforcer.enforce(authorizationDecision, event.getParameters());
    } else {
      LOG.info("Event will be inhibited");
      event.clearParameters();
      throw new InhibitException("Event is not allowed according to policy");
    }
  }

  /**
   * Fetch the {@link AuthorizationDecision} from PDP against the {@link Event}.
   *
   * @param event Event to send.
   * @return The decision from PDP
   * @throws EvaluationUndecidableException if Event is not allowed.
   * @throws IOException if connection to pdp could not be established.
   */
  @Override
  public AuthorizationDecision getDecision(Event event) throws EvaluationUndecidableException, IOException {
    LOG.info("Event is going to PDP for Authorization Decision");
    if (this.assurePDP()) {
      return this.remotePDP.decisionRequest(event);
    } else {
      return null;
    }
  }

  /**
   * @return The id of the component.
   * @throws IOException not used here
   */
  @Override
  public ComponentId getId() throws IOException {
    return null;
  }

  /**
   * Default initializer that establishes the connection to PDP.
   *
   * @param strings not needed here.
   * @return true, if connection could be established.
   * @throws IOException If no connection could be established.
   */
  @Override
  public boolean initialize(String... strings) throws IOException {
    return this.initPpdConnection() != null;
  }

  /**
   * Initialize the PDP connection
   *
   * @return The connection to PDP.
   * @throws IOException If connection could not be established.
   */
  public IPolicyDecisionPoint initPpdConnection() throws IOException {
    final IPolicyManagementPoint pmp = ConnectorFactory.getPmpClient(this.pmpUrl, this.oauthClientCredentials);
    if (pmp == null) {
      LOG.error("PMP is found null");
      throw new IOException("PMP is found null");
    }
    final Component component = pmp.lookupPdp();
    LOG.info("List of PDP components fetched successfully from database");
    final List<URI> uris = component.getUrls();
    for (final URI uri : uris) {
      if (uri == null) {
        continue;
      }
      if (("https".equals(uri.getScheme())) || ("http".equals(uri.getScheme()))) {
        this.remotePDP = ConnectorFactory.getPdp(uri, this.oauthClientCredentials);
        if (this.remotePDP != null) {
          LOG.debug("Successfully connected to a PDP");
        }
        return this.remotePDP;
      } else {
        this.remotePDP = ConnectorFactory.getPdp(uri);
        LOG.debug("Successfully connected to a PDP");
      }
    }
    return this.remotePDP;
  }

  /**
   * Reset the PDP connection.
   *
   * @return true, if successful.
   * @throws IOException if connection could not be reset.
   */
  @Override
  public boolean reset() throws IOException {
    return this.initialize();
  }

  @Override
  public String getHealth() throws IOException {
    return "{\"status\":{\"code\":\"UP\",\"description\":\"\"},\"details\":{}}";
  }
}
