/*-
 * =================================LICENSE_START=================================
 * IND2UCE
 * %%
 * Copyright (C) 2016 Fraunhofer IESE (www.iese.fraunhofer.de)
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =================================LICENSE_END=================================
 */

package de.fraunhofer.iese.ind2uce.json.schema;

import de.fraunhofer.iese.ind2uce.api.component.description.ClassTypeDescription;
import de.fraunhofer.iese.ind2uce.api.component.description.JsonType;
import de.fraunhofer.iese.ind2uce.api.component.description.TypeDescription;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.lang.reflect.Field;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

/***
 *
 */
public class JsonSchemaGenerator {

  private static final Logger LOG = LoggerFactory.getLogger(JsonSchemaGenerator.class);

  public static final String MAP_FULL_QUALIFIED_CLASSNAME = "java.util.Map";

  public static final String GENERIC_BEGIN = "<";

  public static final String GENERIC_END = ">";

  private JsonSchemaGenerator() {

  }

  private static TypeDescription addArrayType(Map<String, TypeDescription> map, Class aClass) throws ClassNotFoundException {
    final TypeDescription arrayTypeDescription = new TypeDescription();
    arrayTypeDescription.setJsonType(JsonType.ARRAY);
    arrayTypeDescription.setTypeName(aClass.getComponentType().getName());
    if (!map.containsKey(aClass.getComponentType().getTypeName())) {
      createMap(map, aClass.getComponentType(), aClass.getComponentType());
    }

    return arrayTypeDescription;
  }

  private static void addComplexType(Map<String, TypeDescription> map, Type type, Class aClass) throws ClassNotFoundException {
    final ClassTypeDescription classTypeDescription = new ClassTypeDescription();
    classTypeDescription.setJsonType(JsonType.OBJECT);
    classTypeDescription.setTypeName(type.getTypeName());
    // put in global map
    map.put(type.getTypeName(), classTypeDescription);

    final Field[] allComplexObjectFields = JavaReflectionUtils.getAllFields(aClass);

    addFieldDescriptionOfComplexType(map, classTypeDescription, allComplexObjectFields);
  }

  private static void addDescriptionForField(Map<String, TypeDescription> map, ClassTypeDescription classTypeDescription, Field complexObjectFields) throws ClassNotFoundException {
    final Class objectFieldClass = complexObjectFields.getType();
    if (JavaReflectionUtils.isIterable(objectFieldClass)) {
      final ParameterizedType parameterizedType = (ParameterizedType)complexObjectFields.getGenericType();
      final Type[] iterableParametrizedTypes = parameterizedType.getActualTypeArguments();
      if (iterableParametrizedTypes != null && iterableParametrizedTypes.length > 0) {
        final Class iterableParametrizedClass = Class.forName(iterableParametrizedTypes[0].getTypeName());
        if (!map.containsKey(iterableParametrizedTypes[0].getTypeName())) {
          createMap(map, iterableParametrizedTypes[0], iterableParametrizedClass);
        }
        classTypeDescription.addFields(complexObjectFields.getName(), createReferenceTypeDescription(iterableParametrizedTypes[0].getTypeName(), JsonType.ARRAY));
      } else {
        // set type object
        final TypeDescription objectTypeDescription = new TypeDescription();
        objectTypeDescription.setJsonType(JsonType.PRIMITIVE);
        objectTypeDescription.setTypeName(Object.class.getSimpleName());
        classTypeDescription.addFields(complexObjectFields.getName(), objectTypeDescription);
      }
    } else if (objectFieldClass.isArray()) {
      classTypeDescription.addFields(complexObjectFields.getName(), addArrayType(map, objectFieldClass));
    } else if (JavaReflectionUtils.isInd2ucePrimitive(objectFieldClass)) {
      classTypeDescription.addFields(complexObjectFields.getName(), addPrimitiveType(map, objectFieldClass));
    } else if (JavaReflectionUtils.isMap(objectFieldClass)) {
      if (complexObjectFields.getGenericType() instanceof ParameterizedType) {
        final ClassTypeDescription mapTypeDescription = addMapType(map, complexObjectFields.getGenericType());
        classTypeDescription.addFields(complexObjectFields.getName(), mapTypeDescription);
      }
    } else {
      // nested Object
      if (!map.containsKey(objectFieldClass.getName())) {
        createMap(map, objectFieldClass, objectFieldClass);
      }
      classTypeDescription.addFields(complexObjectFields.getName(), createReferenceTypeDescription(objectFieldClass.getName(), JsonType.OBJECT));
    }
  }

  private static void addFieldDescriptionOfComplexType(Map<String, TypeDescription> map, ClassTypeDescription classTypeDescription, Field[] allComplexObjectFields) throws ClassNotFoundException {
    for (final Field complexObjectFields : allComplexObjectFields) {
      addDescriptionForField(map, classTypeDescription, complexObjectFields);
    }
  }

  private static void addIterableType(Map<String, TypeDescription> map, ParameterizedType type) {
    final TypeDescription iterableTypeDescription = new TypeDescription();
    iterableTypeDescription.setJsonType(JsonType.ARRAY);
    final ParameterizedType parameterizedType = type;
    final Type[] iterableParametrizedTypes = parameterizedType.getActualTypeArguments();
    if (iterableParametrizedTypes != null && iterableParametrizedTypes.length > 0) {
      try {
        final Class iterableParametrizedClass = Class.forName(iterableParametrizedTypes[0].getTypeName());
        if (!map.containsKey(iterableParametrizedTypes[0].getTypeName())) {
          createMap(map, iterableParametrizedTypes[0], iterableParametrizedClass);
        }
        iterableTypeDescription.setTypeName(iterableParametrizedTypes[0].getTypeName());
        map.put(iterableParametrizedTypes[0].getTypeName() + "List", iterableTypeDescription);
      } catch (final Exception e) {
        LOG.error("Exception in JsonSchemaGenerator.addIterableType", e);
      }
    } else {
      // set type object
      iterableTypeDescription.setTypeName(Object.class.getSimpleName());
    }
  }

  private static ClassTypeDescription addMapType(Map<String, TypeDescription> map, Type type) {
    if (type instanceof ParameterizedType) {
      final Type mapType = ((ParameterizedType)type).getActualTypeArguments()[1];
      try {

        return createMapTypeDescription(map, mapType);
      } catch (final Exception e) {
        LOG.error("Exception in JsonSchemaGenerator.addMapType", e);
      }
    }
    return null;
  }

  private static TypeDescription addPrimitiveType(Map<String, TypeDescription> map, Class aClass) {
    final TypeDescription primitiveTypeDescription = new TypeDescription();
    primitiveTypeDescription.setJsonType(JsonType.PRIMITIVE);
    primitiveTypeDescription.setTypeName(aClass.getName());
    return primitiveTypeDescription;
  }

  private static void createMap(Map<String, TypeDescription> map, Type type, Class aClass) throws ClassNotFoundException {
    if (JavaReflectionUtils.isIterable(aClass)) {
      addIterableType(map, (ParameterizedType)type);
    } else if (aClass.isArray()) {
      map.put(aClass.getComponentType().getName(), addArrayType(map, aClass));
    } else if (JavaReflectionUtils.isMap(aClass)) {
      addMapType(map, type);
    } else if (JavaReflectionUtils.isInd2ucePrimitive(aClass)) {
      map.put(aClass.getName(), addPrimitiveType(map, aClass));
    } else {
      addComplexType(map, type, aClass);
    }

  }

  private static ClassTypeDescription createMapTypeDescription(Map<String, TypeDescription> map, Type mapType) throws ClassNotFoundException {
    final Class mapTypeClass = Class.forName(mapType.getTypeName());
    if (!map.containsKey(MAP_FULL_QUALIFIED_CLASSNAME + GENERIC_BEGIN + mapType.getTypeName() + GENERIC_END)) {
      createMap(map, mapType, mapTypeClass);
    }
    final ClassTypeDescription mapClassTypeDescription = new ClassTypeDescription();
    mapClassTypeDescription.setJsonType(JsonType.OBJECT);
    mapClassTypeDescription.setTypeName(MAP_FULL_QUALIFIED_CLASSNAME + GENERIC_BEGIN + mapType.getTypeName() + GENERIC_END);
    if (JavaReflectionUtils.isInd2ucePrimitive(mapTypeClass)) {
      mapClassTypeDescription.addFields("*", createReferenceTypeDescription(mapType.getTypeName(), JsonType.PRIMITIVE));
    } else {
      mapClassTypeDescription.addFields("*", createReferenceTypeDescription(mapType.getTypeName(), JsonType.OBJECT));
    }
    map.put(MAP_FULL_QUALIFIED_CLASSNAME + GENERIC_BEGIN + mapType.getTypeName() + GENERIC_END, mapClassTypeDescription);
    return mapClassTypeDescription;
  }

  private static TypeDescription createReferenceTypeDescription(String typeName, JsonType jsonType) {
    final TypeDescription typeDescription = new TypeDescription();
    typeDescription.setTypeName(typeName);
    typeDescription.setJsonType(jsonType);
    return typeDescription;
  }

  /***
   * Generates a list of {@link TypeDescription} for a given class.
   * <ul>
   * <li>For Primitive Types see
   * {@link JavaReflectionUtils#isInd2ucePrimitive(Class)} this Method returns a
   * singleton list with a {@link TypeDescription} containing the name of the
   * primitive like java.lang.Long and JsonType.PRIMITIVE</li>
   * <li>For Iterable Types (Collections) this Method returns a list with a
   * {@link TypeDescription} containing the name of the generic content of the
   * list (e.g. for List&lt;String&gt; name will be java.lang.String) and
   * JsonType.ARRAY. If the generic Type of the list is a non primitive type,
   * this method also adds the description for this type (recursively).</li>
   * <li>Array types are handled in the same manner as iterable types.</li>
   * <li>Array types are handled in the same manner as iterable types.</li>
   * <li>For Maps this Method returns a list with a {@link TypeDescription}
   * containing the name Map &lt; SOME_TYPE &gt; and JsonType.OBJECT. If the
   * generic Type (SOME_TYPE) of the map is a non primitive type, this method
   * also adds the description for this type (recursively).</li>
   * </ul>
   *
   * @param type Type
   * @param typeClass Class
   * @return List of TypeDescription
   * @throws ClassNotFoundException class not found
   */
  public static List<TypeDescription> createTypeDescription(Type type, Class typeClass) throws ClassNotFoundException {
    final Map<String, TypeDescription> typeNameAndTypeDescriptionMap = new HashMap<>();
    createMap(typeNameAndTypeDescriptionMap, type, typeClass);
    return new LinkedList<>(typeNameAndTypeDescriptionMap.values());
  }

  /**
   * Returns a jsonType for a given class.
   * <ul>
   * <li>All Primitives including the wrapper types, enums, date and string are
   * mapped to Primitive</li>
   * <li>All Iterables and primitive arrays are mapped to JsonType.ARRAY</li>
   * <li>The rest is JsonType.OBJECT</li>
   * </ul>
   *
   * @param type Class
   * @return JsonType
   */
  public static JsonType getJsonType(Class type) {
    if (JavaReflectionUtils.isInd2ucePrimitive(type)) {
      return JsonType.PRIMITIVE;
    } else if (JavaReflectionUtils.isIterable(type) || type.isArray()) {
      return JsonType.ARRAY;
    } else {
      return JsonType.OBJECT;
    }
  }

}
